/*
 *     Copyright (c)2001-2003 DemiVision, LLC. All Rights Reserved.
 *
 * The information contained herein is the CONFIDENTIAL and PROPRIETARY
 *                  information of DemiVision, LLC.
 */

package com.bejeweled2_j2me;

import javax.microedition.lcdui.*;


/**
 * A <tt>Dialog</tt> is a simple implementation of a standard modal dialog box.
 * There are Ok, Ok/Cancel, and Yes/No variations, as well as a Wait dialog with
 * an animated progress bar suitable for use during network communications
 * or other long tasks. The Wait dialog can optionally provide timeout callbacks.
 * Dialog events are handled by a callback method in <tt>BaseScreen</tt>.<p>
 *
 * Following are examples of each available Dialog type:
 *
 * <table cellspacing="20"><tr>
 *   <td align="center"><img src="doc-files/Dialog-1.png"><br>Ok</td>
 *   <td align="center"><img src="doc-files/Dialog-2.png"><br>Ok/Cancel</td>
 *   <td align="center"><img src="doc-files/Dialog-3.png"><br>Yes/No</td>
 *   <td align="center"><img src="doc-files/Dialog-4.png"><br>Wait</td>
 * </tr></table>
 *
 * @see         BaseScreen
 *
 * @author      Barry Sohl
 * @version     1.1.0
 */
final class Dialog implements CommandListener, DialogListener
{
    /* Constants */

    // Command results codes

    /** Command result code indicating an error occurred during dialog display. */
    public static final byte CMD_ERR = 0;

    /** Command result code indicating 'Ok' response to an Ok or Ok/Cancel dialog. */
    public static final byte CMD_OK = 1;

    /** Command result code indicating 'Yes' response to a Yes/No dialog. */
    public static final byte CMD_YES = 2;

    /** Command result code indicating 'No' response to a Yes/No dialog. */
    public static final byte CMD_NO = 3;

    /** Command result code indicating user cancelled a Wait or Ok/Cancel dialog. */
    public static final byte CMD_CANCEL  = 4;

    /** Command result code indicating that a Wait dialog timed out. */
    public static final byte CMD_TIMEOUT = 5;

    // Dialog types

    /** Type code for a dialog displaying only an 'Ok' option. */
    public static final byte DLG_OK = 0;

    /** Type code for a dialog displaying 'Ok' and 'Cancel' options. */
    public static final byte DLG_OKCANCEL = 1;

    /** Type code for a dialog displaying 'Yes' and 'No' options. */
    public static final byte DLG_YESNO = 2;

    /** Type code for application help information. */
    public static final byte DLG_HELP = 3;
    
    /** Type code for the main menu */
    public static final byte DLG_MAIN_MENU = 4;

    // Vodafone dialogs
    public static final byte DLG_VODA_START_EXIT = 5;
    
    public static final byte DLG_VODA_GET_EXIT = 6;
    
    public static final byte DLG_VODA_GET_BACK = 7;

    // Text formatting
    static final char LINE_BREAK = '|';
    static final char HARD_SPACE = '_';

    public static final char NEW_LINE   = '\n';
    private static final char SPACE_CHAR = ' ';

    // Word wrap settings
    private static final int WRAP_PADDING = 8;

    /* Data Fields */
    private GameEngine gameEngine;      // External references

    private static Dialog dialog;       // Singleton instance
    private boolean showing;
    public byte curType;

    public DialogListener listener; // Callback
    private DialogListener helpListener;// Help dialog has two levels so we
                                        // must save its listener.
    private Display display;

    private DialogCanvas dialogForm;    // Interface widgets
    private DialogCanvas helpList;

    private String[] helpItems;         // Help menu items

    private Thread textThread;          // Text resource support
    private String textTitle;
    private String textFile;
    private int textOffset;

    public Command okCmd;               // High-level events
    public Command yesCmd;
    public Command noCmd;
    public Command cancelCmd;
    public Command backCmd;
    public static boolean fromExit=false;

    /**
     * Called back by <tt>Dialog</tt> when a dialog has been dismissed. Allows
     * any <tt>BaseScreen</tt> to handle dialog events.
     *
     * @param   result  resulting command type code.
     */
    public void dialogDismissed(byte result) {
        // This should only be called from the help text resources,
        // so just go back to the help list using the stored help user.
        showHelp(helpListener);
    }

    /**
     * Launches an independent thread to load text from an external text
     * resource file and display the results in a modal dialog. This
     * functionality must be threaded because some devices are extremely
     * slow loading from local, persistent storage.
     *
     * @param   title   title to be used for resulting dialog.
     * @param   file    resource file containing text being loaded.
     * @param   offset  subitem offset of resource being loaded.
     */
    public void showTextResource(
        final DialogListener listener,
        final String title, final String file, int offset)
    {
        textTitle = title;
        textFile = file;
        textOffset = offset;

        // Start thread to load resource
        textThread = new Thread() {
            public void run() {
                // Load external resource
                String msg = gameEngine.getTextResource(textFile, textOffset);

                // Display result
                show(listener, Dialog.DLG_OK, textTitle, msg);
            }
        };
        textThread.start();
    }

    /**
     * Private constructor protects against instantiation. Initializes
     * all interface widgets.
     */
    private Dialog()
    {
        gameEngine = GameEngine.getInstance();

        // Layout base dialog
        dialogForm = new DialogCanvas(this, DialogCanvas.TEXT, null );

        // Setup high-level events
        okCmd = new Command(StringTable.SPACE, Command.OK, 0);
        yesCmd = new Command(StringTable.SPACE, Command.OK, 0);
        noCmd     = new Command( StringTable.NO_SOFTKEY, Command.CANCEL, 0 );
        cancelCmd = new Command( StringTable.CANCEL_SOFTKEY, Command.CANCEL, 0 );

        dialogForm.setCommandListener( this );

        // Read settings for help menu
        //helpItems = gameEngine.getPropertyList(StringTable.PROP_HELP_LIST);
        // Convert help item titles to localized text.
        //for(int index = 0; index < helpItems.length; ++index) {
        //    helpItems[index] = StringTable.lookup(helpItems[index]);
        //}
        
        helpItems = new String[] {
            StringTable.lookup("HELP_TITLE_GOAL"),
            StringTable.lookup("HELP_TITLE_MODE"),
            StringTable.lookup("HELP_TITLE_CONTROL"),
            StringTable.lookup("HELP_TITLE_SCORE"),
        };

        

        // Layout help menu
        helpList = new DialogCanvas(this, DialogCanvas.LIST, StringTable.HELP_TITLE);
        helpList.setList(helpItems, null);

        // Setup help event handling
        backCmd = new Command(StringTable.BACK_SOFTKEY, Command.BACK, 0);
        
    }

    /**
     * Returns a reference to the singleton <tt>Dialog</tt> instance.
     *
     * @return  reference to singleton.
     */
    public static Dialog getInstance()
    {
        // Lazy instantiation
        if ( dialog == null )
        {
            dialog = new Dialog();
        }

        return dialog;
    }

    /**
     * Returns a reference to the <tt>Displayable</tt> object that should be
     * given focus for the current dialog.
     *
     * @return  reference to current <tt>Displayable</tt> object.
     */
    public Displayable getDisplayable()
    {
        switch(curType) {
            //case DLG_MAIN_MENU:
            //    return mainMenuList;
            case DLG_HELP:
                return (helpList);
        }
        return(dialogForm);
    }

    /**
     * Determines whether or not this <tt>Dialog</tt> is currently being shown.
     *
     * @return  true if dialog currently showing, else false.
     */
    public boolean isShowing()
    {
        return showing;
    }


    /**
     * Draw a word wrapped string that can also contain special formatting
     * characters. Space is the only word break. The formatting characters
     * are:
     *
     * \n - advance to next line
     *
     * @param string text with formatting characters
     * @param   font    font string will be displayed with
     * @param   width   width to wrap within
     * @return  word wrapped string.
     */
    public static String wordWrap(String string, Font font, int width) {
        StringBuffer wrappedString = new StringBuffer();
        int end = 0, current = 0;

        while(string.length() > 0) {
            char nextCharacter = NEW_LINE;

            int length = string.length();
            if(current != length) {
                int nextSpace = string.indexOf(SPACE_CHAR, current);
                int nextNewline = string.indexOf(NEW_LINE, current);
                current = Math.min((nextSpace < 0) ? length : nextSpace,
                    (nextNewline < 0) ? length : nextNewline);
            }

            if(current != length) {
                nextCharacter = string.charAt(current);
            }

            boolean performDraw = false;
            if(nextCharacter == NEW_LINE) {
                performDraw = true;
                if(font.substringWidth(string, 0, current) <= width) {
                    end = current;
                }
            }

            if((font.substringWidth(string, 0, current) > width) || performDraw) {
                wrappedString.append(string.substring(0, end));
                wrappedString.append(NEW_LINE);
                if(end != length) {
                    string = string.substring(end + 1);
                } else {
                    string = "";
                }

                end = current = 0;
            } else {
                end = current;
                ++current;
            }
        }

        return (wrappedString.toString());
    }

    /**
     * Shows a modal dialog box with the specified attributes. The specified
     * <tt>BaseScreen</tt> will be called back when the dialog is dismissed.
     *
     * @param   screen      <tt>BaseScreen</tt> to be called back.
     * @param   type        dialog type constant.
     * @param   title       dialog screen title.
     * @param   msg         dialog text message.
     */
    public void show( DialogListener screen, byte type, String title, String msg )
    {
        try
        {
            hide();

            listener = screen;
            curType = type;

            // Format message text
            String msgStr = null;

            if(msg != null) {
                msgStr = StringTable.NEW_LINE + msg;

                // Parse for special formatting characters
                if ( msgStr.indexOf( LINE_BREAK ) >= 0 )
                {
                    msgStr = msgStr.replace( LINE_BREAK, NEW_LINE );
                }

                // Space placeholders must be replaced after word wrap
                if ( msgStr.indexOf( HARD_SPACE ) >= 0 )
                {
                    msgStr = msgStr.replace( HARD_SPACE, SPACE_CHAR );
                }
            }

            // Setup help dialog
            if ( curType == DLG_HELP )
            {
                helpListener = listener;
                display.setCurrent( helpList );
            }
            else if ( curType == DLG_MAIN_MENU )
            {
                // Do nothing
            }
            // Setup all other dialog types
            else
            {
                // Must create new Form each time in order to reset cursor position
                dialogForm = new DialogCanvas(this, DialogCanvas.TEXT, null );

                dialogForm.setCommandListener( this );
                dialogForm.setDialogTitle( ' ' + title );
                dialogForm.setText( msgStr );

                display.setCurrent( dialogForm );
            }

            showing = true;
        }
        catch ( Exception ex )
        {
            if (Build.DEBUG) gameEngine.writeDebug( StringTable.DBG_DS1 , ex );
        }
    }

    /**
     * Convenience method for showing a modal error dialog.
     *
     * @param   screen      <tt>BaseScreen</tt> to be called back.
     * @param   msg         error message.
     */
    public void showError( DialogListener screen, String msg )
    {
        show( screen, DLG_OK, StringTable.ERROR_TITLE , msg );
    }


    /**
     * Covenience method for showing the help dialog.
     *
     * @param screen        <tt>BaseScreen</tt> to be called back.
     */
    public void showHelp(DialogListener screen) {
        show(screen, DLG_HELP, null, null);
    }

    /**
     * Hides the modal dialog if it is currently visible, and resets
     * for the next use.
     */
    public void hide()
    {
        showing = false;
    }

    /**
     * Saves reference to the master display so that the dialog can put itself
     * into focus when shown. Should only be called once.
     *
     * @param   dis     reference to master display.
     */
    void init( Display dis )
    {
        display = dis;
    }

    /**
     * General object cleanup at shutdown.
     */
    void destroy()
    {
        // Do nothing
    }

    /* CommandListener */

    /**
     * Handles high-level events for the dialog box. The orignal requestor
     * is called back. The requestor is responsible for setting the display
     * to the desired screen in the {@link BaseScreen#dialogDismissed(byte)}
     * callback. This allows focus to return to either the previous screen
     * or to be forwarded to a new screen. <i>For internal use only</i>.
     *
     * @param   cmd     command being handled.
     * @param   dis     displayable containing command.
     */
    public void commandAction( Command cmd, Displayable dis )
    {
        byte result = CMD_ERR;
        
        if (dis == helpList) {
            if (cmd == okCmd) {
                int selIndex = helpList.getSelectedIndex();

                showTextResource(this, helpItems[selIndex],
                    StringTable.FILE_HELP, selIndex);
                return;
            }
        }

        // Determine user's response
        if ( cmd == okCmd )
        {
            result = CMD_OK;
        }
        else if ( cmd == yesCmd )
        {
            fromExit=false;
            result = CMD_YES;
        }
        else if ( cmd == noCmd )
        {
            fromExit=false;
            result = CMD_NO;
        }
        else if ( cmd == cancelCmd )
        {
            result = CMD_CANCEL;
        }
        else if ( cmd == backCmd ) 
        {
            result = CMD_NO;
        }

        hide();
        // Callback requestor
        listener.dialogDismissed( result );
    }

}