/*
 *    Copyright (c)2001-2003 Jamdat Mobile, Inc. All Rights Reserved.
 *
 * The information contained herein is the CONFIDENTIAL and PROPRIETARY
 *                 information of Jamdat Mobile, Inc.
 */

package com.bejeweled2_j2me;

import javax.microedition.lcdui.*;
import java.io.IOException;

/**
 * The GameBoard is the primary parent class for the Bejeweled game interface.
 * It combines the Dioskilos API, application specific, GameScreen implementation
 * with the interface layout logic. Though this is less than ideal, it saves
 * significant codespace. The GameBoard maintains child instances of the local
 * and remote gem fields as well as the time bar. The GameBoard is also reponsible
 * for displaying the post-game score screen.
 *
 * @author      DemiVision: Barry Sohl
 * @version     0.9.4
 */
final class GameBoard extends GameScreen
{
    /* Constants */

    // Type of game
    static final int DIFFICULTY_NORMAL = 0;
    static final int DIFFICULTY_ADVANCED = 1;
    static final int DIFFICULTY_EXPERT = 2;

    // Pulse animation values
    static final int NUM_PULSE_FRAMES = 5;
    static final int PULSE_RATE = 125;

    // Border pulse colors
    static final int[] COLOR_PULSE = {
                                       0xFFFF00,
                                       0xE4C700,
                                       0xF39300,
                                       0xF36A00,
                                       0xFF0000
                                     };

    // MIDI support
    static final int MIDI_SWAP       = 0;
    static final int MIDI_GAME_OVER  = 1;
    static final int MIDI_BAD_SWAP   = 2;
    static final int MIDI_NEAR_DEATH = 3;
    static final int MIDI_START      = 4;
    static final int MIDI_BONUS      = 5;

    // Alert sound support
    static final AlertType ALERT_GAME_OVER  = AlertType.WARNING;
    static final AlertType ALERT_SWAP       = AlertType.CONFIRMATION;
    static final AlertType ALERT_BAD_SWAP   = AlertType.ERROR;
    static final AlertType ALERT_NEAR_DEATH = AlertType.CONFIRMATION;
    static final AlertType ALERT_START      = AlertType.CONFIRMATION;

    static final int COLOR_BGND = 0x000000;

    private final int COLOR_SCORE = 0xFFFFFF;
    private final int COLOR_OVERLAY        = 0xC0C0C0;
    private final int COLOR_OVERLAY_SHADOW = 0x000000;
    private final int COLOR_OVERLAY_BORDER = 0x880088;
    private final int COLOR_OVERLAY_TEXT   = 0x000088;

    // Overlay settings
    private final char OVERLAY_DELIM_LINE  = '\n';
    private final int OVERLAY_DURATION    = 2000;
    private final int OVERLAY_MAX_LINES   = 4;
    private final int OVERLAY_HPAD        = 16;
    private final int OVERLAY_VPAD        = 10;
    private final int OVERLAY_SHADOW      = 4;

    // Additional screen states
    private final byte STATE_SCORE = STATE_CUSTOM;

    /* Data Fields */
    private GemField localField;                    // Gem fields

    private TimeBar timeBar;                        // Time/power bar

    private String[] overlayLine;                   // Overlay handling
    private AnimatedImage overlayImage;
    private boolean overlayDirty;

    private int overlayWd;
    private int overlayNumLines;
    private long overlayStart;

    private boolean scoreDirty;

    private Image helpSoftkey;
    private Image exitSoftkey;

    private Image sidebarLeft;
    private Image sidebarRight;

    int p1Score;
    int p1Power;
    int p1Matches;
    int p1Cascades;
    int p1Longest;

    int p1Swaps;
    int p1BadSwaps;
    int p1Biggest;
    int p1NoMatches;

    private int localY, localWidth, localHeight;                  // UI object bounds
    private int timeBarY;

    private int scoreWd;
    private int scoreX, scoreY;

    private Font scoreFont;
    private Font overlayFont;

    private int scoreFontHt;
    private int overlayFontHt;

    private int difficulty;
    public Image softKeysImage;
    /**
     * Public default constructor necessary for access from outside package.
     */
    public GameBoard()
    {
        isPaused = false;
        try{
            softKeysImage = gameEngine.loadImage(StringTable.FILE_SOFT_KEYS, true);
        }catch(Exception e){}
    }

    /**
     * Returns either the current local or remote total score.
     *
     * @return  local or remote total score.
     */
    int getScore()
    {
        return (p1Score);
    }

    /**
     * Adds to the total score of either the local or remote player. Optionally,
     * this method can be used to set an entirely new points value rather than
     * incrementing the current value. The scores will be redrawn in the interface.
     *
     * @param   points      total points to be added or set.
     */
    void addScore( int points)
    {
        // The harder the game, the better the score.
        p1Score += points * (getDifficulty() + 1);
        scoreDirty = true;
    }

    /**
     * Ends the game currently in progress. Single player games will return to
     * the main menu, while multiplayer games will perform end-of-game handshaking
     * and then return to the lobby. In multiplayer mode, the client must indicate
     * whether it believes it won or lost the game given its current state.
     */
    void gameOver()
    {
        if ( isGameplayState() )
        {
            // Show message
            if(!gameEngine.isDemo()) {
                showOverlay( StringTable.GAME_OVER );

                // Sound and vibration
                gameEngine.vibrate(2000, false);
                gameEngine.playMidi(MIDI_GAME_OVER, false, false);
                gameEngine.playAlertSound(ALERT_GAME_OVER, false);
            }

            // Multiplayer starts handshaking, single player goes direct to score screen
            handleNodeEnd( NODE_END_DIED );
        }
    }

    /**
     * Hide the current overlay image, if any.
     */
    void hideOverlay() {
        overlayDirty = false;
        markDirty();
    }

    /**
     * Show an animated image as an overlay.
     *
     * @param image to display centered as overlay
     */
    void showOverlay(AnimatedImage image)
    {
        showOverlay(null, image);
    }

    /**
     * Displays a text overlay given a string.
     *
     * @param   string   for message to be displayed.
     */
    void showOverlay(String string) {
        showOverlay(string, null);
    }

    /**
     * Displays a text overlay above the local gem field with the specified
     * text message. The overlay will be drawn for a pretermined amount of
     * time and then will be erased. Formatting for the message is parsed
     * from the string itself.
     *
     * @param   msg     text message to be shown in overlay.
     * @param   img     alternative image to use, if not null this is displayed
     *                  instead of text
     */
    private void showOverlay( String msg, AnimatedImage img)
    {
        int delimiter;
        int prevDelimiter = 0;
        int strWd;

        overlayWd = 0;
        overlayNumLines = 0;
        overlayImage = null;

        // Either image or string, not both
        if(img != null) {
            overlayImage = img;
            overlayImage.reset();
        } else if(msg.length() > 0) {
            // Parse each line of message for display
            for ( int i = 0; i < OVERLAY_MAX_LINES; i++ )
            {
                delimiter = msg.indexOf( OVERLAY_DELIM_LINE, prevDelimiter );

                if ( delimiter > 0 )
                {
                    // Extract line
                    overlayLine[i] = msg.substring( prevDelimiter, delimiter );
                }
                else
                {
                    // Extract last line
                    overlayLine[i] = msg.substring( prevDelimiter );
                }

                // Track width for later drawing
                strWd = overlayFont.stringWidth( overlayLine[i] );

                if ( strWd > overlayWd )
                {
                    overlayWd = strWd;
                }

                prevDelimiter = (delimiter + 1);
                overlayNumLines++;

                if ( delimiter <= 0 )
                {
                    break;
                }
            }
        }

        overlayDirty = true;
        overlayStart = System.currentTimeMillis();
    }

    /**
     * Draws the currently defined text overlay message. The message will be drawn
     * in a windoid centered horizontally and vertically over the local gem field.
     *
     * @param   gc      graphics context used for drawing.
     */
    private void drawOverlay( Graphics gc )
    {
        if(overlayImage != null) {
            overlayImage.draw(gc);
        } else {
            // Dynamically determine X position
            int overlayWd2 = (overlayWd + OVERLAY_HPAD);
            int midX = (wd / 2);
            int overlayX = (midX - (overlayWd2 / 2));

            // Dynamically determine Y position
            int overlayHt = ((overlayNumLines * overlayFontHt) + OVERLAY_VPAD);
            int overlayY = (localY + ((localHeight - overlayHt) / 2));

            // Draw drop shadow
            gc.setColor( COLOR_OVERLAY_SHADOW );
            gc.fillRect( (overlayX + OVERLAY_SHADOW), (overlayY + OVERLAY_SHADOW), overlayWd2, overlayHt );

            // Draw background
            gc.setColor( COLOR_OVERLAY );
            gc.fillRect( overlayX, overlayY, overlayWd2, overlayHt );

            // Draw outline
            gc.setColor( COLOR_OVERLAY_BORDER );
            gc.drawRect( overlayX, overlayY, overlayWd2, overlayHt );

            // Draw each line of text, incrementing vertically, centering horizontally
            gc.setColor(COLOR_OVERLAY_TEXT);
            overlayY += ((OVERLAY_VPAD / 2) + 1);
            for ( int i = 0; i < overlayNumLines; i++ )
            {
                CustomFont.drawString(overlayFont, gc, overlayLine[i], midX,
                    (overlayY + (i * overlayFontHt)), GameEngine.CENTERED);
            }
        }
    }

    /**
     * Draws the post-game scoring screen with statistics for the game just
     * completed. Stats for one or two players will be displayed depending
     * on whether the game was single or multiplayer.
     *
     * @param   gc  graphics context used for drawing.
     */
    private void drawPostGame( Graphics gc )
    {
       // gameEngine.addCommand( okCmd );
       //if(mDialog.curType == mDialog.DLG_OK)
        {
            int sw = DeviceSpecific.SOFTKEY_WIDTH;
            int sh = DeviceSpecific.SOFTKEY_HEIGHT;
            gc.setClip(0, gameEngine.ht - sh, gameEngine.wd, sh);
            gc.setColor(0x000000);
            gc.fillRect(0,gameEngine.ht - sh, gameEngine.wd, sh);
            gc.setClip(0, gameEngine.ht - sh, sw, sh);
            gc.drawImage(softKeysImage, -4*sw, gameEngine.ht-sh, GameEngine.TOP_LEFT);
            gc.setClip(gameEngine.wd-sw, gameEngine.ht - sh, sw, sh);
            gc.setColor(0x000000);
            gc.fillRect(gameEngine.wd-sw, gameEngine.ht - sh, sw, sh);
            gc.setClip(0,0,gameEngine.wd,gameEngine.ht);
        }

        gc.drawImage(DialogCanvas.getBackgroundImage(), 0, 0,
            GameEngine.TOP_LEFT);

        gc.setColor(0xFFFFFF);
        CustomFont.drawString(GameEngine.FONT_TITLE, gc,
            StringTable.SCORE_TITLE, DialogCanvas.getTextRect()[0] + 2,
            (DialogCanvas.getTextRect()[1] -
            GameEngine.FONT_TITLE.getHeight()) / 2, GameEngine.TOP_LEFT);

        int left = DialogCanvas.getTextRect()[0] + 2;
        int right = DialogCanvas.getTextRect()[0] +
            DialogCanvas.getTextRect()[2] - 3;
        int y = DialogCanvas.getTextRect()[1] + 1;

        CustomFont.drawString(GameEngine.FONT_PLAIN, gc,
            StringTable.SCORE_TEXT, left, y, GameEngine.TOP_LEFT);
        CustomFont.drawString(GameEngine.FONT_PLAIN, gc,
            String.valueOf(p1Score), right, y, GameEngine.TOP_RIGHT);
        y += GameEngine.FONT_PLAIN.getHeight();
        CustomFont.drawString(GameEngine.FONT_PLAIN, gc,
            StringTable.SCORE_MATCHES, left, y, GameEngine.TOP_LEFT);
        CustomFont.drawString(GameEngine.FONT_PLAIN, gc,
            String.valueOf(p1Matches), right, y, GameEngine.TOP_RIGHT);
        y += GameEngine.FONT_PLAIN.getHeight();
        CustomFont.drawString(GameEngine.FONT_PLAIN, gc,
            StringTable.SCORE_CASCADES, left, y, GameEngine.TOP_LEFT);
        CustomFont.drawString(GameEngine.FONT_PLAIN, gc,
            String.valueOf(p1Cascades), right, y, GameEngine.TOP_RIGHT);
        y += GameEngine.FONT_PLAIN.getHeight();
        CustomFont.drawString(GameEngine.FONT_PLAIN, gc,
            StringTable.SCORE_LONGEST, left, y, GameEngine.TOP_LEFT);
        CustomFont.drawString(GameEngine.FONT_PLAIN, gc,
            String.valueOf(p1Longest), right, y, GameEngine.TOP_RIGHT);
        y += GameEngine.FONT_PLAIN.getHeight();
    }

    /* GameScreen */

    /**
     * Initializes the game board and all of its children. Significant memory
     * and resource allocation occurs here. All interface layout calculations
     * are made.
     *
     * @throws  IOException if error occurs loading resources.
     */
    public void init() throws IOException
    {
        super.init();

        // Load MIDI resources
        gameEngine.loadMidi();

        // Not displaying command bar
        setBounds( 0, 0, wd, getDisplayHeight() );

        // Prep for overlay display
        overlayLine = new String[ OVERLAY_MAX_LINES ];

        // Choose fonts
        scoreFont = GameEngine.FONT_PLAIN;
        scoreFontHt = scoreFont.getHeight();
        scoreWd = scoreFont.stringWidth(StringTable.EMPTY_SCORE);

        overlayFont = GameEngine.FONT_BOLD;
        overlayFontHt = overlayFont.getHeight();

        // Layout UI objects depending on screen size
        // First, load and obtain size of individual gems
        int localDim  = Gem.loadGemStrip();
        localWidth = (localDim * GemField.NUM_COLS);
        localHeight = (localDim * GemField.NUM_ROWS);

        int localX    = ((wd - localWidth) / 2);
        localY    = 2;

        int timeBarX = 2;
        int timeBarWd = wd - scoreWd - 5;
        int timeBarHt = 5;

        scoreX = timeBarX + timeBarWd + 1;

        // Center the score and time bar in the remaining space below
        // the gem field.
        int localBottom = localY + localHeight;
        int uiHt = Math.max(scoreFontHt, timeBarHt + 2); // Skull is 2 pixels bigger than bar
        int uiY = localBottom + (((ht - localBottom) - uiHt) / 2);
        scoreY = uiY + ((uiHt - scoreFontHt) / 2);
        timeBarY = uiY + ((uiHt - timeBarHt) / 2);

        // Create time/power bar
        timeBar = new TimeBar( this );
        timeBar.setBounds( timeBarX, timeBarY, timeBarWd, timeBarHt );
        timeBar.init();

        // Create gem fields
        localField = new GemField( this, timeBar, localDim);
        localField.setBounds( localX, localY, localWidth, localHeight );
        localField.init();

        // Load a tiny exit softkey to replace the normal big one.
        helpSoftkey = gameEngine.loadImage(StringTable.FILE_PAUSE_SOFTKEY, true);
        exitSoftkey = gameEngine.loadImage(StringTable.FILE_EXIT_SOFTKEY, true);
    }

    /**
     * Called at the start of each game. The game logic is reset and child
     * objects are started.
     *
     * @param   offset      level offfset of current game node.
     */
    public void startNode( byte offset )
    {
        // Reset game logic

        // Reset scoring
        p1Score = 0;

        p1Matches   = 0;
        p1Power     = 0;
        p1Cascades  = 0;
        p1Longest   = 0;
        p1Swaps     = 0;
        p1BadSwaps  = 0;
        p1Biggest   = 0;
        p1NoMatches = 0;

        // Start children
        localField.startNode( offset );
        timeBar.startNode( offset );

        // Show initial overlay
        showOverlay( StringTable.READY , null);

        // Must occur last
        super.startNode( offset );
    }

    /**
     * Called when the current game node has been ended by either this player
     * or the opponent. Game over state is handled here for games ended by
     * the opponent.
     *
     * @param   type    end type for current game node.
     * @param   remote  game ended by remote player?
     */
    public void endNode( byte type, boolean remote )
    {
        super.endNode( type, remote );
    }

    /**
     * Marks the game board and all of its children as dirty and therefore
     * requiring a redraw. All objects will be redrawn in the next game loop.
     */
    public void markDirty()
    {
        super.markDirty();

        if ( isGameplayState() || isPostGameState() )
        {
            scoreDirty  = true;

            // Mark children dirty
            localField.markDirty();
            timeBar.markDirty();
        }
    }

    /**
     * Draws all UI objects on the game board and instructs the child gem
     * fields and time bar to draw themselves.
     *
     * @param   gc  graphics context used for drawing.
     */
    public void draw( Graphics gc )
    {
        // Let parent draw
        if ( dirty )
        {
            super.draw( gc );
        }

        if ( isGameplayState() || isPostGameState() )
        {
            if ( dirty )
            {
                // Draw local area background
                gc.setColor( COLOR_BGND );
                gc.fillRect( 0, 0, wd, ht );
                gc.drawImage(helpSoftkey, 0, ht, GameEngine.BOTTOM_LEFT);
                //gc.drawImage(exitSoftkey, wd, ht, GameEngine.BOTTOM_RIGHT);

                dirty = false;
            }

            if ( scoreDirty )
            {
                drawScore(gc);

            }

            // Instruct children to draw
            localField.draw( gc );
            timeBar.draw( gc );

            // Overlay draws on top of everything and draws
            // continuously until turned off.
            //
            if ( overlayDirty )
            {
                drawOverlay( gc );
            }
       }
        // Score screen drawing
        else if ( getState() == STATE_SCORE )
        {
            if ( dirty )
            {
                drawPostGame( gc );
            }

            dirty = false;
        }
    }

    /**
     * Draw the current score in game.
     *
     * @param gc graphics context
     */
    private void drawScore(Graphics gc) {
        // Erase local score
        gc.setColor( COLOR_BGND );
        gc.fillRect( scoreX, scoreY, scoreWd, scoreFontHt );

        // Draw local score
        gc.setColor( COLOR_SCORE );
        CustomFont.drawString(scoreFont, gc, Integer.toString( p1Score ),
            scoreX + scoreWd, scoreY, GameEngine.TOP_RIGHT );

        // Score overwrites exit softkey slightly on the i730, so we
        // need to draw it again.
        //gc.drawImage(exitSoftkey, wd, ht, GameEngine.BOTTOM_RIGHT);

        scoreDirty = false;
    }

    /**
     * Called for each iteration of the game loop. Handles all time dependent
     * functionality for the game board and its children.
     *
     * @param   now     current system time in milliseconds.
     */
    public void heartbeat( long now )
    {
        super.heartbeat( now );

        if ( isGameplayState() )
        {
            // Might be time to turn off overlay
            if ( overlayDirty ) {
                if(overlayImage != null) {
                    overlayImage.heartbeat(now);
                    if(overlayImage.isDone()) {
                        hideOverlay();
                    }
                } else if((now - overlayStart) > OVERLAY_DURATION) {
                    hideOverlay();
                }
            }

            // Only pass to children during game
            localField.heartbeat( now );
            timeBar.heartbeat( now );
        }
        // Switch to score screen after handshaking done
        else if ( isOverState() )
        {
            ((SettingsScreen)gameEngine.getScreenInstance(
                GameEngine.SCREEN_SETTINGS)).reportScore(p1Score);

            // Don't show score for demo
            if(gameEngine.isDemo()) {
                handleGameOver();
            } else {
                setState(STATE_SCORE);
            }
        }
    }

    /**
     * Handles key presses for global game events such as weapon firing. Most
     * actual gameplay events are handled by the individual gem fields.
     *
     * @param   keyCode     key event being handled.
     */
    public void keyPressed( int keyCode )
    {
        // revisit this method
        if(getState() == STATE_SCORE)
        {
            if(keyCode == DeviceSpecific.KEY_CODE_SOFT2)
            {
                //handleGameOver();
                return;
            } else if(keyCode == DeviceSpecific.KEY_CODE_SOFT1)
            {
                return;
            }
        }
        else {
            super.keyPressed( keyCode );
        }


        if ( isGameplayState() )
        {
            // Forward event to children
            localField.keyPressed( keyCode );
        }
    }

    /**
     * Extends the superclass method to handle soft key commands on the
     * custom scoring subscreen.
     *
     * @param   cmd     command being handled.
     * @param   dis     displayable containing command.
     */
    public void commandAction( Command cmd, Displayable dis )
    {
        super.commandAction( cmd, dis );

        // User OK'd game over
        if ( getState() == STATE_SCORE )
        {
            handleGameOver();
        }
    }

    /**
     * Resumes single player gameplay, typically following a regain of focus by
     * the appllication. Multiplayer gameplay does not pause or resume in this way.
     *
     * @param   init    initial start following transition?
     * @param   now     current system time
     */
    public void start(boolean init, long now) {
        super.start(init, now);
        localField.start(init, now);
        timeBar.start(init, now);
    }

    /**
     * Pauses single player gameplay, typically following a loss of focus by
     * the application. Multiplayer gameplay does not pause.
     *
     * @param now current system time
     */
    public void pause(long now) {
        super.pause(now);
        localField.pause(now);
        timeBar.pause(now);
    }

    /**
     * Get the current difficulty level.
     *
     * @return DIFFICULTY_NORMAL, DIFFICULTY_ADVANCED, or DIFFICULTY_EXPERT
     */
    public int getDifficulty() {
        return difficulty;
    }

    /**
     * Set a new difficulty level.
     *
     * @param difficulty is DIFFICULTY_NORMAL, DIFFICULTY_ADVANCED, or
     *          DIFFICULTY_EXPERT
     */
    public void setDifficulty(int difficulty) {
        this.difficulty = difficulty;
    }

}

/**/
