/*
 *     Copyright (c)2001-2002 DemiVision, LLC. All Rights Reserved.
 *
 * The information contained herein is the CONFIDENTIAL and PROPRIETARY
 *                  information of DemiVision, LLC.
 */

package com.bejeweled2_j2me;

import javax.microedition.lcdui.Graphics;
import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;


/**
 * <tt>GameObject</tt> is a parent class providing common functionality for both
 * interface and gameplay objects. It groups a number of common methods so that
 * container classes can easily delegate functionality to a list of GameObjects
 * that they contain. Note that a GameObject is not by default heirarchial;
 * it does not include child GameObjects. A GameObject has life cycle methods as
 * well as methods for drawing, heartbeat, and event handling. Stub implementations
 * are provided for all optional methods.
 *
 * Examples of GameObjects include: sprites, background layers, as well as
 * <i>Dioskilos</i> interface screens. GameObject is an abstract class, derived
 * classes must provide their own {@link draw(Graphics)} method. Developers are
 * encouraged to make use of GameObject in order to avoid redundant code.
 *
 * @author      Barry Sohl
 * @version     0.9.9
 */
abstract class GameObject
{
    /* Constants */

    // Game modes

    /** Screen mode indicating the <tt>GameScreen</tt> or a child object is in single player mode. */
    public static final byte MODE_SINGLE_PLAYER = 0;

    /* Data Fields */

    /** Is the object currently dirty and in need of a redraw? */
    protected boolean dirty;

    /** Background color used for erasing the object. */
    protected int bgndColor = 0xFFFFFF;

    /** X coordinate of the object's current position within the drawing area. */
    protected int x;

    /** Y coordinate of the object's current position within the drawing area. */
    protected int y;

    /** Total pixel width of the object. */
    protected int wd;

    /** Total pixel height of the object. */
    protected int ht;

    /** Current state of this object, use {@link getState()} and {@link setState()} for threadsafe access. */
    protected byte state;

    /** Time in milliseconds at which this object last changed states. */
    protected long stateTime;

    /** Time in milliseconds at which this object was paused. */
    protected long pauseTime;

    /** Convenience reference to singleton <tt>GameEngine</tt> instance. */
    protected static GameEngine gameEngine;


    /**
     * Default constructor saves reference to master <tt>GameEngine</tt>.
     */
    GameObject()
    {
        if ( gameEngine == null )
        {
            gameEngine = GameEngine.getInstance();
        }
    }

    /**
     * Returns the currrent state of this <tt>GameObject</tt>. The definition of
     * state can vary for each object type. Protects against concurrent access by
     * the network and game loop threads. Developers should use this method for
     * accessing their own custom object states only.
     *
     * @return  current state of this object.
     */
    public final synchronized byte getState()
    {
        return state;
    }

    /**
     * Sets the current state of this <tt>GameObject</tt>. Protects against
     * concurrent access by the network and game loop threads. Developers should
     * use this method for setting their own custom states only.
     *
     * @param   newState    switch to this new game state.
     */
    public synchronized void setState( byte newState )
    {
        state = newState;
        stateTime = System.currentTimeMillis();
        if(pauseTime != 0) {
            pauseTime = stateTime;
        }
    }

    /**
     * Switches to a new object state if and only if the current state has not
     * already advanced beyond the specified state. Can be used as a gate to
     * allow multiple callers to initiate a state change in a concurrent
     * environment.
     *
     * @param   newState    requested new object state.
     * @return  true if state successfully switched, else false.
     */
    public synchronized boolean switchState( byte newState )
    {
        boolean success;

        // Only switch states if not already there
        if ( state >= newState )
        {
            success = false;
        }
        else
        {
            setState( newState );
            success = true;
        }

        return success;
    }

    /**
     * Sets the location of the object within the drawing area.
     *
     * @param   xPos    offset to left edge of object.
     * @param   yPos    offset to top edge of object.
     * @param   width   total width of object.
     * @param   height  total height of object.
     */
    public final void setBounds( int xPos, int yPos, int width, int height )
    {
        x = xPos;
        y = yPos;
        wd = width;
        ht = height;
    }

    /**
     * Sets the color that this object should use for repainting its
     * background. Defaults to white.
     *
     * @param   color   object repaints with this color.
     */
    public final void setBgndColor( int color )
    {
        bgndColor = color;
    }

    /**
     * Marks this object as dirty and therefore requiring a redraw. It will
     * be redrawn in the next iteration of the game loop.
     */
    public void markDirty()
    {
        dirty = true;
    }

    /**
     * Performs all once-only initialization for the object. Whenever possible,
     * all memory allocation and interface setup should occur here. Default
     * does nothing.
     *
     * @throws  IOException if error occurs during initialization.
     */
    public void init() throws IOException
    {}

    /**
     * Activates (shows) the game object and performs any necessary setup.
     * Default does nothing.
     *
     * @param   init    initial start after screen transition?
     * @param   now     current time
     */
    public void start( boolean init, long now )
    {
        markDirty();

        if(init) {
            // Need to reset state time in case pause time was added to it
            // between startNode and start(true...)
            stateTime = now;
            pauseTime = 0;
        } else if(pauseTime != 0) {
            stateTime += (now - pauseTime);
            pauseTime = 0;
        }
    }

    /**
     * Hides the game object and performs any necessary cleanup. Default
     * does nothing.
     *
     * @param   now     current time
     */
    public void pause( long now )
    {
        if(pauseTime == 0) {
            pauseTime = now;
        }
    }

    /**
     * Peforms general cleanup at shutdown. All resources should be released
     * here. Default does nothing.
     */
    public void destroy()
    {}

    /**
     * Resets the object to its initial state. Default does nothing.
     */
    public void reset()
    {}

    /**
     * Performs any functionality specific to the start of a game node.
     * Default does nothing.
     *
     * @param   offset      level offset of node.
     */
    public void startNode( byte offset )
    {}

    /**
     * Performs any functionality specific to the end of a game node.
     * Default does nothing.
     *
     * @param   endState    state in which node ended.
     */
    public void endNode( byte endState )
    {}

    /**
     * Instructs the object to draw itself onto the drawing surface. This
     * surface may be either an offscreen buffer or direct to the screen.
     * The provided <tt>Graphics</tt> reference will automatically be linked
     * to the appropriate surface. <i>Required for all game objects</i>.
     *
     * @param   gc      graphics context used for drawing.
     */
    public abstract void draw( Graphics gc );

    /**
     * Handles one iteration of the game loop. Used for animation and timed
     * events. Default does nothing.
     *
     * @param   now     current time as milliseconds since unix epoch.
     */
    public void heartbeat( long now )
    {}

    /**
     * Handles all key press events for the object. Default does nothing.
     *
     * @param   keyCode     key code being handled.
     */
    public void keyPressed( int keyCode )
    {}

    /**
     * Handles all key release events for the object. Default does nothing.
     *
     * @param   keyCode     key code being handled.
     */
    public void keyReleased( int keyCode )
    {}

}

/**/
