/*
 *    Copyright (c)2001-2003 Jamdat Mobile, Inc. All Rights Reserved.
 *
 * The information contained herein is the CONFIDENTIAL and PROPRIETARY
 *                 information of Jamdat Mobile, Inc.
 */

package com.bejeweled2_j2me;

import javax.microedition.lcdui.Graphics;
import javax.microedition.lcdui.Image;
import java.io.IOException;

/**
 * A Gem is the most primitive game object on the game board. It represents one
 * of the individual playing pieces on the GemField. A Gem is responsible for
 * maintaining its own state, as well as for its own drawing and animation.
 * Because many instances of Gem are in existence at the same time, resources
 * are shared via statics wherever possible. A Gem can represent individual gems
 * on either the local or remote gem fields.
 *
 * @author      DemiVision: Barry Sohl
 * @version     0.8.0
 */
final class Gem extends GameObject
{
    /* Constants */

    // Number of frames of animation per gem.
    static final byte NUM_FRAMES = 8;

    // Undefined gem type
    static final byte TYPE_EMPTY = 127;

    // Gem states
    static final byte STATE_INIT         = 0;
    static final byte STATE_FULL_CASCADE = 1;
    static final byte STATE_NORMAL       = 2;
    static final byte STATE_SELECT       = 3;
    static final byte STATE_SWAP         = 4;
    static final byte STATE_CASCADE      = 5;
    static final byte STATE_CLEAR        = 6;
    static final byte STATE_EMPTY        = 7;
    static final byte STATE_CAVE_IN      = 8;

    // Animation settings
    private static final int ANIM_RATE = 125;
    private static final int FLASH_RATE   = 250;

    // Convenience values used so booleans can be specified as bytes in case the KVM uses something larger.
    private static final byte FALSE     = 0;
    private static final byte TRUE      = 1;

    // Number of unique gem types
    static byte numberOfTypes = 0;

    private static boolean initialized;
    private static int numAnimFrames;           // Image animation
    private static int animRate;

    private byte gemType;                       // State info
    private byte index;

    private byte fieldRow, fieldCol;            // Location within gem field
    private int fieldY;

    private byte hasFocus;                      // Cursor handling
    private byte cursorDirty;

    private byte cursorFrame;
    private long cursorTime;

    private byte swapNegative;
    private byte swapVertical;

    private byte swapValid;                      // Double-swap handling
    private byte swapReversing;

    private byte dropCount;                      // Cascade handling

    private byte animFrame;
    private long animTime;

    private int animX, animY;                  // Movement animation
    private int animStartY;
    private int animDistance;
    private int animDuration;
    private int animDelay;

    private byte flashOn;                        // Gem flashing
    private long flashStart;
    private long flashTime;

    // All gems in large sheet containing individual animation frames.
    private static Image gemStrip;

    /**
     * Constructor saves information about parent GemField needed for drawing.
     *
     * @param   row     this gem's row within parent gem field.
     * @param   col     this gem's column within parent gem field.
     * @param   yPos    drawing surface y position of parent gem field.
     */
    Gem(GemField field, byte row, byte col, int yPos )
    {
        fieldY = field.y;
        fieldRow = row;
        fieldCol = col;
    }

    /**
     * Determines if this gem location is currently empty, meaning that it
     * contains no gem. This typically occurs during an anti-gravity or
     * no-fill weapons hit.
     *
     * @return  true    if gem currently empty, else false.
     */
    boolean isEmpty()
    {
        return ((getState() == STATE_EMPTY) || (gemType == TYPE_EMPTY));
    }

    /**
     * Toggles cursor focus for this gem. If the gem has focus, a pulsating
     * cursor outline will be displayed.
     *
     * @param   focus   turn focus on for this gem?
     */
    void setFocus( boolean focus )
    {
        hasFocus = focus ? TRUE : FALSE;

        markDirty();
    }

    /**
     * Sets the gem type for this gem. The gem will start in the specified state.
     *
     * @param   type        unique gem type.
     * @param   newState    state in which new gem type should start.
     */
    void setType( byte type, byte newState )
    {
        gemType = type;

        setState( newState );
    }

    /**
     * Starts a gem swap animation for this gem. The gem will smoothly change
     * positions with the specified destination gem. If the swap was not valid,
     * the gem will then smoothly move back to its original position.
     *
     * @param   valid   was this swap attempt valid?
     * @param   row     destination field row.
     * @param   col     destination field column.
     */
    void swap( boolean valid, int row, int col )
    {
        swapValid = valid ? TRUE : FALSE;
        int swapRow = row;
        int swapCol = col;

        setState( STATE_SWAP );

        // Determine if swapping up, down, left, or right
        swapNegative = (byte) (((swapRow < fieldRow) || (swapCol < fieldCol)) ? -1 : 1);
        swapVertical = (swapRow != fieldRow) ? TRUE : FALSE;
    }

    /**
     * Starts a cascade animation for this gem. The gem will smoothly drop the
     * specified number of positions on the gem field. The gem takes on the type
     * and powerup status of the gem cascading from above.
     *
     * @param   count       number of rows gem will cascade.
     * @param   newType     gem type of gem cascading down.
     */
    void cascade( int count, byte newType)
    {
        dropCount = (byte) count;

        setType( newType, STATE_CASCADE );
    }

    /**
     * Turns on or off flashing for this gem. When flashing is on, the gem will
     * flash for a predetermined amount of time.
     *
     * @param   start   turn flashing on?
     * @param   on      initial state if flashing is turned on
     */
    void flash( boolean start, boolean on )
    {
        if ( start )
        {
            long now = System.currentTimeMillis();

            flashOn = on ? TRUE : FALSE;
            flashStart = now;
            flashTime = now;

            markDirty();
        }
        else
        {
            // Always end with gem drawn on
            if ( flashOn == FALSE )
            {
                markDirty();
            }

            flashOn = TRUE;
            flashStart = 0;
        }
    }

    /* GameObject */

    /**
     * Sets the current state for this gem. Prepares for animation in the
     * new gem state.
     *
     * @param   newState   new state for this gem.
     */
    public void setState( byte newState )
    {
        super.setState( newState );

        switch ( newState )
        {
            // Reset cursor
            case STATE_INIT:
            {
                setFocus( false );
                break;
            }
            // Gems will pop into view after a random delay
            case STATE_FULL_CASCADE:
            {
                animDelay = gameEngine.genRandomNumber( GemField.DURATION_FULL_CASCADE2 );
                break;
            }
            // Reset image animation info
            case STATE_NORMAL:
            case STATE_SELECT:
            case STATE_CLEAR:
            {
                animX = x;
                animY = y;

                long now = System.currentTimeMillis();

                animFrame = 0;
                animTime = now;

                cursorTime = now;
                break;
            }
            // Prepare for movement animation in any direction
            case STATE_SWAP:
            {
                animX = x;
                animY = y;

                animDistance = wd;
                swapReversing = FALSE;
                break;
            }
            // Gem falls at fixed rate in unison with other cascading gems
            case STATE_CASCADE:
            {
                animDuration = (dropCount * GemField.DURATION_CASCADE);
                animDistance = (dropCount * ht);

                animStartY = (y - animDistance);
                animY = animStartY;
                break;
            }
            // Gems will dissapear after a random delay
            case STATE_CAVE_IN:
            {
                animDelay = gameEngine.genRandomNumber( GemField.DURATION_CAVE_IN );
                break;
            }
        }

        // Changing states will always turn off flashing
        flash( false, false);

        // Clearing animation is on last row
        if ( newState == STATE_CLEAR )
        {
            index = (byte) (numberOfTypes * 2);
        }
        // Select animation is on first row for each gem type, normal state
        // uses first frame of select animation. Powerup animation is second
        // row for each gem type.
        //
        else
        {
            index = (byte) (gemType * 2);
        }

        markDirty();
    }

    /**
     * Load the gem strip containing all gem images and return the
     * dimensions (assumes width == height) of an individual gem.
     *
     * @return Width (and height) of a single gem image.
     * @throws IOException if image loading fails
     */
    static public int loadGemStrip() throws IOException {
        if(!initialized) {
            // Read app-specific properties
            numAnimFrames = NUM_FRAMES;
            animRate = ANIM_RATE;

            // Load animation images
            initGemStrip();
            initialized = true;
        }

        // Assume animations are always 8 accross.
        return(size());
    }

    /**
     * Marks this gem as dirty and therefore requiring a redraw. The cursor will
     * also be redrawn if this gem currently has focus.
     */
    public void markDirty()
    {
        super.markDirty();

        cursorDirty = hasFocus;
    }

    /**
     * Handles all drawing for the gem. If the gem is currently performing an image
     * animation (animating frames in place), the background is erased and then the
     * correct animation frame is drawn. If the gem is currently involved in a movement
     * animation (static frame traveling across the screen), the gem is drawn at the
     * correct location and background erasing is left up to the gem field. If the gem
     * currently has cursor focus, the cursor is drawn as well. If the gem is a remote
     * gem, then a simple colored rectangle is drawn in place of the image.
     *
     * @param   gc  graphics context used for drawing.
     */
    public void draw( Graphics gc )
    {
        if ( dirty )
        {
            byte curState = getState();

            // States that cause gem overlap must not erase the background
            if ( curState != STATE_SWAP )
            {
                gc.setColor( bgndColor );
                gc.fillRect( x, y, wd, ht );
            }

            // Gem can be flashed on and off
            if (gemType != TYPE_EMPTY)
            {
                // Local gems use images
                switch ( curState )
                {
                    // Draw only tile background
                    case STATE_INIT:
                    case STATE_FULL_CASCADE:
                    case STATE_EMPTY:
                    {
                        break;
                    }
                    case STATE_CLEAR:
                    {
                        // Clear anim only runs through once... empty tile after
                        if(animFrame >= numAnimFrames) {
                            break;
                        }
                        // Intentional fall-thru
                    }
                    // Draw next frame of image animation sequence
                    default:
                    {
                        draw(this, gc, null, animX, animY,
                            index + ((flashOn == FALSE) ? 1 : 0), animFrame);
                        break;
                    }
                }
            }

            dirty = false;
        }

        // Cursor always drawn on top
        if ( cursorDirty == TRUE)
        {
            gc.setColor( GameBoard.COLOR_PULSE[ cursorFrame ] );
            gc.drawRect( x, y, (wd - 1), (ht - 1) );

            cursorDirty = FALSE;
        }
    }

    /**
     * Handles all time dependent functionality for the gem, primarily advancing
     * frames for the cursor and images animations as well as advancing location
     * for movement animations.
     *
     * @param   now     current time as milliseconds since unix epoch.
     */
    public void heartbeat( long now )
    {
        long elapsed = (now - stateTime);

        // If gem is flashing, toggle flash state
        if ( ((now - flashStart) < GemField.DURATION_FLASH) )
        {
            if ( (now - flashTime) > FLASH_RATE )
            {
                flashOn = (flashOn == FALSE) ? TRUE : FALSE;
                flashTime = now;

                markDirty();
            }
        }
        // Turn off after flash duration
        else
        {
            flash( false, false);
        }

        // If gem has cursor focus, pulse colors
        if ( hasFocus == TRUE )
        {
            if ( (now - cursorTime) > GameBoard.PULSE_RATE )
            {
                cursorFrame = (byte) ((cursorFrame + 1) % GameBoard.NUM_PULSE_FRAMES);
                cursorTime = now;

                cursorDirty = TRUE;
            }
        }

        switch ( getState() )
        {
            // Each gem has a random delay before appearing
            case STATE_FULL_CASCADE:
            {
                if ( elapsed > animDelay )
                {
                    setState( STATE_NORMAL );
                }

                break;
            }
            // Noting to do in normal state.
            case STATE_NORMAL:
            {
                break;
            }
            // Advance frames in image animation
            case STATE_SELECT:
            case STATE_CLEAR:
            {
                // Advance frame
                if ( (now - animTime) > animRate )
                {
                    ++animFrame;

                    if(animFrame >= numAnimFrames) {
                        // Play clear anim only once
                        if(getState() == STATE_CLEAR) {
                            animFrame = (byte) numAnimFrames;
                        } else {
                            animFrame = 0;
                        }
                    }
                    animTime = now;
                    markDirty();
                }

                break;
            }
            // During a swap, the gem moves in one of four directions towards its new location
            case STATE_SWAP:
            {
                int delta = (int) (elapsed * animDistance / GemField.DURATION_SWAP * swapNegative);

                int swapDistance = (wd * swapNegative);
                boolean swapComplete = false;

                // Prevent from moving too far
                if ( Math.abs( delta ) >= wd )
                {
                    delta = swapDistance;
                    swapComplete = true;
                }

                // Move in correct direction
                if ( swapVertical == TRUE )
                {
                    animY = ((y + delta) - ((swapReversing == TRUE) ? swapDistance : 0));
                }
                else
                {
                    animX = ((x + delta) - ((swapReversing == TRUE) ? swapDistance : 0));
                }

                // Stop movement once destination reached
                if ( swapComplete )
                {
                    // Change directions if invalid swap
                    if ( (swapValid == FALSE) && (swapReversing == FALSE) )
                    {
                        swapReversing = TRUE;
                        swapNegative = (byte) -swapNegative;

                        start(true, now);
                    }
                }

                markDirty();
                break;
            }
            // During a cascade or a new gem fill, the gem moves down at a constant rate
            // towards its correct position on the field.
            //
            case STATE_CASCADE:
            {
                animY = (int)(animStartY + (elapsed * animDistance / animDuration));

                // Stop once destination reached
                if ( animY >= y )
                {
                    animY = y;
                    setState( STATE_NORMAL );
                }

                markDirty();
                break;
            }
             // Empty state must clear background on every frame
            case STATE_EMPTY:
            {
                markDirty();
                break;
            }
            // Each gem has a random delay before dissapearing
            case STATE_CAVE_IN:
            {
                if ( elapsed > animDelay )
                {
                    setState( STATE_INIT );
                }

                break;
            }
        }
    }

    /**
     * Activates (shows) the game object and performs any necessary setup.
     * Default does nothing.
     *
     * @param   init    initial start after screen transition?
     * @param   now     current time
     */
    public void start(boolean init, long now) {
        if(!init && (pauseTime != 0)) {
            long deltaTime = now - pauseTime;
            cursorTime += deltaTime;
            animTime += deltaTime;
            flashTime += deltaTime;
        }
        super.start(init, now);
    }

    public static void initGemStrip() throws IOException {
        // Load animation images
        if(gemStrip == null) {
            gemStrip = GameEngine.getInstance().loadImage(StringTable.FILE_GEM_STRIP, false);
            numberOfTypes = (byte)((gemStrip.getHeight() - size()) / (size() * 2));
        }
    }

    public static int size() {
        return(gemStrip.getWidth() / numAnimFrames);
    }

    public static void draw(Gem gem, Graphics gc, Image bg, int x, int y, int index, int frame) {
        drawOffscreen(gem, gc, bg, x, y, index, frame);
    }

    private static void drawClipped(Gem gem, Graphics gc, int x, int y, int index, int frame) {
        int size = gem.size();
        int yMin = (gem != null) ? gem.fieldY : 0;
        int yFinal = Math.max(yMin, y);
        int ySize = (y < yMin) ? (size + (y - yMin)) : size;
        if(ySize > 0) {
            gameEngine.drawClippedImage(gc, gemStrip, x, yFinal, size, ySize,
                frame * size, index * size);
        }
    }

    private static Image offscreenImage;

    private static void drawOffscreen(Gem gem, Graphics gc, Image bg, int x, int y, int index, int frame) {
        // HACK: Moving gems are not easy to handle, so revert to old method.
        if((gem != null) &&
            ((gem.getState() == STATE_SWAP) ||
            (gem.getState() == STATE_CASCADE))) {
            drawClipped(gem, gc, x, y, index, frame);
            return;
        }

        int size = gem.size();
        if(offscreenImage == null) {
            offscreenImage = Image.createImage(size, size);
        }
        Graphics imageGC = offscreenImage.getGraphics();
        if(bg != null) {
            imageGC.drawImage(bg, -x, -y, Graphics.TOP | Graphics.LEFT);
        } else if(gem != null) {
            imageGC.setColor(gem.bgndColor);
            imageGC.fillRect(0, 0, size, size);
        }
        imageGC.drawImage(gemStrip,
            -(frame * size), -(index * size), Graphics.TOP | Graphics.LEFT);
        gc.drawImage(offscreenImage, x, y, Graphics.TOP | Graphics.LEFT);
    }

}

