/*
 *     Copyright (c)2001-2002 DemiVision, LLC. All Rights Reserved.
 *
 * The information contained herein is the CONFIDENTIAL and PROPRIETARY
 *                  information of DemiVision, LLC.
 */

package com.bejeweled2_j2me;

import javax.microedition.lcdui.*;
import java.io.IOException;


/**
 * The <tt>MainScreen</tt> encapsulates all user interface functionality for
 * the <i>Dioskilos</i> main application menu. This menu allows the interface
 * to branch to the following locations:
 *
 * <ul>
 *   <li>Multiplayer</li>
 *   <li>Single Player</li>
 *   <li>Settings</li>
 *   <li>How to Play</li>
 *   <li>Credits</li>
 * </ul>
 *
 * The MainScreen also handles the help menu subsystem, which displays a list
 * of help topics when the "How To Play" main menu item is selected. Each topic
 * displays a dialog containing help text from an external text resource file.
 * Because MainScreen is a system screen, it is automatically managed by the
 * <tt>GameEngine</tt>. An example of the main menu follows:
 *
 * <table cellspacing="20"><tr>
 *   <td align="center"><img src="doc-files/MainScreen-1.png"><br>Main Menu</td>
 * </tr></table>
 *
 * In most cases, no customization of MainScreen is required. <i>It is most efficient
 * to use only the default MainScreen whenever possible</i>. However, developers can
 * provide custom functionality when necessary by extending MainScreen. The custom
 * screen must be registered with the GameEngine by overriding
 * {@link DioskilosMIDlet#registerScreens()}.
 *
 * @see         GameEngine
 * @see         DioskilosMIDlet
 *
 * @author      Barry Sohl
 * @version     1.1.0
 */
class MainScreen extends BaseScreen
{
    /* Constants */

    // Screen states
    private final byte STATE_MAIN = 0;
    private final byte STATE_HELP = 1;
    private final byte STATE_DEMO = 2;

    private static final int MENU_GET_GAME  = 0;
    private static final int MENU_PLAY      = 1;
    private static final int MENU_GAME_MODE = 2;
    private static final int MENU_SETTINGS  = 3;
    private static final int MENU_HELP      = 4;
    private static final int MENU_CREDITS   = 5;

    private DialogCanvas mainMenuList;

    private Command     mainCmd;
    private Command     resetCmd;
    private Command     exitCmd;
    private Command     buyCmd;

    // Current high score.
    private int highScore;

    public static boolean showTrialOver = false;

    public MainScreen() {
        mainMenuList = new DialogCanvas(Dialog.getInstance(), DialogCanvas.LIST, "");
        if (Build.VODAFONE) {
            mainMenuList.setList(new String[] {
                StringTable.VODAFONE_GET_GAME, // Get the Game!
                StringTable.VODAFONE_PLAY, // Start Trial
                StringTable.GAME_MODE, // Game Mode
                StringTable.OPTION_TITLE, // Options
                StringTable.PAUSE_HELP, // He`lp
                StringTable.CREDITS_TITLE, // About/Credits
            }, null, DeviceSpecific.MAIN_MENU_Y_OFFSET);
        }
        else {
            mainMenuList.setList(new String[] {
                StringTable.PLAY, // Play
                StringTable.GAME_MODE, // Game Mode
                StringTable.OPTION_TITLE, // Options
                StringTable.PAUSE_HELP, // Help
                StringTable.CREDITS_TITLE, // About/Credits
            }, null, DeviceSpecific.MAIN_MENU_Y_OFFSET);
        }
    }

    /**
     * Returns a reference to the main menu or help menu to be used as the
     * displayable widget for this screen depending on the current state.
     *
     * @return  reference to displayable to be given focus.
     */
    public Displayable getDisplayable()
    {
        if (getState() == STATE_DEMO) {
            if (Build.VODAFONE) {
                showGetItScreen();
                return dlg.getDisplayable();
            }
            else {
                gameEngine.exit();
                return null;
            }
        }
        else {
            dlg.show(this, Dialog.DLG_MAIN_MENU, "", "");
            return mainMenuList;
        }
    }

    private void showGetItScreen() {
        DialogListener listener = new DialogListener() {
            public void dialogDismissed( byte result ) {
                if (result == Dialog.CMD_OK) {
                    Build.showBuyURIAndExit(gameEngine);
                }
                else if (result == Dialog.CMD_CANCEL) {
                    gameEngine.exit();
                }
            }
        };

        String message = StringTable.VODAFONE_GET_GAME + "|" + StringTable.VODAFONE_DOWNLOAD;
        if (showTrialOver) {
            showTrialOver = false;
            message = StringTable.VODAFONE_TRIAL_OVER + "||" + message;
        }

        dlg.show(listener, Dialog.DLG_VODA_GET_EXIT, "BEJEWELED", message);
    }


    private void showExitConfirm() {
        DialogListener listener = new DialogListener() {
            public void dialogDismissed( byte result ) {
                if (result == Dialog.CMD_YES) {
                    gameEngine.exit();
                }
                else if (result == Dialog.CMD_NO) {
                    gameEngine.transition( GameEngine.SCREEN_MAIN );
                }
            }
        };
		dlg.show(listener, Dialog.DLG_YESNO, StringTable.EXIT_TITLE,
			StringTable.EXIT_MESSAGE);
    }

    /**
     * Creates the interface and layout for the main and help menus.  All
     * resources for the screen will be loaded and/or allocated here.
     *
     * @throws  IOException if error occurs loading resources.
     */
    public void init() throws IOException
    {
        // Setup main event handling
        mainCmd = new Command(StringTable.SPACE, Command.OK, 0);
        resetCmd = new Command(StringTable.RESET_SOFTKEY, Command.SCREEN, 0);

        buyCmd = new Command( StringTable.BUY_SOFTKEY, Command.SCREEN, 0 );

        exitCmd = new Command(StringTable.EXIT_SOFTKEY, Command.CANCEL, 0);
    }

    /**
     * Change the current high score.
     *
     * @param score new high score
     */
    public void setHighScore(int score) {
        highScore = score;
    }

    public int getHighScore() {
        return highScore;
    }

    /**
     * Handles presses of the dialog buttons for the main screen. Primarily
     * dismissing the how-to and credits dialogs.
     *
     * @param   result  resulting command type code.
     */
    public void dialogDismissed( byte result )
    {
        if (result == Dialog.CMD_NO) {
            // Exit softkey
            if (Build.VODAFONE) {
                showGetItScreen();
            }
            else {
				showExitConfirm();
            }
        }
        else if (result == Dialog.CMD_OK) {
            int index = mainMenuList.getSelectedIndex();
            if (!Build.VODAFONE) {
                // No "Get Game" option
                index++;
            }
            switch (index) {
                case MENU_GET_GAME:
                    Build.showBuyURIAndExit(gameEngine);
                    break;
                case MENU_PLAY:
                    if (Build.VODAFONE) {
                        DialogListener listener = new DialogListener() {
                            public void dialogDismissed( byte result ) {
                                if (result == Dialog.CMD_OK) {
                                    gameEngine.transition( GameEngine.SCREEN_GAME );
                                    setState(STATE_DEMO);
                                }
                                else if (result == Dialog.CMD_CANCEL) {
                                    gameEngine.exit();
                                }
                            }
                        };
                        dlg.show(listener, Dialog.DLG_VODA_START_EXIT, "BEJEWELED",
                            StringTable.VODAFONE_TRIAL + "|" + StringTable.VODAFONE_TRIAL_DESC);
                    }
                    else {
                        gameEngine.transition( GameEngine.SCREEN_GAME );
                        if(gameEngine.isDemo()) {
                            // Come back to demo end after starting a game
                            setState(STATE_DEMO);
                        }
                    }
                    break;
                case MENU_GAME_MODE:
                    gameEngine.transition( GameEngine.SCREEN_GAME_MODE);
                    break;
                case MENU_SETTINGS:
                    gameEngine.transition( GameEngine.SCREEN_SETTINGS );
                    break;
                case MENU_HELP:
                    dlg.showHelp(new DialogListener() {
                        public void dialogDismissed( byte result ) {
                            if (result == Dialog.CMD_NO) {
                                gameEngine.transition( GameEngine.SCREEN_MAIN );
                            }
                        }
                    });
                    break;
                case MENU_CREDITS:
                    dlg.showTextResource(new DialogListener() {
                        public void dialogDismissed( byte result ) {
                            gameEngine.restart( GameEngine.SCREEN_MAIN );
                        }
                    }, StringTable.CREDITS_TITLE, StringTable.FILE_CREDITS, 0 );
                    break;
            }
        }
    }
}
