/*
 *    Copyright (c)2001-2003 Jamdat Mobile, Inc. All Rights Reserved.
 *
 * The information contained herein is the CONFIDENTIAL and PROPRIETARY
 *                 information of Jamdat Mobile, Inc.
 */

package com.bejeweled2_j2me;

import javax.microedition.lcdui.Graphics;
import javax.microedition.lcdui.Image;
import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;


/**
 * The TimeBar represents the primary mechanism for game progress and end of
 * game detection. It has a time component, which expands in real-time from the
 * start of the game, and a power component, which is affected by scoring in
 * the game. If a player's power component reaches zero, they lose. The TimeBar
 * has both single player and multiplayer variations.
 *
 * @author      DemiVision: Barry Sohl
 * @version     1.0.4
 */
final class TimeBar extends GameObject
{
    /* Constants */

    // Tweaking knobs
    private int TOTAL_TIME;

    // Rate based on difficulty level.
    private final int[] DIFFICULTY_FACTOR = { 1, 1, 2 };

    // Starting time percentage based on difficulty level
    private final long[] START_TIME_PERCENT = { 0, 60, 40 };

    // Near-death sound pulsing
    private final int SOUND_INTERVAL = 1500;

    // Power bar stages
    private final int NUM_POWER_LEVELS = 3;
    private final byte[] POWER_LEVELS = { 100, 50, 25, 10 };

    // Power bar stage colors
    private final int[] COLOR_LEVEL = {
                                        0x00FF00,
                                        0xFFFF00,
                                        0xFF0000
                                      };
    // Time bar pulsing colors
    private static final int[] COLOR_TIME = {
                                              0x75D5F9,
                                              0x00B2D8,
                                              0x0000FF,
                                              0x0000C0,
                                              0x000080
                                            };
    // Misc. colors
    private final int COLOR_BORDER   = 0x404040;

    // Internal states
    private final byte STATE_INIT  = 0;
    private final byte STATE_START = 1;

    /* Data Fields */
    private GameBoard gameBoard;        // External references

    private int elapsedTime;            // Real-time component
    private int timeWd;
    private int prevTimeWd;
    private long startTime;
    private boolean isReady;

    private boolean firstUpdate;

    private int localColor;             // Power bar stage colors

    private boolean pulsing;            // Time bar color pulsing
    private int pulseFrame;
    private long pulseTime;

    private long lastSound;             // Near-death effects

    private Image skullImg;             // Image resources
    private int skullWd;


    /**
     * Constructor saves back pointer to parent container.
     *
     * @param   gameBoard   reference to parent object.
     */
    TimeBar( GameBoard gameBoard )
    {
        this.gameBoard = gameBoard;
    }

    /**
     * Start the time bar moving.
     */
    void ready() {
        isReady = true;
        markDirty();
    }

    /**
     * Starts movement of the time bar. In multiplayer mode, this does not occur
     * until both players have been synchronized at the start of the game.
     *
     * @param isGameStart is this the first initialization for this game?
     */
    void initTime(boolean isGameStart)
    {
        setState( STATE_START );

        long now = System.currentTimeMillis();

        // Reset timing based on difficulty
        if(gameBoard.getDifficulty() != GameBoard.DIFFICULTY_NORMAL) {
            long percent = START_TIME_PERCENT[gameBoard.getDifficulty()];
            elapsedTime = (int)((getTotalTime() * (100 - percent)) / 100);
        } else {
            elapsedTime = 0;
        }
        startTime = now - elapsedTime;

        // Sound effects
        if(isGameStart) {
            firstUpdate = true;
            gameEngine.playMidi( GameBoard.MIDI_START, false, false);
            gameEngine.playAlertSound( GameBoard.ALERT_START, false);
        }
    }

    /**
     * Add bonus time back to the time bar (or add time for normal mode).
     *
     * @param percentage of TOTAL_TIME bonus adds
     */
    void addTimeBonus( int percentage )
    {
        int bonus = (percentage * getTotalTime()) / 100;
        if(gameEngine.isDemo()) {
            bonus *= 4;
        }
        if(gameBoard.getDifficulty() != GameBoard.DIFFICULTY_NORMAL) {
            startTime += bonus;
        } else {
            updateTime(elapsedTime + bonus);
        }
    }

    /**
     * Updates the time bar interface. The size and color of all three bar
     * components (local power, time, and remote power) will be updated to
     * reflect the current status. This method can be used to adjust the
     * time relative to another time clock in order to keep the local and
     * remote players in sync given latency differences.
     *
     * @param   elapsed     total time elapsed (ms) since start of game.
     */
    void updateTime( int elapsed )
    {
        // Update internal clock, time may be an internal value or
        // adjusted from the remote player's clock.
        //
        elapsedTime = elapsed;

        int halfWd = (wd / 2);

        // Calculate new size of time component
        timeWd = (elapsedTime * halfWd / (getTotalTime() / 2));
        timeWd = Math.max( Math.min( timeWd, wd ), 0 );

        // Reverse time with for nomal mode
        if(gameBoard.getDifficulty() == GameBoard.DIFFICULTY_NORMAL) {
            timeWd = wd - timeWd;
        }

        // Only redraw if necessary
        if (( timeWd != prevTimeWd ) || firstUpdate)
        {
            markDirty();

            prevTimeWd = timeWd;
        }

        int factor = 100;

        // Power level is calculated as the percent of the total
        // potential width of the power bar.
        //
        int localPowerWd = wd - timeWd;
        gameBoard.p1Power = (byte) Math.min( Math.max( (localPowerWd * factor / wd), 0 ), 100 );

        // Protect against instant game end
        if ( !firstUpdate )
        {
            // Game is over if player's power bar is depleted except for normal mode
            if ( ( gameBoard.p1Power <= 0 ) &&
                ( gameBoard.getDifficulty() != GameBoard.DIFFICULTY_NORMAL ) )
            {
                gameBoard.gameOver();
            }
            else
            {
                // Determine colors for both power bars
                for ( int i = 0; i < NUM_POWER_LEVELS; i++ )
                {
                    if ( gameBoard.p1Power < POWER_LEVELS[i] )
                    {
                        localColor = COLOR_LEVEL[i];
                    }
                }
            }

            // Time bar pulses during last 10%
            pulsing = (gameBoard.p1Power < POWER_LEVELS[ NUM_POWER_LEVELS ]);
        }

        firstUpdate = false;
    }

    /* GameObject */

    /**
     * Performs once-only initialization for the time bar. Tweaking values
     * are read from the external properties.
     *
     * @throws  IOException if error occurs loading resources.
     */
    public void init() throws IOException
    {
        TOTAL_TIME = 180000;

        // Load images
        skullImg = gameEngine.loadImage( StringTable.FILE_SKULL, false);
        skullWd = (skullImg.getWidth() / 2);
    }

    /**
     * Called at the start of each game. Resets time and power values.
     *
     * @param   offset      level offset of current game node.
     */
    public void startNode( byte offset )
    {
        setState( STATE_INIT );

        // Determine starting bar positions
        timeWd = 0;

        // Both time bars start green
        localColor = COLOR_LEVEL[0];

        // Initialize time, but wait until we are ready to start
        initTime(true);
        isReady = false;
    }

    /**
     * Handles all drawing for the time bar. The local and remote power
     * components, along with the real-time time component, are drawn in
     * their current color and size.
     *
     * @param   gc  graphics context used for drawing.
     */
    public void draw( Graphics gc )
    {
        if ( dirty )
        {
            // Different color placement for normal vs. other difficulties
            int fillColor = localColor;
            int emptyColor =
                pulsing ? GameBoard.COLOR_PULSE[pulseFrame] : COLOR_TIME[pulseFrame];
            if(gameBoard.getDifficulty() == GameBoard.DIFFICULTY_NORMAL) {
                fillColor = emptyColor;
                emptyColor = COLOR_LEVEL[0];
            }

            // Draw local power bar
            int timeX = x + (wd - timeWd);

            gc.setColor( fillColor );
            gc.fillRect( x, y, timeX - 1, ht );

            // Draw time bar (pulsing colors)
            int timeFullWd = Math.min( timeWd + 1 , (x + wd - timeX) );

            gc.setColor( emptyColor );
            gc.fillRect( timeX, y, timeFullWd, ht );

            // Draw outlines
            int ht2 = (ht - 1);

            gc.setColor( COLOR_BORDER );
            gc.drawRect( x, y, wd, ht2 );
            gc.drawRect( timeX, y, timeFullWd, ht2 );

            // Erase midpoint overhang and underhang
            int x2 = (x + wd + 1);
            int y2 = (y - 1);
            int y3 = (y + ht);

            gc.setColor( GameBoard.COLOR_BGND );
            gc.drawLine( x, y2, x2, y2 );
            gc.drawLine( x, y3, x2, y3 );

            int x3 = (timeX - skullWd);

            // Draw local skull, only if we are ready to play though
            if(isReady) {
                gc.drawImage( skullImg, x3, y2, GameEngine.TOP_LEFT );
            }

            dirty = false;
        }
    }

    /**
     * Handles all time dependent functionality for the time bar, primarily
     * updating the real-time time component, pulsing the bar colors, and
     * sending time sync events.
     *
     * @param   now     current system time in milliseconds.
     */
    public void heartbeat( long now )
    {
        // Heartbeat only happens after we have start and not when we
        // are paused.
        if ( ( getState() == STATE_START ) && ( pauseTime == 0) )
        {
            // Update time bar, adjust for difficulty level.
            if(gameBoard.getDifficulty() != GameBoard.DIFFICULTY_NORMAL) {
                if(!isReady) {
                    startTime = now - elapsedTime;
                }
                updateTime( (int) (now - startTime) );
            } else {
                updateTime(elapsedTime);
            }

            // Pulse fill colors at 10fps
            if ( (now - pulseTime) > GameBoard.PULSE_RATE )
            {
                pulseFrame = (pulseFrame + 1) % GameBoard.NUM_PULSE_FRAMES;
                pulseTime = now;

                markDirty();
            }

            // Near-death sound played at increasing rate as local player approaches death
            if ( isNearDeath() )
            {
                if ( (now - lastSound) > SOUND_INTERVAL )
                {
                    gameEngine.playMidi( GameBoard.MIDI_NEAR_DEATH, false, false);
                    gameEngine.playAlertSound( GameBoard.ALERT_NEAR_DEATH, false);

                    lastSound = now;
                }
            }
        }
    }

    /**
     * Check if player is near death.
     *
     * @return true if timed game and time is less than lowest power level
     */
    public boolean isNearDeath() {
        return (gameBoard.getDifficulty() != GameBoard.DIFFICULTY_NORMAL ) &&
            (gameBoard.p1Power < POWER_LEVELS[ NUM_POWER_LEVELS ]) &&
            (gameBoard.p1Power > 0);
    }
    
    /**
     * Resumes progression of the time bar during single player gameplay.
     * Multiplayer gameplay should never be paused.
     *
     * @param   init    initial start after transition?
     * @param   now     current system time
     */
    public void start(boolean init, long now) {
        if(!init && (pauseTime != 0)) {
            long deltaTime = now - pauseTime;
            startTime += deltaTime;
            pulseTime += deltaTime;
        }
        super.start(init, now);
    }

    /**
     * Get the total time length the bar represents.
     *
     * @return number of milliseconds
     */
    public int getTotalTime() {
        return TOTAL_TIME / DIFFICULTY_FACTOR[gameBoard.getDifficulty()];
    }

}

/**/
