package shell.core;

import java.io.InputStream;
import java.util.Vector;
import javax.microedition.lcdui.Display;
import javax.microedition.lcdui.game.GameCanvas;
import javax.microedition.lcdui.Graphics;
import javax.microedition.lcdui.Image;

public class App extends GameCanvas implements Runnable {
    
    public static final int STATE_INIT = 0;
    public static final int STATE_ACTIVE = 1;
    public static final int STATE_PAUSED = 2;
    public static final int STATE_DESTROYED = 3;
    
    public static final int VK_0 = 0;
    public static final int VK_1 = 1;
    public static final int VK_2 = 2;
    public static final int VK_3 = 3;
    public static final int VK_4 = 4;
    public static final int VK_5 = 5;
    public static final int VK_6 = 6;
    public static final int VK_7 = 7;
    public static final int VK_8 = 8;
    public static final int VK_9 = 9;
    public static final int VK_POUND = 10;
    public static final int VK_STAR = 11;
    public static final int VK_OK = 12;
    public static final int VK_SOFT1 = 13;
    public static final int VK_SOFT2 = 14;
    public static final int VK_LEFT = 15;
    public static final int VK_RIGHT = 16;
    public static final int VK_UP = 17;
    public static final int VK_DOWN = 18;
    public static final int NUM_KEYS = 19;
    
    private static final byte KEY_STATE_UP = 0;
    private static final byte KEY_STATE_DOWN = 1;
    private static final byte KEY_STATE_PRESSED = 2;
    private static final byte KEY_STATE_RELEASED = 3;
    
    private static int currState = STATE_INIT;
    private static int nextState = STATE_INIT;
    private static Scene currScene = null;
    private static Scene nextScene = null;
    private static Vector sceneStack = new Vector();
    private static byte[] keyStates = new byte[NUM_KEYS];
    
    private static int pressedKeys;
    private static int releasedKeys;
    
    private static boolean isShown;
    
    private static Image brokenImage;
    
    public App() {
        super(false);
        setFullScreenMode(true);
    }

    
    public void run() {
        
        Graphics g = getGraphics();
        long lastTime = System.currentTimeMillis();
        
        while (currScene != null || nextScene != null) {
            
            try {
                boolean changed = changeState(nextState);
                changed |= changeScene(nextScene);
                
                if (!isShown || currScene == null || currState != STATE_ACTIVE) {
                    try {
                        Thread.sleep(10);
                    }
                    catch (InterruptedException ex) { }
                }
                else if (changed) {
                    lastTime = System.currentTimeMillis();
                    clearInput();
                    currScene.updateScene(0);
                    currScene.draw(this, g, true);
                }
                else {
                    processInput();
                    int elapsedTime = (int)(System.currentTimeMillis() - lastTime);
                    lastTime += elapsedTime;
                
                    currScene.updateScene(elapsedTime);
                    boolean somethingDrawn = currScene.draw(this, g, false);
                    
                    if (!somethingDrawn) {
                        try {
                            Thread.sleep(10);
                        }
                        catch (InterruptedException ex) { }
                    }
                }
            }
            catch (Throwable t) {
                clearSceneStack();
                currScene = null;
                nextScene = null;
                t.printStackTrace();
            }
        }
        
        // Reset
        clearInput();
        clearSceneStack();
        nextState = STATE_INIT;
        
        if (currState == STATE_DESTROYED) {
            currState = STATE_INIT;
            Main.instance.notifyDestroyed();
        }
        
        currState = STATE_INIT;
    }
    
    
    // Package-private
    static void setState(int newState) {
        
        if (nextState == newState) {
            return;
        }
        
        nextState = newState;
        
        if (nextState == STATE_ACTIVE && currState == STATE_INIT) {
            currScene = null;
            nextScene = null;
            App app = new App();
            
            Display.getDisplay(Main.instance).setCurrent(app);
            
            setScene(Main.createFirstScene());
            
            new Thread(app).start();
        }
    }
    
    //
    // These public methods must be called from the animation thread
    //
    
    public static String getProperty(String name) {
        return Main.instance.getAppProperty(name);
    }
    
    
    public static int getIntProperty(String name, int defaultValue) {
        String s = getProperty(name);
        if (s == null) {
            return defaultValue;
        }
        try {
            return Integer.parseInt(s);
        }
        catch (NumberFormatException ex) {
            return defaultValue;
        }
    }
    
    public static void launchBrowser(String url) {
		try {
			Main.instance.platformRequest(url);
		}
		catch (Exception e) {
		}
        exit();
	}    
    
    
    public static boolean isPressed(int keyCode) {
        byte state = keyStates[keyCode];
        return (state == KEY_STATE_PRESSED);
    }
    
    
    public static boolean isReleased(int keyCode) {
        byte state = keyStates[keyCode];
        return (state == KEY_STATE_RELEASED);
    }
    
    
    public static boolean isDown(int keyCode) {
        byte state = keyStates[keyCode];
        return (state == KEY_STATE_DOWN || state == KEY_STATE_PRESSED);
    }
    
    
    public static void exit() {
        setState(App.STATE_DESTROYED);
    }
    
    
    public static void setScene(Scene newScene) {
        nextScene = newScene;
    }
    
    
    public static void pushScene(Scene newScene) {
        if (currScene != null) {
            sceneStack.addElement(currScene);
        }
        setScene(newScene);
    }
    
    
    public static void popScene() {
        if (sceneStack.size() > 0) {
            setScene((Scene)sceneStack.elementAt(sceneStack.size() - 1));
            sceneStack.removeElementAt(sceneStack.size() - 1);
        }
    }

    
    public static boolean isSceneStackEmpty() {
        return (sceneStack.size() == 0);
    }
    
    
    public static void clearSceneStack() {
        sceneStack.removeAllElements();
    }
    
    
    //
    // These private methods must be called from the animation thread
    //
    
    
    private void clearInput() {
        for (int i = 0; i < NUM_KEYS; i++) {
            keyStates[i] = KEY_STATE_UP;
        }
        
        synchronized (keyStates) {
            pressedKeys = 0;
            releasedKeys = 0;
        }
    }
    
    
    private void processInput() {
        
        int savedPressedKeys;
        int savedReleasedKeys;
        
        synchronized (keyStates) {
            savedPressedKeys = pressedKeys;
            savedReleasedKeys = releasedKeys;
            pressedKeys = 0;
            releasedKeys = 0;
        }
        
        for (int i = 0; i < NUM_KEYS; i++) {
            int bitMask = (1 << i);
            byte state = keyStates[i];
            boolean isPressed = (savedPressedKeys & bitMask) != 0;
            boolean isReleased = (savedReleasedKeys & bitMask) != 0;
            
            if (isPressed) {
                state = KEY_STATE_PRESSED;
                
                if (isReleased) {
                    // Save for next time
                    synchronized (keyStates) {
                        releasedKeys |= bitMask;
                    }
                }
            }
            else if (isReleased) {
                if (state == KEY_STATE_PRESSED || state == KEY_STATE_DOWN) {
                    state = KEY_STATE_RELEASED;
                }
                else {
                    state = KEY_STATE_UP;
                }
            }
            else {
                if (state == KEY_STATE_PRESSED) {
                    state = KEY_STATE_DOWN;
                }
                else if (state != KEY_STATE_DOWN) {
                    state = KEY_STATE_UP;
                }
            }
            
            keyStates[i] = state;
        }
    }
    
    
    private boolean changeState(int newState) {
        if (currState == newState || currState == STATE_DESTROYED) {
            return false;
        }
        
        currState = newState;
        
        if (currState == STATE_DESTROYED) {
            clearSceneStack();
            setScene(null);
        }
        
        return (currState == STATE_ACTIVE);
    }
    
    
    private boolean changeScene(Scene newScene) {
        
        if (newScene == currScene) {
            return false;
        }
        
        if (currScene != null) {
            currScene.unload();
        }
        currScene = newScene;
        if (currScene != null) {
            currScene.load();
            // Check if we exited on load()
            if (nextState == STATE_DESTROYED) {
                currState = STATE_DESTROYED;
                nextScene = null;
                currScene = null;
            }
        }
        
        return true;
    }
    
    
    //
    // Canvas methods
    //
    
    
    protected void showNotify() {
        isShown = true;
        if (nextState != STATE_DESTROYED) {
            setState(App.STATE_ACTIVE);
        }
    }
    
    
    protected void hideNotify() {
        isShown = false;
        if (nextState != STATE_DESTROYED) {
            setState(App.STATE_PAUSED);
        }
    }
    
    
    protected void keyPressed(int keyCode) {
        //System.out.println("Pressed: " + keyCode);
        
        int keyIndex = convertKeyCode(keyCode);
        if (keyIndex != -1) {
            synchronized (keyStates) {
                pressedKeys |= (1 << keyIndex);
            }
        }
    }
    
    
    protected void keyReleased(int keyCode) {
        //System.out.println("Released: " + keyCode);
        
        int keyIndex = convertKeyCode(keyCode);
        if (keyIndex != -1) {
            synchronized (keyStates) {
                releasedKeys |= (1 << keyIndex);
            }
        }
    }
    
    
    private int convertKeyCode(int keyCode) {
        // Converts a device-specific keycode to an index in the keyStates array.
        switch (keyCode) {
            case Device.KEY_CODE_OK:    return VK_OK;
            case Device.KEY_CODE_SOFT1: return VK_SOFT1;
            case Device.KEY_CODE_SOFT2: return VK_SOFT2;
            case Device.KEY_CODE_LEFT:  return VK_LEFT;
            case Device.KEY_CODE_RIGHT: return VK_RIGHT;
            case Device.KEY_CODE_UP:    return VK_UP;
            case Device.KEY_CODE_DOWN:  return VK_DOWN;
            case GameCanvas.KEY_POUND:  return VK_POUND;
            case GameCanvas.KEY_STAR:   return VK_STAR;
            case GameCanvas.KEY_NUM0:   return VK_0;
            case GameCanvas.KEY_NUM1:   return VK_1;
            case GameCanvas.KEY_NUM2:   return VK_2;
            case GameCanvas.KEY_NUM3:   return VK_3;
            case GameCanvas.KEY_NUM4:   return VK_4;
            case GameCanvas.KEY_NUM5:   return VK_5;
            case GameCanvas.KEY_NUM6:   return VK_6;
            case GameCanvas.KEY_NUM7:   return VK_7;
            case GameCanvas.KEY_NUM8:   return VK_8;
            case GameCanvas.KEY_NUM9:   return VK_9;
            default: return -1;
        }
    }
    
    //
    // Utility Methods
    //
    
    public static Image createImage(String resource) {
        try {
            return Image.createImage(getResourceName(resource));
        }
        catch (Exception ex) {
            return getBrokenImage();
        }
    }
    
    
    public static InputStream getResource(String resource) {
        return Main.instance.getClass().getResourceAsStream(getResourceName(resource));
    }
    
    
    private static String getResourceName(String resource) {
        if (resource.charAt(0) == '/') {
            return "/shell" + resource;
        }
        else {
            return "/shell/" + resource;
        }
    }
    
    
    /** Creates an IE-like red "X" image */
    public static Image getBrokenImage() {
        if (brokenImage == null) {
            int size = 24;
            
            brokenImage = Image.createImage(size, size);
            Graphics g = brokenImage.getGraphics();
            g.setColor(0xffffff);
            g.fillRect(0, 0, size, size);
            g.setColor(0x000000);
            g.drawRect(0, 0, size-1, size-1);
            g.setColor(0xff0000);
            g.drawLine(2, 2, size-3, size-3);
            g.drawLine(size-3, 2, 2, size-3);
        }
        
        return brokenImage;
    }    
}