package shell.core;

import java.io.DataInputStream;
import java.io.InputStream;
import java.io.IOException;
import java.lang.StringBuffer;
import java.util.Vector;
import javax.microedition.lcdui.Graphics;
import javax.microedition.lcdui.Image;

public class ImageFont {
    
    private String chars;
    private short[] offsets;
    private Image image;
    private int tracking;
    private int bgColor;
    
    /**
        Creates an image font of the specified name. This constructor loads both the font
        png file and the data file. For example, if your font name is "MyFont",
        "/MyFont.png" and "/MyFont.data" is loaded.
        
        See the FontCreator in the tools directory to create image fonts.
    */
    public ImageFont(String name) {
        image = App.createImage("/" + name + ".png");
        InputStream is = App.getResource("/" + name + ".data");
        
        if (is != null) {
            try {
                StringBuffer buffer = new StringBuffer();
                DataInputStream in = new DataInputStream(is);
                
                bgColor = in.readInt();
                if (bgColor == 0) {
                    bgColor = -1;
                }
                else {
                    bgColor &= 0xffffff;
                }
                tracking = in.readUnsignedByte();
                int numChars = in.readUnsignedShort();
                
                for (int i = 0; i < numChars; i++) {
                    buffer.append(in.readChar());
                }
                offsets = new short[numChars + 1];
                for (int i = 0; i < numChars + 1; i++) {
                    offsets[i] = (short)in.readShort();
                }
                in.close();
                chars = buffer.toString();
                
                
                return;
            }
            catch (IOException ex) {
                // Fall through
            }
        }
            
        image = App.getBrokenImage();
        chars = "";
        offsets = null;
    }
    
    
    private int getIndex(char ch) {
        int index = chars.indexOf(ch);
        if (index == -1) {
            index = chars.indexOf('X');
            if (index == -1) {
                return 0;
            }
        }
        return index;
    }
    
    
    public int getBackgroundColor() {
        return bgColor;
    }
    
    
    public boolean canDisplay(char ch) {
        return (chars.indexOf(ch) != -1);
    }
    
    
    public int getWidth(String s) {
        return getWidth(s, 0, s.length());
    }
    
    
    public int getWidth(String s, int start, int length) {
        int width = 0;
        int height = getHeight();
        for (int i = start; i < start + length; i++) {
            int charWidth = getWidth(s.charAt(i));
            if (charWidth > 0) {
                if (width == 0) {
                    width = charWidth;
                }
                else {
                    width += tracking + charWidth;
                }
            }
        }
        return width;
    }
    
    
    public int getWidth(char ch) {
        int index = getIndex(ch);
        if (index == -1) {
            return 0;
        }
        return offsets[index + 1] - offsets[index];
    }
    
    
    public int getHeight() {
        return image.getHeight();
    }
    
    
    public void draw(Graphics g, String text, int x, int y) {
        int clipX = g.getClipX();
        int clipY = g.getClipY();
        int clipWidth = g.getClipWidth();
        int clipHeight = g.getClipHeight();
        
        if (text == null) {
            text = "";
        }
        int height = getHeight();
        for (int i = 0; i < text.length(); i++) {
            int index = getIndex(text.charAt(i));
            if (index != -1) {
                if (x != 0 && tracking > 0) {
                    if (bgColor >= 0) {
                        g.setColor(bgColor);
                        g.setClip(x, y, tracking, height);
                        g.fillRect(x, y, tracking, height);
                    }
                    x += tracking;
                }
                int charX = offsets[index];
                int charWidth = offsets[index + 1] - charX;
                g.setClip(x, y, charWidth, height);
                g.drawImage(image, x - charX, y, Graphics.TOP | Graphics.LEFT);
                
                x += charWidth;
            }
        }
        
        g.setClip(clipX, clipY, clipWidth, clipHeight);
    }
    
    
    public Image renderText(String text) {
        return renderText(text, null, 0, 0);
    }
    
    
    public Image renderText(String text, Image background, int backgroundX, int backgroundY) {
        if (offsets == null) {
            return image;
        }
        
        Image renderedText;
        if (text == null || text.length() == 0) {
            renderedText = Image.createImage(1, getHeight());
            if (bgColor >= 0) {
                Graphics g = renderedText.getGraphics();
                g.setColor(bgColor);
                g.fillRect(0, 0, 1, getHeight());
            }
        }
        else {
            renderedText = Image.createImage(getWidth(text), getHeight());
            Graphics g = renderedText.getGraphics();
            int oldBgColor = bgColor;
            if (background != null) {
                bgColor = -1;
                g.drawImage(background, backgroundX, backgroundY, Graphics.TOP | Graphics.LEFT);
            }
            draw(g, text, 0, 0);
            bgColor = oldBgColor;
        }
        return renderedText;
    }
    
    
    /**
        Creates a two-frame Visual.
        This method assumes the two font's metrics are identical.
    */
    public static Visual createButton(String text, ImageFont normal, ImageFont highlight) {
        if (normal.offsets == null) {
            return new Visual(null);
        }
        
        int width = normal.getWidth(text);
        int height = normal.getHeight() * 2;
        
        Image renderedText = Image.createImage(width, height);
        Graphics g = renderedText.getGraphics();
        normal.draw(g, text, 0, 0);
        highlight.draw(g, text, 0, normal.getHeight());
        
        return new Visual(renderedText, 1, 2, 0, 0);
    }
    
    
    /**
        Word-wraps a line of text to multiple lines. Newline characters ('\n')
        can be used for explicit line breaks.
    */
    public String[] wordWrapText(String text, int maxWidth) {
        if (text == null) {
            return null;
        }
        Vector splitText = new Vector();
        int startIndex = 0;
        int lastGoodIndex = -1;
        
        for (int i = 0; i < text.length(); i++) {
            char ch = text.charAt(i);
            if (ch == ' ' || ch == '\t') {
                lastGoodIndex = i;
            }
            else if (ch == '\n') {
                splitText.addElement(text.substring(startIndex, i));
                startIndex = i + 1;
                lastGoodIndex = -1;
            }
            else {
                int currentLineWidth = getWidth(text, startIndex, i - startIndex + 1);
                if (currentLineWidth > maxWidth) {
                    if (lastGoodIndex != -1) {
                        // wrap this word
                        i = lastGoodIndex;
                    }
                    else if (startIndex < i) {
                        // this word doesn't fit on the line... split it
                        i--;
                    }
                    else { // if (startIndex == i)
                        // this 1 character doesn't fit on the line... add it anyway
                    }
                    splitText.addElement(text.substring(startIndex, i + 1));
                    startIndex = i + 1;
                    lastGoodIndex = -1;
                }
            }
        }
        
        // add remaining chars
        if (startIndex < text.length()) {
            splitText.addElement(text.substring(startIndex));
        }
        
        // convert to String array
        String[] retVal = new String[splitText.size()];
        splitText.copyInto(retVal);
        return retVal;
    }    
}
