package shell.core;



import javax.microedition.lcdui.game.GameCanvas;

import javax.microedition.lcdui.game.Layer;

import javax.microedition.lcdui.game.Sprite;

import javax.microedition.lcdui.Graphics;

import javax.microedition.lcdui.Image;





public class Visual {

    

    public static final int EASE_NONE = 0;

    public static final int EASE_IN = 1;

    public static final int EASE_OUT = 2;

    public static final int EASE_IN_OUT = 3;



    // Layer info

    

    private final Layer layer;

    private boolean wasVisible;

    private boolean dirty;

    

    // Motion

    

    private int startX;

    private int startY;

    private int endX;

    private int endY;

    

    /** The start time of the motion, in milliseconds, relative to currTime. */

    private int startTime = 0;

    

    /** The end time of the motion, in milliseconds, relative to currTime. */

    private int endTime = 0;

    

    /** The current time of the motion, in milliseconds. */

    private int currTime = 0;

    

    /** The easing applied to the motion. Valid values are

        EASE_NONE, EASE_IN, EASE_OUT, and EASE_IN_OUT. */

    private int easing = EASE_NONE;

    

    private int anchor = Graphics.TOP | Graphics.LEFT;

    

    public Visual(String imageResource, int framesAcross, int framesDown, int x, int y) {

        this(App.createImage(imageResource), framesAcross, framesDown, x, y);

    }

    

    

    public Visual(String imageResource, int x, int y) {

        this(App.createImage(imageResource), x, y, true);

    }

        

    

    public Visual(Image image, int framesAcross, int framesDown, int x, int y) {

        int w = image.getWidth() / framesAcross;

        int h = image.getHeight() / framesDown;

        layer = new Sprite(image, w, h);

        layer.setPosition(x, y);

        

        init();

    }

    

    

    public Visual(Image image, int x, int y) {

        this(image, x, y, true);

    }

    

    

    public Visual(Image image, int x, int y, boolean visible) {

        if (image == null) {

            image = App.getBrokenImage();

        }

        layer = new Sprite(image);

        layer.setPosition(x, y);

        layer.setVisible(visible);

        

        init();

    }

    

    

    public Visual(Layer layer) {

        if (layer == null) {

            layer = new Sprite(App.getBrokenImage());

        }

        this.layer = layer;

        init();

    }

    

    

    private void init() {

        startX = layer.getX();

        startY = layer.getY();

        endX = layer.getX();

        endY = layer.getY();

    }

    

    

    /**

        Returns the x value of this Visual. The returned value does not include the translation 

        based on the anchor.

    */

    public int getX() {

        return layer.getX();

    }

    

    

    /**

        Returns the y value of this Visual. The returned value does not include the translation 

        based on the anchor.

    */

    public int getY() {

        return layer.getY();

    }

    

    

    public int getScreenX() {

        return layer.getX() + getAnchorX();

    }

    

    

    public int getScreenY() {

        return layer.getY() + getAnchorY();

    }

    

    

    /**

        Returns the anchor x offset. 

    */

    public int getAnchorX() {

        if ((anchor & Graphics.RIGHT) != 0) {

            return -layer.getWidth();

        }

        else if ((anchor & Graphics.HCENTER) != 0) {

            return -layer.getWidth() / 2;

        }

        else { 

            return 0;

        }

    }

    

    

    /**

        Returns the anchor y offset. 

    */

    public int getAnchorY() {

        if ((anchor & Graphics.BOTTOM) != 0) {

            return -layer.getHeight();

        }

        else if ((anchor & Graphics.VCENTER) != 0) {

            return -layer.getHeight() / 2;

        }

        else { 

            return 0;

        }

    }

    

    

    public void setAnchor(int anchor) {

        if (this.anchor != anchor) {

            this.anchor = anchor;

            dirty = true;

        }

    }

    

    

    public void setX(int x) {

        moveTo(x, getY(), 0, 0);

    }

    

    

    public void setY(int y) {

        moveTo(getX(), y, 0, 0);

    }

    

    

    public void setLocation(int x, int y) {

        moveTo(x, y, 0, 0);

    }

    

    

    public int getWidth() {

        return layer.getWidth();

    }

    

    

    public int getHeight() {

        return layer.getHeight();

    }

    

    

    public void setVisible(boolean visible) {

        if (layer.isVisible() != visible) {

            layer.setVisible(visible);

            dirty = true;

        }

    }

    

    

    public int getFrame() {

        if (layer instanceof Sprite) {

            return ((Sprite)layer).getFrame();

        }

        else {

            return 0;

        }

    }

    

    

    public void setFrame(int index) {

        if (layer instanceof Sprite) {

            Sprite s = (Sprite)layer;

            if (s.getFrame() != index) {

                s.setFrame(index);

                dirty = true;

            }

        }

    }

    

    

    /**

        Returns true if the layer is dirty.

    */

    public boolean update(int elapsedTime) {

        currTime += elapsedTime;

        

        int newX = tweenValue(startX, endX);

        int newY = tweenValue(startY, endY);

        if (newX != getX() || newY != getY()) {

            layer.setPosition(newX, newY);

            return true;

        }

        else {

            return dirty;

        }

    }

    

    

    public void draw(Graphics g) {

        if (layer.isVisible()) {

            layer.move(getAnchorX(), getAnchorY());

            layer.paint(g);

            layer.move(-getAnchorX(), -getAnchorY());

        }

        dirty = false;

    }

    

    

    private int tweenValue(int startValue, int endValue) {

        

        if (currTime >= endTime) {

            return endValue;

        }

        else if (currTime <= startTime) {

            return startValue;

        }

        else {

            // Fixed point value. The range 0..0x10000 maps to 0..1

            int p = (int)(((long)(currTime - startTime) << 16) / (endTime - startTime));

            

            switch (easing) {

                case EASE_NONE:

                    // Do nothing

                    break;

                    

                case EASE_IN:

                    p = (int)(((long)p * p) >> 16);

                    break;

                    

                case EASE_OUT:

                    p = 0x10000 - p;

                    p = (int)(((long)p * p) >> 16);

                    p = 0x10000 - p;

                    break;

                    

                case EASE_IN_OUT:

                    if (p < 0x8000) {

                        p = (p << 1);

                        p = (int)(((long)p * p) >> 16);

                        p = (p >> 1);

                    }

                    else {

                        p = (0x10000 - p) << 1;

                        p = (int)(((long)p * p) >> 16);

                        p = 0x10000 - (p >> 1);

                    }

                    break;

            }

            

            return startValue + (int)(((long)p * (endValue - startValue)) >> 16); 

        }

    }

    

    

    public void moveTo(int x, int y, int startTime, int endTime) {

        move(getX(), getY(), x, y, startTime, endTime, EASE_NONE);

    }

    

    

    public void moveTo(int x, int y, int startTime, int endTime, int easing) {

        move(getX(), getY(), x, y, startTime, endTime, easing);

    }

    

    

    public void move(int startX, int startY, int endX, int endY, int startTime, int endTime) {

        move(startX, startY, endX, endY, startTime, endTime, EASE_NONE);

    }

    

    

    public void move(int startX, int startY, int endX, int endY, int startTime, int endTime,

        int easing)

    {

        this.startX = startX;

        this.startY = startY;

        this.endX = endX;

        this.endY = endY;

        this.startTime = startTime;

        this.endTime = endTime;

        this.currTime = 0;

        this.easing = easing;

    }

}