package sims2;



/* Copyright Ideaworks3D Ltd. 2005 */

/*

 * Created on Mar 14, 2005

 */







import java.io.ByteArrayInputStream;

import java.io.DataInputStream;

import java.io.IOException;

//import java.util.Hashtable;





//import com.ideaworks3d.debug.Debug;



/**

 * Iso-Engine that obtains data from Sims2Mobile level file format.

 */

public class IsoMap extends Engine

{

	protected MapTileSet	m_TileSet;

	

	/**

	 * Used to rotate grid positions

	 * @param gridPos	Position to rotate - modified in place

	 * @param rotation	Rotation to apply (enums.Rotation enumeration)

	 */

	public static void RotatePosInPlace(int[] gridPos, int rotation)

	{

		int x, y;

		if ((rotation & 1) == 1)

		{

			// 90 degree rotation

			x = gridPos[1];

			y = -gridPos[0];

		}

		else

		{

			// 0 degree rotation

			x = gridPos[0];

			y = gridPos[1];

		}

		if ((rotation & 2)==2)

		{

			// Add another 180 degree rotation

			x = -x;

			y = -y;

		}

		gridPos[0] = x;

		gridPos[1] = y;

	}

	

	public IsoMap(int guid, MapTileSet tileset, int id)

	{

		//Debug.println("new IsoMap: " + guid, //Debug.CHAN_CORE);

		byte[] mapData = null;

		try

		{

			mapData = Utility.GetGlobalResource(guid);

		}

		catch (Exception ex)

		{

			//Debug.assertPrintln(false, "IsoMap(): " + ex.getMessage());

		}



		m_TileSet = tileset;

		

		// Copy vital info into Engine.

		floor = tileset.GetFloor();

		walls = tileset.GetWalls();

		

		m_Id = id;

//		ParseTileData(guid, tileData);

		

		try

		{

			ParseMapData(mapData);

		}

		catch (IOException e)

		{

			// TODO Auto-generated catch block

			//Debug.printStackTrace(e);

		}

		

		// Set cache to not cached

		m_CachedLocation = Integer.MIN_VALUE;

		m_CachedTileWallType = NO_TILE_F;

	}

	

	/** @return width of map */

	public int GetWidth()

	{

		return m_Width;

	}

	

	/** @return height of map */

	public int GetHeight()

	{

		return m_Height;

	}

	

	/**

	 * Return the floor type at this location.

	 * Overrides Engine method.

	 * 

	 * @param x	X position within map.

	 * @param y	Y position within map.

	 * @param h	Height within map.

	 * @return	Index into tile textures, or -1 for no tile.

	 */

	public int getFloor(int x, int y, int h)

	{

		int result;		// Floor tile index returned

		int subtile;	// Which subtile on current map tile we lie on

		int location;	// Location of type data byte

		int superTile;	// Super tile type we are on

		

		// If height > 0 then look for roof tile

		if (h > 0)

		{				

			int type;			// Type of wall

			byte[] thisWall;	// Data for this wall

			int height;			// Height of this section

			

			// Check for a screen edge overlay here

			if (h == OVERLAY_HEIGHT)

			{

				// Subtraction of OVERLAY_HEIGHT from x and y finds the floor

				// location over which the overlay lies.

				result = GetOverlayTile(x - OVERLAY_HEIGHT, y - OVERLAY_HEIGHT); 

			}

			else

			{

				result = NO_TILE_F;

			}

			

			// If no screen edge overlay here

			if (result == NO_TILE_F)

			{			

				// Determine which type wall tile lies at this position

				type = GetCachedWallTypeAt(x, y);

				

				// Only proceed if there is a wall here

				if (type != NO_TILE_F)

				{		

					// Get height of wall

					//					height = thisWall[0] & Utility.UNSIGNED_BYTE_MASK;

					height = m_TileSet.GetWallHeight(type); // no need for and since will never be > 127

					

					// If the height if equal to the current

					if (h == (height - 1))

					{

						// Get pointer to wall tile data

						thisWall = m_TileSet.GetWallSectionDescription(type);

											

						//						result = thisWall[(height * 2)] & Utility.UNSIGNED_BYTE_MASK;

						// Get roof tile index

						

						// this byte is 2*height shorts into data (= 4 * height)

						result = thisWall[height << 2] & Utility.UNSIGNED_BYTE_MASK;

						

						// Map 0 tile to -1 (nothing here)

						// First tile contains background blacking tile

						if (result == 0)

						{

							result = -1;

						}

					}

					else	// Else height not equal to current

					{

						// No roof tile here

						result = NO_TILE_F;

					}

				}

				else	// Else if there isn't a wall here

				{

					result = NO_TILE_F;

				}

			} // end If no screen edge overlay here

		}

		else if (h == 0)	// Floor tile (h==0)

		{

			// Only continue if X and Y lie within map

			// (Compare to double width and height, since this is the granularity the engine works at)

			if (((x | y) >= 0) && (x < m_DoubleWidth) && (y < m_DoubleHeight))

			{		

				

				//					// Shift X and Y to make them refer to same granularity as map

				//					// (coordinates passed on are on double the granularity)

				//					x >>= 1;

				//					y >>= 1;

				//						

				//					// Obtain location in data

				//					// This tile data at ((x + y * m_Width) * BYTES_PER_MAP_TILE)

				//					location = (x + y * m_Width) * BYTES_PER_MAP_TILE;

				

				// Replace commented section with optimised code				

				location = ( (x & ALL_BUT_LAST_BIT_MASK) + 

								(y & ALL_BUT_LAST_BIT_MASK) * m_Width

							) << 1;

				

				// Obtain super tile type

				// (reduce this by one since we can't encode -1 into binary)

				superTile = m_MapData[location] & Utility.UNSIGNED_BYTE_MASK;

				superTile--;

				

				// If super tile isn't empty

				if (superTile != NO_TILE_F)

				{

					// Get sub tile type

					//						result = m_SuperTileDescriptions[superTile * TILES_IN_SUPER_TILE + subtile]

					//										& Utility.UNSIGNED_BYTE_MASK;

					

					// Determine which subtile we lie on within map

					subtile = (x & 0x1) + ((y & 0x1) << 1);

					

					result = m_TileSet.GetSuperTileDescription((superTile << 2) + subtile)

													 & Utility.UNSIGNED_BYTE_MASK;

				}

				else

				{

					// Super tile empty

					result = NO_FLOOR_TILE_F;

				}

			}

			else	// X or Y are outside map

			{

				result = NO_FLOOR_TILE_F;

			}	

		} 

		else	// Height out of range

		{

			result = NO_TILE_F;

		}

		

		

		return result;

	}

	

	/**

	 * Determine if grid location is on map and devoid of scenery

	 * @param gridX	Position on grid (not half grid)

	 * @param gridY	Position on grid (not half grid)

	 * @return	True if empty, false if full/off map.

	 */

	public boolean IsGridLocationWalkable(int gridX, int gridY)

	{

		boolean empty;

		int dataPos;	// Offset into tile data of this tile info

		

		if ((gridX >= 0) && (gridY >= 0)

						&& (gridX < m_Width) && (gridY < m_Height))

		{

			dataPos = (gridY * m_Width + gridX) * BYTES_PER_MAP_TILE;

			

			// There must be floor here and no walls

			empty = m_MapData[dataPos++] != 0;

			empty = empty ? (m_MapData[dataPos++] == 0) : false;

			empty = empty ? (m_MapData[dataPos++] == 0) : false;

			empty = empty ? (m_MapData[dataPos++] == 0) : false;

		}

		else

		{

			empty = false;

		}

		

		return empty;

	}

	

	/**

	 * Return the wall type at this location.

	 * Overrides Engine method.

	 * 

	 * @param x		X position of host tile within map.

	 * @param y		Y position of host tile within map.

	 * @param h		Height within map.

	 * @param side	Side of host tile (0=left, 1=right)

	 * @return	Index into wall textures, or -1 for no wall.

	 */

	public int getWall(int x, int y, int h, int side)

	{

		int result;			// Tile type to draw here

		int type;			// Type of wall

		byte[] thisWall;	// Data for this wall

		int height;			// Height of this section

		int location;		// Location in byte data of index info

		

		if (h >= 0)

		{

			// Determine which type wall tile lies at this position

			type = GetCachedWallTypeAt(x, y);

			

			// Only proceed if there is a wall here

			if (type != NO_TILE_F)

			{

				// Get height of wall

				height = m_TileSet.GetWallHeight(type) & Utility.UNSIGNED_BYTE_MASK;

				

				// If this is too high, then return no tile

				if (h >= height)

				{

					result = NO_TILE_F;

				}

				else	// There is a tile here (not too high)

				{

					// Get pointer to wall tile data

					thisWall = m_TileSet.GetWallSectionDescription(type);

					

					// Determine location of index info

					location = ((side != 0) ? height : 0) + h;

					

					// Measured in shorts, not bytes

					location <<= 1;

					

					// Return type of tile at this height

					// Descrease by 1 (since ubyte can't represent -1)

					result = ((thisWall[location] & Utility.UNSIGNED_BYTE_MASK) << 8) |

								(thisWall[location + 1] & Utility.UNSIGNED_BYTE_MASK);

					result--;

				}

			}

			else	// No wall section here

			{

				result = NO_TILE_F;

			}

		}

		else	// Height out of range

		{

			result = NO_TILE_F;

		}

		

		return result;

	}

	

	/**

	 * Return the transparency type of this wall texture.

	 * Return areas which contain some pixels.

	 * Return a combination of the enumeration defined in this class.

	 * 

	 * @param wall	Index into wall textures.

	 * @return	Transparency type of this wall texture, -1 for no texture.

	 */

	public int getAny(int wall)

	{

		if (wall < 0)

		{

			return 0;

		}

		else if (wall < m_TileSet.GetNumTopWallTiles())

		{

			return UPPER_WALL_SECTION_MASK;

		}

		else if (wall < m_TileSet.GetNumTopAndMiddleWallTiles())

		{

			return MIDDLE_WALL_SECTION_MASK;

		}

		else

		{

			return LOWER_WALL_SECTION_MASK;

		}

	}

	

	/**

	 * Return the transparency type of this wall texture.

	 * Return areas which are fully opaquely covered.

	 * Return a combination of the enumeration defined in this class.

	 * 

	 * @param wall	Index into wall textures.

	 * @return	Transparency type of this wall texture, -1 for no texture.

	 */

	public int getAll(int wall)

	{

		// if this wall is transparent

		if( isTransparent(wall) )

			return TILE_COVERAGE_NONE;

		// else return the tile coverage

		else

			return getAny(wall);

	}

	

	private boolean isTransparent(int wall)

	{

		if(wall < 0)

		{

			// Missing wall tile

			return true;

		}

		else

		{

			// Check the transparency in the bit array //

			// Get the byte.

			int byteCount = wall >> 3;

			int bit = wall - (byteCount << 3);

			

			//Debug.debugAssert(bit>=0 && bit<8);

			

			// We can do this with a shift, which should be more efficient.

			return (m_TileSet.GetWallTileTransparency(byteCount) & (1 << bit)) != 0;

			

			// It may be better to use an array, I'm following Eben's example.

			/*

			switch(bit)

			{

			case 0:

				return (m_TileSet.GetWallTileTransparency(byteCount) & 0x01) != 0;

			case 1:

				return (m_TileSet.GetWallTileTransparency(byteCount) & 0x02) != 0;

			case 2:

				return (m_TileSet.GetWallTileTransparency(byteCount) & 0x04) != 0;

			case 3:

				return (m_TileSet.GetWallTileTransparency(byteCount) & 0x08) != 0;

			case 4:

				return (m_TileSet.GetWallTileTransparency(byteCount) & 0x10) != 0;

			case 5:

				return (m_TileSet.GetWallTileTransparency(byteCount) & 0x20) != 0;

			case 6:

				return (m_TileSet.GetWallTileTransparency(byteCount) & 0x40) != 0;

			case 7:

				return (m_TileSet.GetWallTileTransparency(byteCount) & 0x80) != 0;

			}

			*/

		}

	}



	/**

	 * Return the transparency type of this floor texture.

	 * Return areas which are fully opaquely covered.

	 * Return a combination of the enumeration defined in this class.

	 * 

	 * @param floor	Index into floor textures.

	 * @return	Transparency type of this floor texture, -1 for no texture.

	 */

	public int getAllFloor(int floor)

	{

		int opacity = 0;

		

		if (floor < m_TileSet.GetNumNonOverlayFloorTiles())

			opacity = TILE_BOTTOM_F | TILE_LOWER_MIDDLE_F | TILE_UPPER_MIDDLE_F | TILE_TOP_F;

		

		return opacity;

	}

	

	/**

	 * Parse input binary map data

	 * 

	 * @param mapData	Byte array containing map data

	 * @throws IOException

	 */

	private void ParseMapData(byte[] mapData) throws IOException

	{			

		int pos;		// Position in input data

		int dataSize;	// Number of bytes for element being serialised

		

		pos = 0;

		

		// Get map width and height

		m_Width = Utility.ByteSerializeType(0, mapData, pos, 2, false);

		m_DoubleWidth = m_Width * 2;

		pos += 2;

		m_Height = Utility.ByteSerializeType(0, mapData, pos, 2, false);

		m_DoubleHeight = m_Height * 2;

		pos += 2;

		

		// Read map data

		dataSize = m_Width * m_Height * BYTES_PER_MAP_TILE;

		//Debug.assertPrintln(dataSize < 500000, "I don't beleive the size of your map data: " + dataSize);

		m_MapData = new byte[dataSize];

		System.arraycopy(mapData, pos, m_MapData, 0, dataSize);

		pos += dataSize;

		

		// Parse the sector data.

		// This is not very efficient.

		byte[] temp = new byte[mapData.length - pos];

		System.arraycopy(mapData, pos, temp, 0, temp.length);

		ParseSectorData(temp);

	}

	

	/**

	 * Parse input binary sector data

	 * 

	 * @param mapData	Byte array containing sector data

	 * @throws IOException

	 */

	protected /*final*/ void ParseSectorData(byte[] sectorData) throws IOException

	{

		ByteArrayInputStream inByte = new ByteArrayInputStream(sectorData);

		DataInputStream inData = new DataInputStream(inByte);

		

		// Verify map width and height

		short width = inData.readShort();

		short height = inData.readShort();

		

		//Debug.debugAssert(width == m_Width);

		//Debug.debugAssert(height == m_Height);

		

		// Get parameters.

		m_NumSectors = inData.readByte();

		m_NumPortals = inData.readByte();

		m_MaxLinksPerPortal = inData.readByte();

		m_MaxPortalsPerSector = inData.readByte();

		

		// Calculate remaining params.

		m_NumSectorsAdjoiningPortal = 

			m_MaxLinksPerPortal / (m_MaxPortalsPerSector - 1);

		

		// The number of bits for locations.

		int xValue = width;

		int xBits;

		for(xBits = 16; xBits > 0; --xBits)

		{

			if((xValue & 0x8000) != 0) break;

			

			// Move on to the next bit.

			xValue <<= 1;

		}

		

		int yValue = height;

		int yBits;

		for(yBits = 16; yBits > 0; --yBits)

		{

			if((yValue & 0x8000) != 0) break;

			

			// Move on to the next bit.

			yValue <<= 1;

		}

		

		// This shouldn't happen.

		//Debug.debugAssert(xBits + yBits <= 16);

		

//		m_NumBitsX = xBits;

		m_NumBitsY = yBits;

		

		m_BitmaskY = (1 << yBits) - 1;

		m_BitmaskX = (1 << (xBits + yBits)) - 1;

		m_BitmaskX -= m_BitmaskY;

		

		// For now adjoining portals must be two.

		//Debug.debugAssert(m_NumSectorsAdjoiningPortal == 2);

		

		// Get the sector data.

		m_SectorData = new byte[height][width];

		for(int y = 0; y < height; ++y)

			inData.read(m_SectorData[y]);

		

		// Get the portal data.

		m_SectorPortals = new byte[m_NumSectors][m_MaxPortalsPerSector];

		for(int i = 0; i < m_NumSectors; ++i)

		{

			inData.read(m_SectorPortals[i]);

			

			byte[] input = m_SectorPortals[i];

			

			int size = 0;

			for (int j = 0; j < input.length; j++)

			{

				if (input[j] != -1)

					size++;

			}

			

			byte[] build = new byte[size];

			size = 0;

			for (int j = 0; j < input.length; j++)

			{

				if (input[j] != -1)

					build[size++] = input[j];

			}

			

			m_SectorPortals[i] = build;

		}

		

		// Get the links data.

		m_PortalLinks = 

			new byte[m_NumPortals][m_MaxLinksPerPortal];

		for(int i = 0; i < m_NumPortals; ++i)

		{

			inData.read(m_PortalLinks[i]);

//			if (Debug.ENABLED)

//			{

//				for(int j = 0; j < m_PortalLinks[i].length; ++j)

//				{

//					//Debug.println("portal " + i + "->" + m_PortalLinks[i][j], //Debug.CHAN_PATHFINDING);

//				}

//			}

			

			// Organise the data.

			// Count the number of links.

			int count = 0;

			for(int j = 0; j < m_MaxLinksPerPortal; ++j)

				if(m_PortalLinks[i][j] != -1) count++;

			

			byte[] array = new byte[count];

			count = 0;

			for(int j = 0; j < m_MaxLinksPerPortal; ++j)

				if(m_PortalLinks[i][j] != -1) array[count++] = m_PortalLinks[i][j];

				

			// Copy back.

			m_PortalLinks[i] = array;

		}

		

		// Get the locations data.

		m_PortalLocations = new short[m_NumPortals];

		for(int i = 0; i < m_NumPortals; ++i)

			m_PortalLocations[i] = inData.readShort();

		

		// Create the hashtable for broken portals.

		// Ensure that all booleans are false.

		m_IsPortalBroken = new boolean[m_NumPortals];

	}

	

	private int GetCachedWallTypeAt(int x, int y)

	{

		int newLoc = (x << Utility.FIXED_POS) | y;

		int subtile;	// Which subtile on current map tile we lie on

		int location;	// Location of type data byte

		

		// If requested tile not cached

		if (m_CachedLocation != newLoc)

		{

			// Store cached ID

			m_CachedLocation = newLoc;	

			

			int xOrY = (x | y);

			

			// Only continue if X and Y lie within map

			// (Compare to double width and height, since this is the granularity the engine works at)

			// Last check is that subtile is not zero -- never a wall here

			if ((xOrY >= 0) && (x < m_DoubleWidth) && (y < m_DoubleHeight)

			&& ((xOrY & 0x1) != 0))

			{

				// Determine which subtile we lie on within map

				subtile = (x & 0x1) + ((y & 0x1) << 1);



				// Shift X and Y to make them refer to same granularity as map

				// (coordinates passed on are on double the granularity)

				x >>= 1;

				y >>= 1;

				

				// Obtain location in data

				// This tile data at ((x + y * m_Width) * BYTES_PER_MAP_TILE)

				// Then add offset = subtile

				//						location = (x + y * m_Width) * BYTES_PER_MAP_TILE;

				location = (x + y * m_Width) << 2;

				location += subtile;

				

				// Obtain type

				m_CachedTileWallType = m_MapData[location] & Utility.UNSIGNED_BYTE_MASK;;

				

				// Subtract by 1, to map 0 to -1 and so forth

				// (can't store -1 in a byte!)

				m_CachedTileWallType--;

			}

			else	// Does not lie within map

			{

				m_CachedTileWallType = NO_TILE_F;

			}

		}

		

		

		

		return m_CachedTileWallType;

	}

	

	/**

	 * Determines if an overlay tile is here and if so, which type

	 * 

	 * @param	x	X position on map to check

	 * @param	y	Y position on map to check

	 * @return		Tile type if there is an overlay here (else NO_TILE_F)

	 */

	private int GetOverlayTile(int x, int y)

	{

		int result;

		

		// Determine grid x and y

		// (x and y passed in are on half grid basis)

		int gridX = x >> 1;

		int gridY = y >> 1;

	

		// If tile within world

		// and the location lies on the edge of the world

		if (((x | y) >= 0) && (x < m_DoubleWidth) && (y < m_DoubleHeight) &&

				( (gridX == 0) || (gridY == 0) || 

					(gridX == (GetWidth() - 1)) || (gridY == (GetHeight() - 1))) )

		{

			// Determine type of overlay super tile we are on.

			// This is determined by the fade direction.

			int pos = 0;

			if (gridX == 0)

			{

				// Add nothing

			}

			else if (gridX == (GetWidth() - 1))

			{

				pos += 2;

			}

			else

			{

				pos += 1;

			}

			if (gridY == 0)

			{

				// Add nothing

			}

			else if (gridY == (GetHeight() - 1))

			{

				pos += 6;

			}

			else

			{

				pos += 3;

			}

			

			// Subtract one from the overlay super tile if it is >= 4.

			// This is because we don't store a tile for being dead centre

			// in the map (see previous calculations).

			if (pos >= 4)

			{

				pos--;

			}

			

			// Determine which subtile we lie on within map

			int subtile = (x & 0x1) + ((y & 0x1) << 1);

			

			// Determine result location in file

			result = m_TileSet.GetNumNonOverlayFloorTiles() + (pos * MapTileSet.TILES_IN_SUPER_TILE) + subtile;

		}

		else // If the location doesn't lie on the edge of the world

		{

			result = NO_TILE_F;

		}

		

		return result;

	}

	

	public byte[] getPortalLinks(byte id)

	{

//		// Count the number of links.

//		int count = 0;

//		for(int i = 0; i < m_MaxLinksPerPortal; ++i)

//			if(m_PortalLinks[id][i] != -1) count++;

//		

//		byte[] array = new byte[count];

//		count = 0;

//		for(int i = 0; i < m_MaxLinksPerPortal; ++i)

//			if(m_PortalLinks[id][i] != -1) array[count++] = m_PortalLinks[id][i];

//			

//		return array;

		

		return m_PortalLinks[id];

	}

	

	public int[] getPortalLocation(byte id)

	{

		int x;

		int y;

		

		short location = m_PortalLocations[id];

		y = (location & m_BitmaskY) >> 0;

		x = (location & m_BitmaskX) >> m_NumBitsY;

		

		return new int [] {x, y};

	}

	

	public byte getSector(int x, int y)

	{

		// This should never be called outside the map.

		//Debug.debugAssert(y >= 0);

		//Debug.debugAssert(x >= 0);

		//Debug.debugAssert(y < m_SectorData.length);

		//Debug.debugAssert(x < m_SectorData[y].length);

		

		return m_SectorData[y][x];

	}

	

	/**

	 * Check grid location x, y for a portal.

	 * 

	 * @param x		x grid coordinate

	 * @param y		y grid coordinate

	 * @return		Is the tile a portal

	 */

	public boolean isPortal(int x, int y)

	{

		byte sector = getSector(x, y);

		

		// Unwalkable.

		if(sector == -1) return false;

		// Check portal flag.

		return (sector & 0x80) != 0;

	}

	

	public byte[] getPortals(byte sectorId)

	{

//		// Return only non-(-1) values

//		

//		byte[] input = m_SectorPortals[sectorId];

//		

//		int size = 0;

//		for (int i = 0; i < input.length; i++)

//		{

//			if (input[i] != -1)

//				size++;

//		}

//		

//		byte[] build = new byte[size];

//		size = 0;

//		for (int i = 0; i < input.length; i++)

//		{

//			if (input[i] != -1)

//				build[size++] = input[i];

//		}

//		

//		return build;

		

		return m_SectorPortals[sectorId];

	}

	

//	public byte GetNearestSector(byte portalId)

//	{

//		byte result = -1;

//		Point2 portalLoc = getPortalLocation(portalId);

//		

//		// Search up, down, left right

//		int [][] offsets =

//		{

//			{ -1, 0 }, { 1, 0 }, { 0, -1 }, { 0, 1 }

//		};

//		for (int i = 0; i < offsets.length; ++i)

//		{

//			byte curSect = getSector(portalLoc.x + offsets[i][0], portalLoc.y + offsets[i][1]);

//			

//			if (curSect >= 0)

//			{

//				result = curSect;

//				break;

//			}

//		}

//		

//		return result;

//	}

	

	/**

	 * Split a portal by making a location unwalkable.

	 * 

	 * This ensures that the broken portal data is used,

	 * necessary portals are added to the Vector and 

	 * the coverage, links and location of each sub-portal

	 * is correct.

	 * 

	 * @param x		X location in the map

	 * @param y		Y location in the map

	 */

	public void SplitPortal(int x, int y)

	{

		// Check other worldliness.

		if(!IsSpaceInWorld(x, y))

			return;

		

		// Find the portal id here.

		byte sector = getSector(x, y);

		if((sector & 0x80) != 0)

		{

			int portalId = sector & ~0x80;

			

			// Ensure that the portal is broken.

			m_IsPortalBroken[portalId] = true;

		}

	}

	

	/**

	 * Merge broken portals by making a location walkable.

	 * 

	 * This ensures that broken portal data is up to date.

	 * If this unites the whole of the original portal then

	 * the portal will be marked as not broken. 

	 * 

	 * @param x		X location in the map

	 * @param y		Y location in the map

	 */

	/*public void MergePortals(int x, int y)

	{

		// Check other worldliness.

		if(!IsSpaceInWorld(x, y))

			return;

		

		// Find the portal id here.

		byte sector = getSector(x, y);

		if((sector & 0x80) != 0)

		{

			int portalId = sector & ~0x80;

		}

	}*/

	

	/** Get the id of this lot in the block view */

	public int GetLotId()

	{

		return m_Id;

	}

	

	/**

	 * Determine if a space is inside the world

	 * 

	 * @param x			x position of coordinate to check

	 * @param y			y position of coordinate to check

	 * @return			true if the coordinate is in the world

	 */

	public boolean IsSpaceInWorld(int x, int y)

	{

		boolean ret = !((x < 0) || (y < 0) || (x >= GetWidth()) || (y >= GetHeight()));

		return ret;

	}



	/** NO TILE flag used to return to overridden "get" methods */

	private static final int NO_TILE_F = -1;

	

	/** First floor tile used as "no tile" */

	private static final int NO_FLOOR_TILE_F = 0;

	

	/** Flag for zero wall tile coverage */

	private static final int TILE_COVERAGE_NONE = 0;

	/** Flag to refer to the bottom part of a tile */

	private static final int TILE_BOTTOM_F = 1;

	/** Flag to refer to the lower middle of a tile */

	private static final int TILE_LOWER_MIDDLE_F = 2;

	/** Flag to refer to the upper middle of a tile */

	private static final int TILE_UPPER_MIDDLE_F = 4;	

	/** Flag to refer to the top part of a tile */

	private static final int TILE_TOP_F = 8;

	

	/** Mask to refer to the bottom section of a wall */

	private static final int LOWER_WALL_SECTION_MASK = TILE_TOP_F | 

	TILE_LOWER_MIDDLE_F | TILE_UPPER_MIDDLE_F;

	

	/** Mask to refer to the middle section of a wall */

	private static final int MIDDLE_WALL_SECTION_MASK = TILE_BOTTOM_F | 

	TILE_LOWER_MIDDLE_F | TILE_UPPER_MIDDLE_F | TILE_TOP_F;

	

	/** Mask to refer to the upper section of a wall */

	private static final int UPPER_WALL_SECTION_MASK = TILE_BOTTOM_F; 

	

	/** Number of wall sections in a super tile */

	private static final int WALL_SECTIONS_IN_SUPER_TILE = 3;

	

	/** Number of bytes needed to describe one map tile */

	private static final int BYTES_PER_MAP_TILE = WALL_SECTIONS_IN_SUPER_TILE + 1;

	

	/** Mask used in calculating map positions */

	private static final int ALL_BUT_LAST_BIT_MASK = 0xFFFFFFFE;

	

	/** Height at which we place the overlay tiles on the edge of the world */

	private static final int OVERLAY_HEIGHT = 1;

	

	/** Width of map (in super tiles) */

	private int m_Width;

	

	/** Cached to speed up calculations */

	private int m_DoubleWidth;

	

	/** Height of map (in super tiles) */	

	private int m_Height;

	

	/** Cached to speed up calculations */

	private int m_DoubleHeight;

	

	/** Map tile data */

	private byte[] m_MapData;

	

	/** Number of sectors */

	private int m_NumSectors;

	

	/** Number of portals */

	private int m_NumPortals;

	

	/** Maximum number of links per portal */

	private int m_MaxLinksPerPortal;

	

	/** Maximum number of portals per sector */

	private int m_MaxPortalsPerSector;

	

	/** Number of sectors adjoining a portal */

	private int m_NumSectorsAdjoiningPortal;

	

	/** Sector tile data */

	private byte[][] m_SectorData;

	

	/** Sector array of portals */

	private byte[][] m_SectorPortals;

	

	/** Portal array of links */

	private byte[][] m_PortalLinks;

	

	/** Number of bits for x in locations */

//	private int m_NumBitsX;

	

	/** Number of bits for y in locations */

	private int m_NumBitsY;

	

	/** Bitmask for x in locations */

	private int m_BitmaskX;

	

	/** Bitmask for y in locations */

	private int m_BitmaskY;

	

	/** Portal array of locations */

	private short[] m_PortalLocations;

	

	/** Array of broken portal booleans */

	private boolean[] m_IsPortalBroken;

	

	/** Hashtable of broken portal data.

	 * 

	 * Hashtable element	{ Vector { Object[3] } };

	 * 

	 * Object[3] element	{ coverage, links, location };

	 * 

	 * coverage	{ short[] }

	 * links	{ byte[] }

	 * location	{ short }

	 */

//	private Hashtable m_BrokenPortalData;

	

	/** location of cached tile */

	private int m_CachedLocation;

	

	/** Wall type of cached tile */

	private int m_CachedTileWallType;

	

	/** The id number of this lot in the block view */

	protected int m_Id = -1;



}

