/*
 *     Copyright (c)2001-2002 DemiVision, LLC. All Rights Reserved.
 *
 * The information contained herein is the CONFIDENTIAL and PROPRIETARY
 *                  information of DemiVision, LLC.
 */

package com.bejeweled2_j2me;

import javax.microedition.lcdui.Canvas;
import javax.microedition.lcdui.Font;
import javax.microedition.lcdui.Graphics;
import javax.microedition.lcdui.Image;
import javax.microedition.midlet.MIDlet;
import java.util.Timer;
import java.util.TimerTask;
import java.io.*;


/**
 * The <tt>SplashScreen</tt> is responsible for showing title and credits
 * splash screens while the application is loading and initializing. The
 * SplashScreen also shows a progress bar, which indicates loading progress.
 * The SplashScreen is threaded so that the splash images can transition, and
 * the progress bar can update, while loading is in progress. The SplashScreen
 * is entirely self-contained, with no references to the rest of the library,
 * so that it does not cause the loading of other classes during construction.
 * Actual loading of the library takes place in the <tt>DioskilosMIDlet</tt>.
 * Note that the SplashScreen does <i>not</i> derive from <tt>BaseScreen</tt>
 * and is therefore not handled in the same way as other screens in the library.
 * <b>For internal use only</b>.
 *
 * @see         DioskilosMIDlet
 *
 * @author      Barry Sohl
 * @version     0.9.9
 */
final class SplashScreen extends Canvas implements Runnable {
    
    public static final String[] LANGUAGE_NAMES = {
        "English", "Fran\u00e7ais", "Italiano", "Deutsch", "Espa\u00f1ol"
    };
    
    public static final String[] LANGUAGE_CODES = {
        "en", "fr", "it", "de", "es"
    };
    
    public static final Font LANGUAGE_FONT = 
        Font.getFont( Font.FACE_SYSTEM, Font.STYLE_PLAIN,  DeviceSpecific.PLAIN_FONT_SIZE );

    // Splash states
    public static final int STATE_FLAG_SELECT = 0;
    public static final int STATE_LOGO        = 1;
    public static final int STATE_TITLE       = 2;

    // Progress bar frame rate
    private final int PROGRESS_RATE = 100;

    // Minimum splash display time
    final static int MIN_DURATION = 3000;

    // Shorthand
    private final int TOP_LEFT = (Graphics.TOP | Graphics.LEFT);

    /* Data Fields */
    int state;                 // Splash screen state
    private int splashDuration;
    private int languageIndex;

    private boolean dirty;              // Draw control
    private boolean progressDirty;

    // Updates progress bar
    private Timer progressTimer;
    private long progressStart;
    private boolean running;

    private int barTime;                // Progress bar settings
    private int barColor;
    private int[] barBounds;

    private Image offscreenImg;         // Maintains double buffer
    private Graphics offscreenGc;

    Image titleImage;                   // External resources
    Image logoImage;
    Image eaLogo;
    Image flagImage;
    String defaultLocale = "en";

    /**
     * Initializes the splash screen. Loads external property settings.
     *
     * @param   parent      reference to parent MIDlet.
     */
    void init( DioskilosMIDlet parent )
    {
        setFullScreenMode(true);
        
        // Read progress bar settings
        barTime = DeviceSpecific.BAR_TIME;
        barColor = 8127330;
        barBounds = DeviceSpecific.BAR_BOUNDS;
        
        // Minimum splash time, regardless of load time
        splashDuration = Math.max( barTime, MIN_DURATION ) / 2;

        // Allocate double buffer if necessary
        if ( !isDoubleBuffered() )
        {
            offscreenImg = Image.createImage( getWidth(), getHeight() );
            offscreenGc = offscreenImg.getGraphics();
        }
    }


    /**
     * Draws the first splash image and starts the independent thread that
     * will handle updating the progress bar.
     */
    void start()
    {
        // Must set before drawing or starting thread
        progressStart = System.currentTimeMillis();
        
        if (Build.SHOW_FLAG_SELECT) 
        {
            state = STATE_FLAG_SELECT;
            languageIndex = 0;
            for (int i = 0; i < LANGUAGE_CODES.length; i++) {
                if (LANGUAGE_CODES[i].equals(defaultLocale)) {
                    languageIndex = i;
                    break;
                }
            }
        }
        else 
        {
            state = STATE_LOGO;
        }

        // Draw first splash
        dirty = true;

        repaint();
        serviceRepaints();

        running = true;

        // Start progress bar
        progressTimer = new Timer();
        progressTimer.schedule(new RunnableTask(this), 0, PROGRESS_RATE);
    }

    /**
     * Stops the progress bar thread. The splash screen will consume no
     * further resources.
     */
    void stop()
    {
        running = false;
    }

    /**
     * Draws the current splash screen as well as the progress bar in its
     * current state to the offscreen buffer.
     *
     * @param   gc  graphics context used for drawing.
     */
    public void draw( Graphics gc )
    {
        switch(state)
        {
            case STATE_FLAG_SELECT:
                if (Build.SHOW_FLAG_SELECT && dirty) 
                {
                    int flagWidth = flagImage.getWidth() / 2;
                    int flagHeight = flagImage.getHeight() / LANGUAGE_CODES.length;
                    gc.setColor(0x000000);
                    gc.fillRect(0, 0, DeviceSpecific.WIDTH, DeviceSpecific.HEIGHT);
                    for (int i = 0; i < LANGUAGE_CODES.length; i++)
                    {
                        int x = DeviceSpecific.FLAG_X;
                        int y = DeviceSpecific.FLAG_Y + DeviceSpecific.FLAG_DY * i;
                        int xOffset = 0;
                        int yOffset = i * flagHeight;
                        if (languageIndex == i)
                        {
                            xOffset = flagWidth;
                            gc.setColor(0xB0D75E);
                        }
                        else
                        {
                            gc.setColor(0xffffff);
                        }
                        gc.setClip(x, y, flagWidth, flagHeight);
                        gc.drawImage(flagImage, x - xOffset, y - yOffset, TOP_LEFT);
                        gc.setClip(0, 0, DeviceSpecific.WIDTH, DeviceSpecific.HEIGHT);
                        gc.setFont(LANGUAGE_FONT);
                        gc.drawString(LANGUAGE_NAMES[i], DeviceSpecific.FLAG_NAME_X, 
                            y + DeviceSpecific.FLAG_NAME_Y_OFFSET, TOP_LEFT);
                        
                        dirty = false;
                    }
                }
                break;
                
            case STATE_LOGO:
                gc.drawImage(eaLogo, 0, 0, TOP_LEFT);
                break;
                
            case STATE_TITLE:
                // Draw current splash screen
                if(dirty) {
                    gc.drawImage( titleImage, 0, 0, TOP_LEFT );
                    gc.drawImage( logoImage, 0, DeviceSpecific.SPLASH_LOADING_Y, TOP_LEFT );
                    gc.setFont(GameEngine.FONT_PLAIN);
                    gc.setColor(0xee66dd);
                    gc.drawString(StringTable.LOADING, DeviceSpecific.WIDTH / 2,
                        DeviceSpecific.LOADING_Y, Graphics.TOP | Graphics.HCENTER);
                    dirty = false;
                }

                // Draw updated progress bar
                if(progressDirty) {
                    int barWd = (int) ((Math.min( (barBounds[2] *
                        (System.currentTimeMillis() - (progressStart +
                        splashDuration)) / splashDuration), barBounds[2] ))*5);

                    // Calculate % of total bar width
                    gc.setColor( barColor );
                    gc.fillRect( barBounds[0], barBounds[1],
                        barWd, barBounds[3] );
                    progressDirty = false;
                }

                break;
        }
    }

    /* Canvas */

    /**
     * Called when the screen (re)gains focus. Marks all object as dirty and
     * therefore requiring a redraw.
     */
    protected void showNotify()
    {
        dirty = true;
        progressDirty = true;
    }

    /**
     * Draws the current screen contents to the underlying drawing surface. The
     * surface is either the native graphics context or the offscreen buffer
     * depending on whether or not the device automatically double buffers.
     *
     * @param   gc  graphics context used for drawing.
     */
    public void paint( Graphics gc )
    {
        if ( isDoubleBuffered() )
        {
            draw( gc );
        }
        else
        {
            draw( offscreenGc );
            gc.drawImage( offscreenImg, 0, 0, TOP_LEFT );
        }

    }

    /* Runnable */

    /**
     * Controls threaded functionality of the splash screen. Transitions splash
     * images half way through the loading process and updates the progress bar
     * at fixed intervals.
     */
    public void run()
    {
        long now = System.currentTimeMillis();

        switch(state) {
            case STATE_LOGO:
                if((now - progressStart) > splashDuration) {
                    state = STATE_TITLE;
                    dirty = true;
                    progressDirty = true;
                }
                break;
            case STATE_TITLE:
                if((now - progressStart) < (splashDuration * 2)) {
                    progressDirty = true;
                }
                break;
        }
        repaint();
        serviceRepaints();

        if(!running) {
            progressTimer.cancel();
            progressTimer = null;

            titleImage = null;
            eaLogo = null;
            logoImage = null;
            flagImage = null;

            // Cleanup double buffer
            if(offscreenImg != null) {
                offscreenImg = null;
                offscreenGc = null;
            }
        }
        System.gc();
    }
    
    public void keyPressed(int keyCode) {
        
        if (Build.SHOW_FLAG_SELECT && state == STATE_FLAG_SELECT)
        {
            dirty = true;
            
            int gameCode;
            if (keyCode == Canvas.KEY_NUM5)
            {
                gameCode = Canvas.FIRE;
            }
            else 
            {
                gameCode = getGameAction(keyCode);
            }
            
            switch (gameCode) {
                case Canvas.FIRE:
                    StringTable.setLocale(LANGUAGE_CODES[languageIndex]);
                    state = STATE_LOGO;
                    progressStart = System.currentTimeMillis();
                    break;
                    
                case Canvas.UP:
                    if (languageIndex == 0) {
                        languageIndex = LANGUAGE_CODES.length - 1;
                    }
                    else {
                        languageIndex--;
                    }
                    break;
                    
                case Canvas.DOWN:
                    if (languageIndex == LANGUAGE_CODES.length - 1) {
                        languageIndex = 0;
                    }
                    else {
                        languageIndex++;
                    }
                    break;
            }
        }
    }

    /**
     * Wrap a runnable in a timer task.
     */
    private class RunnableTask extends TimerTask {

        // Task to run.
        Runnable task;

        // Initialize.
        RunnableTask(Runnable task) {
            this.task = task;
        }

        // Execute.
        public void run() {
            task.run();
        }

    }

}
