package sims2;



/* Copyright Ideaworks3D Ltd. 2005 */

/*

 * Created on Mar 7, 2005

 */







import javax.microedition.lcdui.Graphics;



//import 



//import GameDefines;

//import SimsApp;









/**

 * Describes an Animation in terms of images and sequence.

 * Implemented as static methods for simple refactoring to a utility class.

 */

public class Animation

{

	/**

	 * Determine which rotation defines this one, under symmetry laws

	 * @param facing		The facing we want to find the source face for

	 * @param symmetry		The symmetry currently applied to the animation

	 * @return	Face that is the source of the image strip

	 */

	public static int GetSourceRotation(int facing, int symmetry)

	{

		int sourceFacing;

		

		// If mirrored

		if (symmetry == Symmetry.MIRRORED)

		{		

			// Sides determine by either 0 or 90 degrees facing

			if ((facing == Rotation.DEGREES_0) || (facing == Rotation.DEGREES_270))

			{

				sourceFacing = Rotation.DEGREES_0;

			}

			else

			{

				sourceFacing = Rotation.DEGREES_90;

			}

		}

		// If 90 degrees symmetry or mirrored with 180 symmetry

		else if (symmetry == Symmetry.ROTATIONAL_HALF_PI || symmetry == Symmetry.MIRRORED_ROTATIONAL_PI)

		{

			// All sides determined by 0 degrees facing

			sourceFacing = Rotation.DEGREES_0;

		}

		// Else 180 degrees symmetry

		else if (symmetry == Symmetry.ROTATIONAL_PI) 

		{

			// Sides determine by either 0 or 90 degrees facing

			if ((facing == Rotation.DEGREES_0) || (facing == Rotation.DEGREES_180))

			{

				sourceFacing = Rotation.DEGREES_0;

			}

			else

			{

				sourceFacing = Rotation.DEGREES_90;

			}

		}

		else	// No symmetry		

		{		

			// All sides define themself

			sourceFacing = facing;

		}		

		

		return sourceFacing;

	}

	public static  boolean GetSourceMirroring(int facing, int symmetry)

	{

		if (symmetry == Symmetry.MIRRORED)

			return facing > Rotation.DEGREES_90;

		if (symmetry == Symmetry.MIRRORED_ROTATIONAL_PI)

			return (facing & 1) == 1;

		return false;

	}

	

	/**

	 * Construct animation from byte data

	 * @param allAnims	Hash table containing all animation data, referenced by strings

	 * @param index		Index of the required animation

	 */

	public Animation(/*int ownerGuid, */int instanceId,  byte[][] allAnims, int index, int symmetry, int paletteGuID)

	{

		byte[] animData;	// Data for this animation

		int dataPos;		// Position in input data

//		int imageSize;		// Size of image

		int numAnimFrames;	// Number of anim frames in current image		



//		m_OwnerGuid = ownerGuid;

		m_OwnerInstanceId = instanceId;

		

		// Store symmetry

		m_Symmetry = symmetry;

		

		// Store palette ID

		m_PaletteGuId = paletteGuID;

		

		// Store reference to animation table

		m_AllAnimsTable = allAnims;

		m_AnimId = index;

		

		// Grab animation data and set pointer to zero

		// (If we can't find the one we are looking for, take any one)

		animData = allAnims[index];

		dataPos = 0;	

		

		// Read image strips

		int numStrips = Utility.ByteSerializeType(0, animData, dataPos++, 1, false);

		m_ImageStrips = new ImageStrip[numStrips];

		int curImage = 0;

		while (numStrips > 0)

		{

			// Skip until we find an image we need

			// (Works on principle that we only load image strips that define

			// themself)

			while (curImage != GetSourceRotation(curImage, m_Symmetry))

			{

				curImage++;

				//Debug.debugAssert(curImage < Facing.NUM_DIRECTIONS);

			}

			

			// Read guid of image

			int imageGuid = Utility.ByteSerializeType(0, animData, dataPos, 4, false);

			dataPos += 4;

			

			// Load image

			ReadImageStrip(imageGuid, curImage);

			

			// Reduce number of images to load

			curImage++;

			numStrips--;

		}

		

		// Determine number of frames and assign arrays to hold these

		numAnimFrames = Utility.ByteSerializeType(0, animData, dataPos++, 1, false);

		m_FrameIndices = new int[numAnimFrames];

		m_FrameDelays = new int[numAnimFrames];

		

		// Read frame data

		for (int frameNum = 0; frameNum < numAnimFrames; frameNum++)

		{

			m_FrameIndices[frameNum] = Utility.ByteSerializeType(0, animData, dataPos, 2, false);

			dataPos += 2;

			m_FrameDelays[frameNum] = Utility.ByteSerializeType(0, animData, dataPos, 2, false);

			dataPos += 2;

		}

		

		// Set position in animation to first frame

		Reset();

		SetFacing(Facing.NE);

		

		

		// Special magic fudging of offsets for sitting animation.

		// Can't test this at the moment, since the sitting animations are all rotated 90degrees from where they should be??

		// TODO: we're currently loading a duplicate copy of the image strip for standing up/sitting down

		if (numAnimFrames==3 && index==0 || numAnimFrames==4 && index==1)

		{

			int i = m_ImageStrips[0].GetHotSpot(2)[0];

			if (i<8)

			{

//START:IGNORE:adj#

				final byte adj[] = {

					 5,2,	10,3,	15,5,

					 2,-1,	8,-6,	12,-8,

					 -2,-1,	-8,-6,	-12,-8,

					 -5,2,	-10,3,	-15,5,

				};

//END

				for (int frame=0; frame<3; frame++)

					for (int facing=0; facing<m_ImageStrips.length; facing++)

					{

						int[] pos = m_ImageStrips[facing].GetHotSpot(frame);

						pos[0] += adj[facing*6 + frame*2 + 0];

						pos[1] += adj[facing*6 + frame*2 + 1];

						m_ImageStrips[facing].SetHotSpot(frame,0,pos);

					}

			}

		}

	}

	

	public Animation(/*int ownerGuid, */int instanceId, ImageStrip strip, 

					 int frameDelay, boolean loops, int paletteGuId)

	{

		// Store no anims table or id

		m_AllAnimsTable = null;

		m_AnimId = -1;

		

		// Store owner guid and instance id

//		m_OwnerGuid = ownerGuid;

		m_OwnerInstanceId = instanceId;

		m_PaletteGuId = paletteGuId;

		

		// Assume symmetry is full rotational (90 degrees)

		m_Symmetry = Symmetry.ROTATIONAL_HALF_PI;	

		

		// Set up image strips		

//START:image strips#

		m_ImageStrips = new ImageStrip[] { strip };

//END

		

		// Put together static frame sequence data

		int numControlFrames = loops ? 1 : 0;

		int totalFrames = m_ImageStrips[0].GetNumFrames() + numControlFrames;

		m_FrameIndices = new int[totalFrames];

		m_FrameDelays = new int[totalFrames];

		for (int frameNum = 0; frameNum < totalFrames; frameNum++)

		{	

			// Do we need to write a loop frame here?

			if ((frameNum == (totalFrames - 1)) && loops)

			{

				// Jump to frame 0

				m_FrameIndices[frameNum] = COMMAND_JUMP | CONTROL_FRAME_F;

				m_FrameDelays[frameNum] = 0;		

			}

			else	// Normal sequence frame

			{

				m_FrameIndices[frameNum] = frameNum;

				m_FrameDelays[frameNum] = frameDelay;

			}

		}

		

		// Set position in animation to first frame

		Reset();

		SetFacing(Facing.NE);

	}

	

	/** 

	 * Set facing direction

	 * 

	 * @param facing	New facing direction.

	 */

	public void SetFacing(int facing)

	{	

		m_CurFacing = GetSourceRotation(facing, m_Symmetry);

		m_IsMirrored = GetSourceMirroring(facing, m_Symmetry);

		

		m_NeedsRedraw = true;

	}

	

	/** Call to start/unpause animation */

	public void Start()

	{

		m_IsPaused = false;

	}

	

	/** Call to stop/pause animation */

	public void Stop()

	{

		m_IsPaused = true;

	}

	

	/**

	 * Update animation

	 * @param millis	Milliseconds to update by

	 * @return	Currently running animation (so that one animation can link into the next)

	 */

	public Animation Update(int millis)

	{

		

		if (GetIsRunning() )

		{

			boolean finished = false;	// True when finished updating	

			

			// debugging for Player Sim animations.  

//			if (m_OwnerInstanceId == ScriptVM.s_playerInstanceID)

//			{

//				int i = 0;

//			}	

			

			m_CurFrameDelay -= millis;

			

			// Advance through all frames which are spent

			while (!finished && (m_CurFrameDelay <= 0))

			{

				// Pause on final frame

				if (!AdvanceFrame())

				{

					m_IsPaused = true;

				}

				else	// Advance was successful

				{

					// Handle any control frames which come after this one

					Animation nextAnim = HandleControlFrames();

					

					// If there is a new animation linked from this one,

					// then abandon this and update the new one

					if (nextAnim != this)

					{

	//					return nextAnim.Update(-m_CurFrameDelay);

						nextAnim.Reset();

						return nextAnim;

					}

				}

				

				// Increase time by delay of next frame

	//			m_CurFrameDelay += m_FrameDelays[m_CurFrameIndex];

				m_CurFrameDelay = m_FrameDelays[m_CurFrameIndex];

				

				// Exit animation update loop if not allowed more than one

				// update or if now paused.

				finished = !MULTIPLE_FRAME_ADVANCE || m_IsPaused;

			}

		}

		

		return this;

	}

	

	/**

	 * Render current frame, anchored at top left

	 * @param context	Graphics context to render to

	 * @param xPos		X position in context to draw to

	 * @param yPos		Y position in context to draw to

	 */

	public void Draw(Graphics context, int xPos, int yPos)

	{

		// Deprecated

//		Utility.ImageStripDraw(context, m_ImageStrips[m_CurFacing], 

//						m_HeightsPerImage[m_CurFacing],	m_FrameIndices[m_CurFrameIndex], 

//						xPos, yPos, m_IsMirrored);

	}

	

	/**

	 * Make billboard display the animatoin

	 * @param billImage	Billboard to display the animation

	 * @return		True if written, false if no need to draw

	 */

	public boolean DrawToBillBoard(Billboard billboard,  boolean forceRedraw)

	{				

		boolean doRedraw = m_NeedsRedraw || forceRedraw;

		

		if (doRedraw)

		{

			// Use ImageStrip method to redraw

			m_ImageStrips[m_CurFacing].DrawToBillBoard(

							billboard,

							m_FrameIndices[m_CurFrameIndex],

							m_IsMirrored);

			

			// Don't redraw until frame advances

			m_NeedsRedraw = false;

		}

		

		return doRedraw;

	}

	

	/**

	 * @return	Width of image frame

	 */

	public int GetWidth()

	{

		return m_ImageStrips[m_CurFacing].GetWidth(m_FrameIndices[m_CurFrameIndex]);

	}

	

	/**

	 * @return	Height of image frame

	 */

	public int GetHeight()

	{

		return m_ImageStrips[m_CurFacing].GetHeight(m_FrameIndices[m_CurFrameIndex]);

	}

	

	/** 

	 * Check if running

	 * 

	 * @return	True if running

	 */

	public boolean GetIsRunning()

	{

		return !m_IsPaused;

	}

	

	/**

	 * @return	Anim ID

	 */

	public int GetId()

	{

		return m_AnimId;

	}

	

	/**

	 * Helper to read an image strip

	 * @param guid		Guid of image strip to load

	 * @param facing	Which facing direction to load into

	 */

	private void ReadImageStrip(int guid, int facing)

	{

		boolean cacheImage = true;

		

		// only cache the player Sim animations if we are going into the game

//		if (SimsApp.s_MIDlet.m_CurrGameState < GameDefines.GAME_MODE_INITIALIZE_GAME)

//		{

//			cacheImage = false;

//		}

		

		m_ImageStrips[facing] = new ImageStrip(guid, m_PaletteGuId, cacheImage);

	}

	

	/** Restart animation */

	private void Reset()

	{

		m_CurFrameIndex = 0;

		m_CurFrameDelay = m_FrameDelays[m_CurFrameIndex];

		Start();

		m_NeedsRedraw = true;

	}

	

	public void ResetAnimation()

	{

		Reset();

		SetFacing(Facing.NE);

	}

	

	/**

	 * Move on by one frame

	 * 

	 * @return	False if at end of animation

	 */

	private boolean AdvanceFrame()

	{

		if (m_CurFrameIndex < (m_FrameIndices.length - 1))

		{			

			m_CurFrameIndex++;

			m_NeedsRedraw = true;

			return true;

		}

		

		return false;

	}

	

	/**

	 * Handles control frames until a non-control frame is encountered

	 * 

	 * @return	Currently running animation (so that one animation can link into the next)

	 */

	private Animation HandleControlFrames()

	{

		while ((m_FrameIndices[m_CurFrameIndex] & CONTROL_FRAME_F) != 0)

		{

			// Handle command

//START:HandleControlFrames#

			switch (m_FrameIndices[m_CurFrameIndex] & (~CONTROL_FRAME_F))

			{

				case COMMAND_JUMP:

					m_CurFrameIndex = m_FrameDelays[m_CurFrameIndex];

					m_NeedsRedraw = true;

					

					break;

					

				case COMMAND_PLAY:

					// Load this animation and stop processing

					Animation nextAnim = Utility.getAnimation(m_OwnerInstanceId, m_AllAnimsTable, m_FrameDelays[m_CurFrameIndex], m_Symmetry, m_PaletteGuId);

					nextAnim.m_CurFacing = m_CurFacing;

					nextAnim.m_IsMirrored = m_IsMirrored;

					return nextAnim;

				

				case COMMAND_STOP:	// fall through to default

				default:					

					m_IsPaused = true;

				

					// If we can't advance then loop (or else we get into infinite loop)

					if (!AdvanceFrame() && (m_CurFrameIndex != 0))

					{

						m_CurFrameIndex = 0;

						m_NeedsRedraw = true;

					}

					break;

			}

//END

		}

		

		return this;

	}

	

	/** Flag for being a control frame */

	private static final int CONTROL_FRAME_F = (1 << 15);

	

	/** ID for a control frame being a stop command */

	private static final int COMMAND_STOP = 0;

	/** ID for a control frame being a jump command */

	private static final int COMMAND_JUMP = 1;

	/** ID for a control frame being a play command */

	private static final int COMMAND_PLAY = 2;

	

	/** True if an animation can advance more than one frame per call to Update() */

	private static final boolean MULTIPLE_FRAME_ADVANCE = false;

	

	private final byte[][] m_AllAnimsTable;		/** Contains all object animations */

	public final int m_AnimId;					/** The ID of this animation */

	private final int m_PaletteGuId;			/** Palette used by this animation */

	

	private final ImageStrip[] m_ImageStrips;	/** Animation image strips (one per facing) */

	

	private final int[] m_FrameIndices;			/** Image to be displayed on each frame */

	private final int[] m_FrameDelays;			/** How long to display each frame */

	

	private int	m_CurFrameIndex;				/** Current frame of current animation being run */

	private int m_CurFrameDelay;				/** Delay left on current frame */

	private int m_CurFacing;					/** Facing direction */

	private boolean m_IsMirrored;				/** Is the display mirrored */

	private boolean m_IsPaused;					/** Is the animation paused? */

	

	private int m_Symmetry;						/** Store symmetry so that we can construct one animation from another */

	

	private boolean m_NeedsRedraw;				/** True if billboard needs redraw */



//	private int m_OwnerGuid;					/** Store owner guid */

	private int m_OwnerInstanceId;				/** Store owner instance id */

}

