/*
 *     Copyright (c)2001-2003 DemiVision, LLC. All Rights Reserved.
 *
 * The information contained herein is the CONFIDENTIAL and PROPRIETARY
 *                  information of DemiVision, LLC.
 */

package com.bejeweled2_j2me;

import javax.microedition.lcdui.*;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;

/**
 * The <tt>GameScreen</tt> encapsulates all support for actual application
 * gameplay. The default implementation provides basic functionality such as
 * starting and ending games, forfeiting a game, downloading player statistics,
 * as well as drawing a PreGame subscreen. Following is an example of the default
 * PreGame subscreen:
 *
 * <br><img src="doc-files/GameScreen-1.png" vspace="20"><br>
 *
 * The GameScreen exists in one of three primary modes:
 *
 * <ul>
 *   <li>Single Player</li>
 *   <li>Multiplayer</li>
 *   <li>Play-Wait</li>
 * </ul>
 *
 * Single player mode allows the user to play a game by themselves and requires
 * absolutely no network connectivity. Multiplayer mode is the standard head-to-head
 * challenge system. Play-wait mode allows a user to play single player while waiting
 * to be challenged in multiplayer mode.<p>
 *
 * In order to save code space, the presentation layer, business logic, and network
 * event handling are all handled in the same class. Because the GameScreen is a system
 * screen, it is automatically managed by the <tt>GameEngine</tt>. In order to provide
 * application specific gameplay, developers must extend GameScreen. There a number of
 * methods that will need to be extended in order to support custom functionality.
 * Following are the core methods:
 *
 * <ul>
 *   <li>{@link draw(Graphics)} for game blitting</li>
 *   <li>{@link heartbeat(long)} for game loop functionality</li>
 *   <li>{@link keyPressed(int)} for key event handling</li>
 * </ul>
 *
 * Additionally, there a number of read and write methods that should be extended in order
 * to send and receive custom payload data within the game network packets. These methods
 * are described in detail within this document. The extended GameScreen must be registered
 * with the GameEngine by overriding {@link DioskilosMIDlet#registerScreens()}.
 *
 * @see         GameEngine
 * @see         DioskilosMIDlet
 *
 * @author      Barry Sohl
 * @version     1.1.0
 */
class GameScreen extends BaseScreen implements CommandListener
{

    // Game node end type codes

    /** Node end state indicating player was still playing when the node ended. */
    public static final byte NODE_END_PLAYING = 0;

    /** Node end state indicating player successfully completed the current node. */
    public static final byte NODE_END_COMPLETE = 1;

    /** Node end state indicating player died during the current node and did not complete it. */
    public static final byte NODE_END_DIED = 2;

    // Screen states
    private final byte STATE_ERR        = 0;
    private final byte STATE_INIT       = 1;
    private final byte STATE_GAMEPLAY   = 4;
    private final byte STATE_OVER_WAIT3 = 7;
    private final byte STATE_OVER       = 8;
    private final byte STATE_REMATCH    = 9;

    // Pause menu items.
    private final static byte PAUSE_RESUME = 0;
    private final static byte PAUSE_OPTION = 1;
    private final static byte PAUSE_HELP   = 2;
    private final static byte PAUSE_RESTART= 3;
    private final static byte PAUSE_EXIT   = 4;

    /** Minimum value for custom object states, lower values are reserved for internal use. */
    public static final byte STATE_CUSTOM = 100;

    // Interface settings
    private final int WAIT_DUR        = 3000;
    private final int FORCED_REDRAW_DELAY = 500;

    /** Used for centering text around a column on the left-hand side of the screen. */
    protected int COLUMN_LEFT;

    /** Used for centering text around a column at the center of the screen. */
    protected int COLUMN_CENTER;

    /** Used for centering text around a column on the right-hand side of the screen. */
    protected int COLUMN_RIGHT;

    /** Indicates the timestamp time (milliseconds since node start) that a game node ended. */
    protected int endTime;

    // Game tree location

    /** Offset of current game node within current game tree level. */
    protected byte levelOffset;

    // Extra state handling
    private long baseTimestamp;

    private long forcedDrawTime;

    // Player 1 (local) stats

    /** 'OK' <tt>Command</tt> to be reused on subscreens, used internally for PreGame. */
    protected Command okCmd;

    /** 'Cancel' <tt>Command</tt> to be reused on subscreens, used internally for PreGame. */
    protected Command cancelCmd;

    protected Command helpCmd;
    protected Command exitCmd;
    protected Command pauseCmd;
    protected Command selectCmd;

    /** Pause menu dialog. */
    private static DialogCanvas pauseMenu;
    private static String[] pauseItems;

    /** Set when pause dialog should be displayed. */
    static boolean isPaused;


    /**
     * Determines whether the game is currently in the gameplay state, which
     * includes only actual gameplay and excludes any post-game handshaking.
     * Typically developers should only perform drawing and game functionality
     * when this method returns true.
     *
     * @return  true if in gameplay state, else false.
     */
    protected final boolean isGameplayState()
    {
        return (getState() == STATE_GAMEPLAY);
    }

    /**
     * Determines whether the game is currently in the post-game state, which
     * exists from the time gameplay is completed until the time that the node
     * end hanshaking has completed.
     *
     * @return  true if in post-game state, else false.
     */
    protected final boolean isPostGameState()
    {
        byte localState = getState();

        return ((localState > STATE_GAMEPLAY) && (localState < STATE_OVER));
    }

    /**
     * Determines whether the game is currently in the over state, which
     * indicates that all handshaking at the end of the game node has
     * completed. At this point, the developer can either start the next
     * node or display any scoring or final results subscreens.
     *
     * @return Is this game over?
     */
    protected final boolean isOverState()
    {
        byte localState = getState();

        return ((localState >= STATE_OVER) && (localState < STATE_REMATCH));
    }

    /**
     * Returns a timestamp suitable for use in all network packets requiring a
     * local timestamp. Time is measured as milliseconds since the start of the
     * current game node. Therefore, each player will have a timestamp relative
     * to the amount of time they have actually been playing the node, which
     * takes into account their latency. This allows the <i>Kraken</i> server to
     * order packets as well as resolve timing disputes between players. The
     * resulting value is returned as a 32 bit <tt>int</tt> after having been
     * truncated from 64 bits.
     *
     * @return  timestamp as milliseconds since start of current game node.
     */
    public final int getTimestamp()
    {
        return (int) (System.currentTimeMillis() - baseTimestamp);
    }

    /**
     * Called back from {@link endNode(byte)} at the end of each game node while in
     * single player mode so that application specific scoring information can be
     * calculated. Multiplayer scores should be calculated on the server. Developers
     * should extend this method to compute their own single player results. The
     * default implementation does nothing.
     *
     * @param    type    one of {@link GameScreen#NODE_END_COMPLETE} or {@link GameScreen#NODE_END_DIED}.
     */
    protected void computeResults( byte type )
    {}

    /**
     * Truncates the specified string into a shortened format. The returned
     * string will be truncated to a length that will fit inside <tt>pixelWd</tt>.
     * If <tt>str</tt> is already narrower than <tt>pixelWd</tt>, this method
     * has no effect.<p>
     *
     * <b>WARNING</b>: This method allocates memory equal to two times str.length()
     * and should not be called frequently.<p>
     *
     * @param   str         string to be truncated.
     * @param   pixelWd     maximum pixel width allowed before truncation occurs.
     *
     * @return  truncated string, or null if 'str' was null.
     */
    protected String truncate( String str, int pixelWd )
    {
        String truncatedStr = null;

        if ( str != null )
        {
            int len = str.length();
            char[] chars = str.toCharArray();

            // Find first character that causes string to be too long. Replace
            // that character with '.', otherwise string is returned unmodified.
            //
            for ( int i = 0; i < chars.length; i++ )
            {
                if ( gameEngine.FONT_PLAIN.charsWidth( chars, 0, (i + 1) ) > pixelWd )
                {
                    chars[i] = '.';
                    len = (i + 1);
                    break;
                }
            }

            truncatedStr = new String( chars, 0, len );
        }

        return truncatedStr;
    }

    /**
     * Handles unrecoverable error conditions during gameplay. An error dialog
     * will be displayed with the specified error message. After the user
     * acknowledges the error, the interface will be returned to the main menu.
     * In debug mode, the specified exception can be used to output additional
     * trace information.
     *
     * @param   msg     error message to be displayed.
     * @param   ex      exception causing error, null if no exception.
     */
    public void handleFatalError( String msg, Exception ex )
    {
        setState( STATE_ERR );
        dlg.showError( this, msg );
        
        // Additional debug info
        if (Build.DEBUG) {
            ex.printStackTrace();
            gameEngine.writeDebug( msg , ex );
        }
    }

    /**
     * Draws a screen title bar used for manually drawn subscreens of the
     * <tt>GameScreen</tt>. Internally this is used for the PreGame subscreen.
     * Developers are encouraged to use this method for scoring or other custom
     * subscreens in order to maintain consistency with the PreGame subscreen.
     * The title bar supports multiple lines of centered text. This method will
     * not alter the current <tt>Graphics</tt> font and color settings.<p>
     *
     * The drawn title bar will attempt to replicate the native LCDUI <tt>Screen</tt>
     * title bar as closely as possible. If the device has accessible look-and-feel
     * settings, they will be used.
     *
     * @param   gc          graphics context used for drawing.
     * @param   lines       array of strings, one string for each line of text.
     */
    protected void drawTitleBar( Graphics gc, String[] lines )
    {
        int numLines = lines.length;
        int ht0 = (GameEngine.HT_PLAIN * numLines) + ((numLines > 1) ? 2 : 1);

        gc.setColor( COLOR_TITLE_BAR );
        gc.fillRect( 0, 0, wd, ht0 );

        gc.setColor( COLOR_LINE );
        gc.drawLine( 0, ht0, (wd - 1), ht0 );


        // Draw title bar text
        gc.setColor(COLOR_TITLE_TEXT);
        for ( int i = 0; i < numLines; i++ )
        {
            CustomFont.drawString(GameEngine.FONT_BOLD, gc, lines[i],
                COLUMN_CENTER, (GameEngine.HT_PLAIN * i) + 1,
                GameEngine.CENTERED);
        }
    }

    /**
     * Provides standard interface support for starting the next game node. In
     * single player or play-wait mode, gameplay starts immediately. In multiplayer
     * mode, the start node handshaking process begins. Developers should call this
     * method from wherever they want to start the next game node.
     */
    public final void handleNextNode()
    {
        byte nextOffset = (byte) (levelOffset + 1);
        startNode( nextOffset );
    }

    /**
     * Initiates the process of ending the current game node when in multiplayer
     * mode. A node end request will be sent to the server. The server will then
     * respond with results and scoring information for the node. The developer
     * must indicate whether the node was ended by succesful completion or by
     * dying before finishing.
     *
     * @param   type    one of {@link GameScreen#NODE_END_COMPLETE} or
     *                         {@link GameScreen#NODE_END_DIED}.
     */
    public final void handleNodeEnd( byte type )
    {
        endNode( type, false );
    }

    /**
     * Standard interface for displaying pause menu.
     */
    public final void handlePause()
    {
        // Game paused while dialog up
        gameEngine.pause();
        isPaused = true;
        pauseMenu.setSelected(0);
        gameEngine.getMIDlet().getDisplay().setCurrent(pauseMenu);
    }
    
    /**
     * Provides standard interface support for exitting the game. A confirmation dialog
     * will be displayed for the user. The user's input will automatically be handled
     * depending on the current game mode. Developers should call this method if they
     * are providing a custom way to exit the game beyond the default behavior. If the
     * <tt>GameScreen</tt> is currently in mode {@link GameObject#MODE_CUSTOM}, this
     * method will return the user to either the main menu or to the lobby depending on
     * from where they entered custom mode.
     */
    public final void handleExit()
    {
        // Game paused while dialog up
        gameEngine.pause();
        dlg.show(this, Dialog.DLG_YESNO, StringTable.EXIT_TITLE,
            StringTable.EXIT_MESSAGE);
    }

    /**
     * Standard interface for in game help.
     */
    public final void handleHelp()
    {
        // Game paused while dialog up
        gameEngine.pause();
        dlg.showHelp(this);
    }

    /**
     * Provides standard interface support for finishing a game. In single player
     * mode, the user will be returned to the main menu. In play-wait mode, a new
     * game will be started. In multiplayer mode, the user will be given the option
     * to rematch with their previous opponent. Developers should call this method
     * from wherever they are finishing the game within their custom <tt>GameScreen</tt>.
     */
    public final void handleGameOver()
    {
        isPaused = false;
        MainScreen.showTrialOver = true;
        gameEngine.transition( GameEngine.SCREEN_MAIN );
    }

    /**
     * Forces a redraw after a short delay. Used for platforms that have problems
     * redrawing immediately after a new display has been set.
     */
    private void forceRedraw()
    {
        forcedDrawTime = System.currentTimeMillis();
    }
    
    /* BaseScreen */

    /**
     * The <tt>GameEngine</tt> itself is always the current displayable
     * during gameplay.
     *
     * @return  reference to displayable to be given focus.
     */
    public Displayable getDisplayable()
    {
        if(isPaused) {
            return(pauseMenu);
        }
        return gameEngine;
    }

    /**
     * Loads resources and prepares the screen for display. Developers should
     * extend this method to load and/or allocate all resources for their
     * custom game functionality. <i>Do not override this method</i>.
     *
     * @throws  IOException if error occurs loading resources.
     */
    public void init() throws IOException
    {
        // Setup for text drawing
        COLUMN_LEFT   = (wd / 5);
        COLUMN_CENTER = (wd / 2);
        COLUMN_RIGHT  = (COLUMN_LEFT * 4);

        // Setup interface
        okCmd = new Command(StringTable.SPACE, Command.OK, 0);
        cancelCmd = new Command(StringTable.CANCEL_SOFTKEY, Command.CANCEL, 1 );
        pauseCmd = new Command(StringTable.PAUSE_SOFTKEY, Command.SCREEN, 1 );
        selectCmd = new Command(StringTable.SELECT_SOFTKEY, Command.OK, 0 );

        pauseMenu = new DialogCanvas(DialogCanvas.LIST, StringTable.PAUSE_TITLE);
        
        if (Build.VODAFONE) {
            pauseItems = new String[] {
                StringTable.PAUSE_RESUME,
                StringTable.PAUSE_OPTION,
                StringTable.PAUSE_HELP,
                StringTable.VODAFONE_RESTART,
                StringTable.EXIT_TITLE,
            };
        }
        else {
            pauseItems = new String[] {
                StringTable.PAUSE_RESUME,
                StringTable.PAUSE_OPTION,
                StringTable.PAUSE_HELP,
                StringTable.EXIT_TITLE,
            };
        }
        pauseMenu.setList(pauseItems, null);
        //pauseMenu.addCommand(selectCmd);
        pauseMenu.setCommandListener(this);

        helpCmd = new Command(StringTable.HELP_SOFTKEY, Command.OK, 1);
        exitCmd = new Command(StringTable.EXIT_SOFTKEY, Command.EXIT, 1);
        gameEngine.setCommandListener( this );
    }

    /**
     * Starts a new game the first time the screen is shown. Subsequent calls
     * will not restart the game unless <tt>init</tt> is true. Multiplayer
     * games will go to the PreGame subscreen, single player games will go
     * directly into gameplay. Developers can extend this method to provide
     * custom startup functionality for the game. <i>Do not override this
     * method</i>.
     *
     * @param   init    initial showing following screen transition?
     * @param   now     current system time
     */
    public void start( boolean init, long now)
    {
        // Start a new game
        if ( init )
        {
            // Set initial state
            gameEngine.soundOn =
                gameEngine.getSettings().isOn(SettingsScreen.FLAG_SOUND);
            gameEngine.vibrateOn =
                gameEngine.getSettings().isOn(SettingsScreen.FLAG_VIBRATE);
            gameEngine.funlightsOn =
                gameEngine.getSettings().isOn(SettingsScreen.FLAG_FUNLIGHTS);
            setState( STATE_INIT );

            gameEngine.removeCommand( pauseCmd );
            isPaused = false;

            // Multiplayer goes first to pregame screen
            startNode( (byte)0 );
        }
        
        forceRedraw();

        // Make sure everything redraws
        super.start(init, now);
    }

    /**
     * Pauses game execution, typically following loss of focus by the application.
     * Execution will be resumed with a call to {@link start(boolean)} with an
     * <tt>init</tt> value of false. Developers can extend this method to provide
     * custom functionality such as pausing internal timers and the like. <i>Do not
     * override this method</i>.
     *
     * @param now current system time
     */
    public void pause(long now)
    {
        super.pause(now);
    }

    /**
     * Cleans up all resources in use by the <tt>GameScreen</tt>. Developers should
     * extend this method to perform any application specific cleanup. <i>Do not
     * override this method</i>.
     */
    public void destroy()
    {}

    /**
     * Called when the next game node at the current level of the game tree
     * is about to start. Developers can extend this method to provide any
     * custom functionality needed at the start of each game node. <i>Do not
     * override this method</i>.
     *
     * @param   offset      level offset within game tree at current level.
     */
    public void startNode( byte offset )
    {
        setState( STATE_GAMEPLAY );

        // Multiplayer names set elsewhere, single player defaults to 'You'

        // Reset timestamps

        // Advance node
        levelOffset = offset;

        // Update buttons
        gameEngine.removeCommand( okCmd );
        gameEngine.removeCommand( cancelCmd );

        //gameEngine.addCommand( pauseCmd );
        
        // Mark this player's start time
        baseTimestamp = System.currentTimeMillis();

        start( false, baseTimestamp );
    }

    /**
     * Called when the game node currently in progress is about to end. A node
     * end may have occurred because a player succesfully completed the node
     * or because the player died during the node. Either player can cause a
     * node end. Developers can extend this method to provide any custom
     * functionality needed at the end of each game node. <i>Do not override
     * this method</i>.
     *
     * @param   type    one of {@link GameScreen#NODE_END_COMPLETE} or
     *                         {@link GameScreen#NODE_END_DIED}.
     * @param   remote  node end caused by opponent?
     */
    public void endNode( byte type, boolean remote )
    {
        baseTimestamp = System.currentTimeMillis();

        // Must set state -after- timestamp
        setState( STATE_OVER_WAIT3 );

        // Developer must manually handle exit after end of node
        gameEngine.removeCommand(pauseCmd);
        
        computeResults( type );
    }

    /**
     * Draws the <tt>GameScreen</tt> in its current state. The default
     * implementation draws only the PreGame subscreen. Developers should
     * extend this method to draw all of their own game functionality,
     * scoring screens, etc. This method will be called once per game loop.
     * Drawing will occur either to an offscreen buffer or directly to the
     * underlying <tt>Canvas</tt> depending on whether or not the device
     * automatically double buffers. The <tt>Graphics</tt> reference will
     * automatically point to the appropriate surface. Developers should
     * remember to set {@link GameObject#dirty} to <i>false</i> after
     * drawing. In the extended method, developers should call the {@link
     * GameObject#draw(Graphics)} method for all child objects of their custom
     * <tt>GameScreen</tt> in order to draw their game object tree. <i>Do not
     * override this method</i>.
     *
     * @param   gc  graphics context used for drawing.
     */
    public void draw( Graphics gc )
    {
        // Do NOT set dirty to false
        if ( dirty )
        {
            int drawHt = getDisplayHeight();

            // Critical workaround for removeCommand bug
            gc.setClip( 0, 0, wd, drawHt );

            // Erase background
            gc.setColor( COLOR_BGND );
            gc.fillRect( 0, 0, wd, drawHt );
        }
    }

    /**
     * The main game loop heartbeat. Internally, the heartbeat handles
     * transitions between internal screen states. Developers should extend
     * this method to provide their own game loop functionality during gameplay.
     * Developers should only handle the heartbeat when {@link isGameplayState()}
     * is <i>true</i>. At all other times, the parent class will handle the
     * heartbeat. In the extended method, developers should call the {@link
     * GameObject#heartbeat(long)} method of <tt>GameObject</tt> for all child
     * objects of their custom <tt>GameScreen</tt> in order to pass on the
     * heartbeat to their game object tree. <i>Do not override this method</i>.
     *
     * @param now current time as milliseconds since unix epoch.
     */
    public void heartbeat( long now )
    {
        switch ( getState() )
        {
            // Once ack received, ready to move
            case STATE_OVER_WAIT3:
            {
                if ( getTimestamp() > WAIT_DUR )
                {
                    setState( STATE_OVER );
                    baseTimestamp = now;

                    gameEngine.restart( GameEngine.SCREEN_GAME );
                }

                break;
            }
        }
        
        // Some platforms have a problem drawing immediately after the
        // current display has been changed. This forces a redraw after
        // a small time delay.
        if ( (forcedDrawTime > 0) && ((now - forcedDrawTime) > FORCED_REDRAW_DELAY) )
        {
            markDirty();
            forcedDrawTime = 0;
        }
    }

    /**
     * Called back by <tt>Dialog</tt> to handle the dismissal of modal dialogs
     * on the <tt>GameScreen</tt>. All internally generated dialogs are
     * automatically handled. Developers should extend this method to handle
     * any dialogs that they generate. <i>Do not override this method</i>.
     *
     * @param result  resulting dialog command type code.
     */
    public void dialogDismissed( byte result )
    {
        switch ( result )
        {
            case Dialog.CMD_NO:
            {
                handlePause();
                //gameEngine.restart( GameEngine.SCREEN_GAME );
                break;
            }
            case Dialog.CMD_YES:
            {
                MainScreen.showTrialOver = false;
                gameEngine.transition( GameEngine.SCREEN_MAIN );
                break;
            }
            default:
            {
                break;
            }
        }
    }

    /**
     * Handles key events for the <tt>GameScreen</tt>. These are physical device
     * keys as opposed to the command soft keys. Internally, the primary key
     * events are for quitting or forfeiting the game. Developers should extend
     * this method to handle application specific key events for their gameplay.
     * Developers should only handle the key events when {@link isGameplayState()}
     * is <i>true</i>. At all other times, control should be left to the parent class.
     * If the developer's extended handling involves exitting the game, they should
     * call {@link handleExit()} to provide the standard exit behavior. <i>Do not
     * override this method</i>.
     *
     * @param   keyCode     key event being handled.
     */
    public void keyPressed( int keyCode )
    {
        // User asking for help.
        if(keyCode == DeviceSpecific.KEY_CODE_SOFT1) {
            //handleHelp();
            if (isPaused) {
                commandAction(List.SELECT_COMMAND, null);
            }
            else {
                handlePause();
            }
            return;
        }

        // User exiting game or forfeiting match
        if (keyCode == GameEngine.KEY_NUM0 /*|| keyCode == DeviceSpecific.KEY_CODE_SOFT2*/)
        {
            handleExit();
            return;
        }

        super.keyPressed( keyCode );
    }

    /**
     * Handles command soft key presses for the <tt>GameScreen</tt>. These are
     * device independent virtual keys as opposed to the physical device buttons.
     * Internally, key presses are primarily related to the PreGame subscreen.
     * Developers should extend this method to handle any custom commands that
     * they have added. These would most likley be for scoring or final results
     * screens. Developers should call {@link handleGameOver()} if their custom
     * command handles the final game over. This method will provide standard
     * behavior for returning the user to the lobby or main menu as appropriate
     * for the current game mode. <i>Do not override this method</i>.
     *
     * @param   cmd     command being handled.
     * @param   dis     displayable owning command.
     */
    public void commandAction( Command cmd, Displayable dis )
    {
        if ( cmd == pauseCmd )
        {
            handlePause();
        }
        if ( cmd == selectCmd || cmd == List.SELECT_COMMAND ) {
            int index = pauseMenu.getSelectedIndex();
            if (pauseItems != null && index >= 0 && index < pauseItems.length) {
                String item = pauseItems[index];
                if (item == StringTable.PAUSE_RESUME) {
                    isPaused = false;
                    gameEngine.restart(GameEngine.SCREEN_GAME);
                    gameEngine.getMIDlet().getDisplay().setCurrent(gameEngine);
                }
                else if (item == StringTable.PAUSE_OPTION) {
                    gameEngine.transition( GameEngine.SCREEN_SETTINGS);
                }
                else if (item == StringTable.PAUSE_HELP) {
                    handleHelp();
                }
                else if (item == StringTable.VODAFONE_RESTART) {
                    DialogListener listener = new DialogListener() {
                        public void dialogDismissed( byte result ) {
                            if (result == Dialog.CMD_OK) {
                                gameEngine.transition( GameEngine.SCREEN_GAME );
                            }
                            else if (result == Dialog.CMD_CANCEL) {
                                gameEngine.exit();
                            }
                        }
                    };
                    dlg.show(listener, Dialog.DLG_VODA_START_EXIT, "BEJEWELED",
                        StringTable.VODAFONE_TRIAL + "|" + StringTable.VODAFONE_TRIAL_DESC);
                }
                else if (item == StringTable.EXIT_TITLE) {
                    handleExit();
                }
            }
            else {
                if (Build.DEBUG) System.err.println("Ignoring pause item " + index);
            }
         }
         if (dis == pauseMenu && cmd.getCommandType() == Command.BACK) 
         {
             isPaused = false;
             gameEngine.restart(GameEngine.SCREEN_GAME);
             gameEngine.getMIDlet().getDisplay().setCurrent(gameEngine);
         }
         // Help game button
         if ( cmd == helpCmd )
         {
             handleHelp();
         }
         // Exit game button
         if ( cmd == exitCmd )
         {
             handleExit();
         }
    }

}
