/*
 *    Copyright (c)2001-2003 Jamdat Mobile, Inc. All Rights Reserved.
 *
 * The information contained herein is the CONFIDENTIAL and PROPRIETARY
 *                 information of Jamdat Mobile, Inc.
 */

package com.bejeweled2_j2me;


import javax.microedition.lcdui.Graphics;
import javax.microedition.lcdui.Image;
import java.io.IOException;

/**
 * The GemField is the primary game logic mechanism for Bejeweled. It contains
 * an 8x8 grid of Gem instances and is responsible for handling matches, cascades,
 * cave-ins, etc. Most of the API events from the parent class (e.g., draw, heartbeat,
 * keypress, etc.) are forwarded to each gem on the field from here. The GemField
 * can be used to represent both the local field for single player use, as well as
 * the remote field for multiplayer. All code for both field types is included here
 * in order to optimize code space.
 *
 * @author      DemiVision: Barry Sohl
 * @version     0.9.9
 */
final class GemField extends GameObject
{
    /* Constants */

    // Number of tiles on both x and y axis
    static final byte NUM_ROWS = (byte)DeviceSpecific.GEM_FIELD_ROWS;
    static final byte NUM_COLS = (byte)DeviceSpecific.GEM_FIELD_COLS;

    // Duration of various animation states
    static final int DURATION_FULL_CASCADE1 = 2000;
    static final int DURATION_FULL_CASCADE2 = 2000;
    static final int DURATION_MOVE_HINT     = 15000;
    static final int DURATION_SWAP          =  400;
    static final int DURATION_CLEAR         = 1000;
    static final int DURATION_CLEAR2        =  250;
    static final int DURATION_CASCADE       =  100;
    static final int DURATION_FILL          =  100;
    static final int DURATION_CAVE_IN       = 2000;
    static final int DURATION_FLASH         = 2000;

    // Scoring info
    private static final byte[]     PTS_MATCH   = { 0, 0, 10, 20, 30, 50 };
    private static final int[]    PTS_BONUS   = { 0, 10, 20, 30, 50, 70, 100, 150, 200 };
    private static final byte[]     TIME_BONUS  = { 0, 2, 3, 5 };
    private static final byte       CLEAR_BONUS_COUNT = 5;
    private static final byte       CLEAR_BONUS_SCORE = 10;
    private static final byte       MAXIMUM_BONUS_MULTIPLIER = 4;

    private static final byte BONUS_ANIMATION_FRAMES = 6;
    private static final long BONUS_ANIMATION_FRAMETIME = 150;

    // Extra fudge factor where needed
    private static final int DELAY_SHORT = 150;

    // Minimum gems required in a match
    private static final byte MIN_MATCH_LEN = 3;

    // Potential match results
    private static final byte POTENTIAL_INDEX_RES   = 0;
    private static final byte POTENTIAL_INDEX_ROW   = 1;
    private static final byte POTENTIAL_INDEX_COL   = 2;
    private static final byte POTENTIAL_MATCH_NONE  = 0;
    private static final byte POTENTIAL_MATCH_HORZ  = 1;
    private static final byte POTENTIAL_MATCH_VERT  = 2;

    // Gem field states
    private static final byte STATE_INIT          = 0;
    private static final byte STATE_FULL_CASCADE  = 1;
    private static final byte STATE_FULL_CASCADE1 = 2;
    private static final byte STATE_FULL_CASCADE2 = 3;
    private static final byte STATE_MOVE          = 5;
    private static final byte STATE_SELECT        = 6;
    private static final byte STATE_SWAP          = 7;
    private static final byte STATE_CLEAR         = 8;
    private static final byte STATE_CASCADE       = 9;
    private static final byte STATE_FILL          = 10;
    private static final byte STATE_CAVE_IN       = 11;
    private static final byte STATE_BONUS         = 12;

    // Color settings
    public static int COLOR_TILE1;
    public static int COLOR_TILE2;
    private static final int COLOR_BORDER = 0x404040;

    /* Data Fields */
    private GameBoard gameBoard;                // External references
    private TimeBar timeBar;

    private int tileSize;                       // Pixel dimensions of tiles

    private int cursorRow, cursorCol;           // Cursor location, swap destination
    private int cursorPrevRow, cursorPrevCol;

    private Gem[][] gemGrid;                    // Master gem matrix for rendering
    private Gem selectedGem;

    private byte[][] workGrid;                  // Gem matrices used for game logic
    private byte[][] matchGrid;

    private boolean validSwap;                  // Was current swap valid?

    private byte[] dropCounts;                  // Cascade handling
    private int maxDropCount;
    private boolean firstCascade;

    private byte cascadeMatches;                // Local score tracking
    private byte cascadeLength;

    private static Image bonusImage;            // Bonus tracking
    private static AnimatedImage[] bonusAnimations;
    private byte bonusCount;
    private byte bonusMultiplier;

    private boolean paused;

    /**
     * Constructor saves settings needed for initialization.
     *
     * @param   gameBoard   reference to parent game board.
     * @param   timeBar     reference to time bar instance.
     * @param   tileSize    pixel dimensions of individual field tile.
     */
    GemField( GameBoard gameBoard, TimeBar timeBar, int tileSize)
    {
        this.gameBoard  = gameBoard;
        this.timeBar    = timeBar;
        this.tileSize   = tileSize;
    }

    /**
     * Sets all of the gems on the gem field to the specified state.
     *
     * @param   newState    new state for each gem.
     */
    private void setGemStates( byte newState )
    {
        int row, col;

        for ( row = 0; row < NUM_ROWS; row++ )
        {
            for ( col = 0; col < NUM_COLS; col++ )
            {
                gemGrid[row][col].setState( newState );
            }
        }
    }

    /**
     * Prepares the field for the start of gameplay following completion of the
     * initial full cascade. In multiplayer mode, gameplay must also wait for a
     * game synchronization event from the opponent.
     */
    private void startPlay()
    {
        setState( STATE_MOVE );

        // Place cursor in middle of field
        cursorCol = ((NUM_COLS / 2) - 1);
        cursorRow = ((NUM_ROWS / 2) - 1);

        // Start with highlighted gem
        selectedGem = gemGrid[ cursorRow ][ cursorCol ];
        selectedGem.setFocus( true );

        // Start time bar up
        timeBar.ready();
    }

    /**
     * Randomly chooses a gem from among the available gem types.
     *
     * @return  type identifier of randomly chosen gem.
     */
    private byte chooseGem()
    {
        return (byte) gameEngine.genRandomNumber( Gem.numberOfTypes );
    }

    /**
     * Causes the gem field to cave-in and then repopulate. The opponent
     * can be notified of the event if desired.
     */
    private void caveInField()
    {
        setState( STATE_CAVE_IN );
        setGemStates( Gem.STATE_CAVE_IN );

        if ( selectedGem != null )
        {
            selectedGem.setFocus( false );
        }
    }

    /**
     * Fills the entire gem field with randomly chosen gems. The field fill
     * is continuously attempted until an entire field has been chosen that
     * contains no matches but does have a potential match.
     */
    private void fillField()
    {
        // Change states to prevent duplicate calls, but do not start animation
        setState( STATE_FULL_CASCADE );

        int row, col;
        byte type;

         do
         {
            do
            {
                // Randomly choose a gem for every position
                for ( row = 0; row < NUM_ROWS; row++ )
                {
                    for ( col = 0; col < NUM_COLS; col++ )
                    {
                        type = chooseGem();

                        // Retain this in case future testing is necessary.
//                      // TEST: Override to generate specific starting pattern.
//                      final int pattern[][] = {
//                          {4, 1, 0},
//                          {4, 2, 0},
//                          {2, 3, 0},
//                          {3, 3, 0},
//                          {5, 3, 0},
//                          {6, 3, 0},
//                          {4, 4, 0},
//                          {4, 5, 0}
//                      };
//                      for(int index = 0; index < pattern.length; ++index) {
//                          if((pattern[index][0] == col) &&
//                              (pattern[index][1] == row)) {
//                              type = (byte)(pattern[index][2]);
//                          }
//                      }

                        gemGrid[row][col].setType( type, Gem.STATE_INIT );

                        workGrid[row][col] = type;
                    }
                }
            }
            while ( findAllMatches( false ) > 0 );
        }
        while ( hasPotentialMatch()[POTENTIAL_INDEX_RES] == POTENTIAL_MATCH_NONE );

        // Above algorithm is slow, must finish before starting animation
        setState( STATE_FULL_CASCADE1 );
    }

    /**
     * Searches for a match given a specific starting location on the gem field.
     * The search can be made either vertically (down) or horizontally (to the right).
     * If desired, the match can be persisted to the match grid.
     *
     * @param   row         row position of search start.
     * @param   col         column position of search start.
     * @param   vertical    search vertically or horizontally?
     * @param   persist     save match location to match grid? (WARNING!!! true causes side effect)
     *
     * @return  true if match found, else false.
     */
    private boolean findMatch( int row, int col, boolean vertical, boolean persist )
    {
        int matchLen = 1;
        byte matchType = workGrid[row][col];

        // No need to check empty gems
        if ( matchType == Gem.TYPE_EMPTY )
        {
            return false;
        }

        boolean matchFound = false;

        int row2, col2;

        // Search the remainder of the column or row for a match
        for ( int i = ((vertical ? row : col) + 1); i < (vertical ? NUM_ROWS : NUM_COLS); i++ )
        {
            row2 = (vertical ? i : row);
            col2 = (vertical ? col : i);

            // Short circuit once match broken
            if ( workGrid[row2][col2] != matchType )
            {
                break;
            }

            // Match already counted as part of larger match
            if ( (matchGrid[row][col] != Gem.TYPE_EMPTY) &&
                 (matchGrid[row][col] == matchGrid[row2][col2]) )
            {
                break;
            }

            matchLen++;
        }

        // If there were at least 3 contiguous gems, we have a match
        if ( matchLen >= MIN_MATCH_LEN )
        {
            matchFound = true;

            // Save to match grid
            if ( persist )
            {
                for ( int i = 0; i < matchLen; i++ )
                {
                    matchGrid[ vertical ? (row + i) : row ][ vertical ? col : (col + i) ] = matchType;
                }

                // Credit match points
                gameBoard.addScore( PTS_MATCH[ matchLen - 1 ]);

                // Record score
                if ( matchLen > gameBoard.p1Biggest )
                {
                    gameBoard.p1Biggest = (byte)matchLen;
                }
            }
        }

        return matchFound;
    }

    /**
     * Finds all matches currently present on the gem field. If desired, the match
     * locations can be persisted to the match grid. A count of the total number
     * of matches found will be returned.
     *
     * @param   persist     save match locations to match grid?
     * @return  total number matches found.
     */
    private int findAllMatches( boolean persist )
    {
        // Start with fresh workspace
        emptyGrid( matchGrid );

        int numMatches = 0;
        int row, col;

        for ( row = 0; row < NUM_ROWS; row++ )
        {
            for ( col = 0; col < NUM_COLS; col++ )
            {
                // Check to the right and then down for each grid cell
                boolean horizontalMatch = findMatch(row, col, false, persist);
                boolean verticalMatch = findMatch(row, col, true, persist);
                if(horizontalMatch || verticalMatch) {
                    numMatches++;
                }
            }
        }

        // Track for scoring purposes
        if ( persist )
        {
            cascadeMatches += numMatches;
            gameBoard.p1Matches += numMatches;
        }

        return numMatches;
    }

    /**
     * Performs a gem swap between the gem at the previous cursor location and
     * the gem at the location the cursor was just moved to. If the requested
     * swap is valid, the change takes places and the swap animation is drawn.
     * If the swap is invalid, the double-swap error animation is drawn, and
     * the change is not saved.
     */
    private void swapGems()
    {
        setState( STATE_SWAP );

        // Track scoring
        gameBoard.p1Swaps++;

        cascadeMatches = 0;
        cascadeLength = 0;

        // Differentiate between matches caused by swapping and
        // those caused by cascades.
        //
        firstCascade = true;

        // Can't swap off edge of field or with an empty gem
        if ( (cursorRow < 0) || (cursorRow >= NUM_ROWS) ||
             (cursorCol < 0) || (cursorCol >= NUM_COLS) ||
             gemGrid[ cursorRow ][ cursorCol ].isEmpty() )
        {
            cursorRow = cursorPrevRow;
            cursorCol = cursorPrevCol;

            invalidateSwap();
            return;
        }

        // Swap two gems in workspace grid
        swapCells( workGrid, cursorPrevRow, cursorPrevCol, cursorRow, cursorCol );

        // Determine if valid swap
        validSwap = (findAllMatches( true ) > 0);

        // Start swap animation
        gemGrid[ cursorPrevRow ][ cursorPrevCol ].swap( validSwap, cursorRow, cursorCol );
        gemGrid[ cursorRow ][ cursorCol ].swap( validSwap, cursorPrevRow, cursorPrevCol );

        // If invalid swap, reverse change
        if ( !validSwap )
        {
            swapCells( workGrid, cursorRow, cursorCol, cursorPrevRow, cursorPrevCol );

            // Sound effects only when not near death
            if(!timeBar.isNearDeath()) {
                gameEngine.playMidi( GameBoard.MIDI_BAD_SWAP, false, false);
                gameEngine.playAlertSound( GameBoard.ALERT_BAD_SWAP, false);
            }
        }
    }

    /**
     * Invalidates an attempted gem swap by returning the selection focus
     * to the previous location and resetting the gem states.
     */
    private void invalidateSwap()
    {
        setState( STATE_MOVE );

        gemGrid[ cursorRow ][ cursorCol ].setState( Gem.STATE_NORMAL );

        // Return focus to previous gem
        selectedGem.setState( Gem.STATE_NORMAL );
        selectedGem.setFocus( true );

        gameBoard.p1BadSwaps++;
    }

    /**
     * Remove a random gem from the field.
     */
    private void clearBonusGems()
    {
        setState(STATE_BONUS);

        // Choose a random cell
        int base_row = gameEngine.genRandomNumber(NUM_ROWS);
        int base_col = gameEngine.genRandomNumber(NUM_COLS);
        int row = base_row;
        int col = base_col;

        // Find a non-empty cell
        while(matchGrid[row][col] != Gem.TYPE_EMPTY) {
            if(++row >= NUM_ROWS) {
                row = 0;
                if(++col >= NUM_COLS) {
                    col = 0;
                }
            }
            if((row == base_row) && (col == base_col)) {
                break;
            }
        }

        // Make sure we found one, then clear it. This should only fail
        // if no gems remain.
        if(matchGrid[row][col] == Gem.TYPE_EMPTY) {
            matchGrid[row][col] = workGrid[row][col];
            gemGrid[row][col].setState(Gem.STATE_CLEAR);

            // Add bonus points
            gameBoard.addScore(bonusMultiplier * CLEAR_BONUS_SCORE);
        }

        // One less bonus to clear
        --bonusCount;
    }

    /**
     * Clears all matches currently present on the field. All cleared gems will
     * display the clearing animation.
     *
     * @param   first   is this first match clearing in cascade chain?
     */
    private void clearMatches( boolean first )
    {
        setState( STATE_CLEAR );

        int row, col;

        for ( row = 0; row < NUM_ROWS; row++ )
        {
            for ( col = 0; col < NUM_COLS; col++ )
            {
                // This gem part of a match
                if ( matchGrid[row][col] != Gem.TYPE_EMPTY )
                {
                    gemGrid[row][col].setState( Gem.STATE_CLEAR );
                }
            }
        }

        // Track cascade scoring
        cascadeLength++;

        // Track cascade chains
        if ( !first )
        {
            if ( cascadeLength > gameBoard.p1Longest )
            {
                gameBoard.p1Longest = cascadeLength;
            }

            gameBoard.p1Cascades++;
        }

        // Sound effects only when not near death
        if ( validSwap && !timeBar.isNearDeath() )
        {
            gameEngine.playMidi( GameBoard.MIDI_SWAP, false, false);
            gameEngine.playAlertSound( GameBoard.ALERT_SWAP, false);
        }
    }

    /**
     * Handles the start of a gem cascade sequence. The action taken depends
     * on whether or not a weapon is currently in effect.
     */
    private void handleCascade()
    {
        setState( STATE_CASCADE );

        int row, col;
        int destRow;

        byte dropCount;
        byte newType;

        maxDropCount = 0;

        // Start cascade for each column
        for ( col = 0; col < NUM_COLS; col++ )
        {
            row = (NUM_ROWS - 1);
            dropCount = 0;

            // There may be multiple matches in the same column
            while ( row >= 0 )
            {
                // Find the first match from starting position
                while ( (row >= 0) && (matchGrid[row][col] == Gem.TYPE_EMPTY) )
                {
                    row--;
                }

                // Count number of consecutive matched gems
                while ( (row >= 0) && (matchGrid[row][col] != Gem.TYPE_EMPTY) )
                {
                    dropCount++;
                    row--;
                }

                // All gems between this match and the next one cascade downward
                while ( (row >= 0) && (matchGrid[row][col] == Gem.TYPE_EMPTY) )
                {
                    destRow = (row + dropCount);
                    newType = workGrid[row][col];

                    gemGrid[ destRow ][col].cascade( dropCount, newType);
                    workGrid[ destRow ][col] = newType;

                    row--;
                }
            }

            // Leave space for new gems to fall in
            for ( row = 0; row < dropCount; row++ )
            {
                gemGrid[row][col].setType( Gem.TYPE_EMPTY, Gem.STATE_EMPTY );
                workGrid[row][col] = Gem.TYPE_EMPTY;
            }

            // Record total gems cleared from column
            dropCounts[col] = dropCount;

            if ( dropCount > maxDropCount )
            {
                maxDropCount = dropCount;
            }
        }

        firstCascade = false;
    }

    /**
     * Fills new gems into the spaces opened by a cascade. Each gem will smoothly
     * animate towards its new position. Single player gems are chosen randomly.
     * Multiplayer gems are chosen from a buffer sent by the opponent.
     */
    private void fillGems()
    {
        setState( STATE_FILL );

        int row, col;
        byte newType;

        for ( col = 0; col < NUM_COLS; col++ )
        {
            // Fill open space at top of column with new gems
            for ( row = 0; row < dropCounts[col]; row++ )
            {
                // Fill randomly
                newType = chooseGem();

                gemGrid[row][col].cascade( dropCounts[col], newType);
                workGrid[row][col] = newType;
            }
        }
    }

    /**
     * Tests an area surrounding the specified gem for matches resulting
     * from swapping the gem either vertically (down) or horizontally
     * (to the right). For efficiency, the minimum possible number of grid
     * positions are tested.
     *
     * @param   vertical    test a vertical (down) swap?
     * @param   row         field row of gem being tested.
     * @param   col         field column of gem being tested.
     *
     * @return  true if area contains a match, else false.
     */
    private boolean testMatchArea( boolean vertical, int row, int col )
    {
        boolean match = false;

        // Position of second gem, either down or to the right
        int row2 = vertical ? (row + 1) : row;
        int col2 = vertical ? col : (col + 1);

        if ( (row2 < NUM_ROWS) && (col2 < NUM_COLS) )
        {
            int minMatch2 = (MIN_MATCH_LEN - 1);
            int numRows2 = (NUM_ROWS - 1);
            int numCols2 = (NUM_COLS - 1);

            int range;
            int startPt, endPt;
            int testRow, testCol;

            // Swap with second gem
            swapCells( workGrid, row, col, row2, col2 );

            // Determine area to be tested
            startPt = Math.max( (row - minMatch2), 0 );
            endPt   = Math.min( (row + (vertical ? MIN_MATCH_LEN : minMatch2)), numRows2 );
            range   = Math.min( (col + (vertical ? 1 : 2)), numCols2 );

            // Test for vertical matches
            for ( testRow = startPt; testRow <= endPt; testRow++ )
            {
                for ( testCol = col; testCol <= range; testCol++ )
                {
                    if ( findMatch( testRow, testCol, true, false ) )
                    {
                        match = true;
                        break;
                    }
                }
            }

            // Only continue if necessary
            if ( !match )
            {
                // Determine area to be tested
                startPt = Math.max( (col - minMatch2), 0 );
                endPt   = Math.min( (col + (vertical ? minMatch2 : MIN_MATCH_LEN)), numCols2 );
                range   = Math.min( (row + (vertical ? 2 : 1)), numRows2 );

                // Test for horizontal matches
                for ( testRow = row; testRow <= range; testRow++ )
                {
                    for ( testCol = startPt; testCol <= endPt; testCol++ )
                    {
                        if ( findMatch( testRow, testCol, false, false ) )
                        {
                            match = true;
                            break;
                        }
                    }
                }
            }

            // Return gems to original positions
            swapCells( workGrid, row2, col2, row, col );
        }

        // Record new hint location

        return match;
    }

    /**
     * Checks if there is currently a match possible on the gem field. Every
     * possible swap combination is attempted until a match is found. The first
     * potential match that is encountered will be recorded as a match hint.
     *
     * @return  int[3] with the following values:
     *
     *          int[POTENTIAL_INDEX_RES] = {
     *              POTENTIAL_MATCH_NONE means no match, other indices are invalid
     *              POTENTIAL_MATCH_HORZ means potential horizontal match, see below
     *              POTENTIAL_MATCH_VERT means potential vertical match, see below
     *          }
     *
     *          if potential horizontal match then
     *              Gem1 @ (int[POTENTIAL_INDEX_ROW],int[POTENTIAL_INDEX_COL])
     *              Gem2 @ (int[POTENTIAL_INDEX_ROW],int[POTENTIAL_INDEX_COL]+1)
     *
     *          if potential vertical match then
     *              Gem1 @ (int[POTENTIAL_INDEX_ROW],int[POTENTIAL_INDEX_COL])
     *              Gem2 @ (int[POTENTIAL_INDEX_ROW]+1,int[POTENTIAL_INDEX_COL])
     */
    private int[] hasPotentialMatch()
    {
        // Start with fresh workspace
        emptyGrid( matchGrid );
        int[] potential = new int[3];
        potential[POTENTIAL_INDEX_RES] = POTENTIAL_MATCH_NONE;

        // Perform every possible swap
        for ( potential[POTENTIAL_INDEX_ROW] = 0;
              potential[POTENTIAL_INDEX_ROW] < NUM_ROWS;
              potential[POTENTIAL_INDEX_ROW]++ )
        {
            for ( potential[POTENTIAL_INDEX_COL] = 0;
                  potential[POTENTIAL_INDEX_COL] < NUM_COLS;
                  potential[POTENTIAL_INDEX_COL]++ )
            {
                // Test both vertical and horizontal swap
                if ( testMatchArea( true,
                    potential[POTENTIAL_INDEX_ROW],
                    potential[POTENTIAL_INDEX_COL] ) )
                {
                    potential[POTENTIAL_INDEX_RES] = POTENTIAL_MATCH_VERT;
                    return potential;
                }
                if ( testMatchArea( false,
                    potential[POTENTIAL_INDEX_ROW],
                    potential[POTENTIAL_INDEX_COL] ) )
                {
                    potential[POTENTIAL_INDEX_RES] = POTENTIAL_MATCH_HORZ;
                    return potential;
                }
            }
        }

        // No matches possible
        return potential;
    }

    /**
     * Swaps the two cells at the specified locations in the provided gem grid.
     *
     * @param   grid    gem grid in which swap will take place.
     *
     * @param   row1    row of first gem in swap.
     * @param   col1    column of first gem in swap.
     *
     * @param   row2    row of second gem in swap.
     * @param   col2    column of second gem in swap.
     */
    private void swapCells( byte[][] grid, int row1, int col1, int row2, int col2 )
    {
        byte temp = grid[row1][col1];

        grid[row1][col1] = grid[row2][col2];
        grid[row2][col2] = temp;
    }

    /**
     * Empties an entire gem grid by setting all values to empty.
     *
     * @param   grid    gem grid to be emptied.
     */
    private void emptyGrid( byte[][] grid )
    {
        int row, col;

        for ( row = 0; row < NUM_ROWS; row++ )
        {
            for ( col = 0; col < NUM_COLS; col++ )
            {
                grid[row][col] = Gem.TYPE_EMPTY;
            }
        }
    }

    /**
     * Erases an individual background tile on the gem field. The tiles are
     * drawn in alternating colors across the field.
     *
     * @param   gc      graphics context used for drawing.
     * @param   row     tile row being erased.
     * @param   col     tile column being erased.
     */
    private void eraseTile( Graphics gc, int row, int col )
    {
        boolean evenRow = ((row % 2) == 0);

        // Rows start with alternating tile color, then alternate across row
        boolean tile1 = ((col % 2) == 0) ? evenRow : !evenRow;

        // Draw background at tile location
        int tileX = x + (col * tileSize);
        int tileY = y + (row * tileSize);

        gc.setColor( tile1 ? COLOR_TILE1 : COLOR_TILE2 );
        gc.fillRect( tileX, tileY, tileSize, tileSize );
    }

    /* GameObject */

    /**
     * Initializes the gem field. Drawing colors are set and all gem matrices
     * are created. Significant image loading and memory allocation occurs here.
     *
     * @throws  IOException if error occurs loading resourcs.
     */
    public void init() throws IOException
    {
        // Read external settings
        COLOR_TILE1 = 7757382;
        COLOR_TILE2 = 3748137;

        // Create gem grid used for all rendering
        gemGrid = new Gem[ NUM_ROWS ][ NUM_COLS ];

        for ( byte row = 0; row < NUM_ROWS; row++ )
        {
            boolean tile1 = ((row % 2) == 0);

            for ( byte col = 0; col < NUM_COLS; col++ )
            {
                // Create and init individual gem
                Gem newGem = new Gem(this, row, col, y );

                int gemX = (x + (col * tileSize));
                int gemY = (y + (row * tileSize));

                newGem.setBounds( gemX, gemY, tileSize, tileSize );
                newGem.setBgndColor( tile1 ? COLOR_TILE1 : COLOR_TILE2 );
                newGem.init();

                gemGrid[row][col] = newGem;

                // Tile color alternates going across row
                tile1 = !tile1;
            }
        }

        // Create game logic working space
        workGrid   = new byte[ NUM_ROWS ][ NUM_COLS ];
        matchGrid  = new byte[ NUM_ROWS ][ NUM_COLS ];
        dropCounts = new byte[ NUM_COLS ];

        // Load and initialize bonus image animations
        if(bonusImage == null) {
            bonusImage = gameEngine.loadImage(
                StringTable.FILE_BONUS_ANIMATIONS, true);
            bonusAnimations = new AnimatedImage[MAXIMUM_BONUS_MULTIPLIER];
            bonusAnimations[0] = null;
            int bonusWd =
                (bonusImage.getWidth() / (bonusAnimations.length - 1));
            int bonusHt =
                (bonusImage.getHeight() / BONUS_ANIMATION_FRAMES);
            int bonusX = (x + ((wd - bonusWd) / 2));
            int bonusY = (y + ((ht - bonusHt) / 2));
            for(int index = 1; index < bonusAnimations.length; ++index) {
                bonusAnimations[index] = new AnimatedImage(
                    bonusImage, bonusX, bonusY, bonusWd, bonusHt,
                    BONUS_ANIMATION_FRAMES,
                    (bonusWd * (index - 1)), 0,
                    0, bonusHt, BONUS_ANIMATION_FRAMETIME, true
                );
            }
        }
    }

    /**
     * Called at the start of each game. The gem field is filled and the
     * game logic is reset. In multiplayer mode, the initial field is
     * sent to the opponent.
     *
     * @param   offset  level offset of node.
     */
    public void startNode( byte offset )
    {
        setState( STATE_INIT );
        setGemStates( Gem.STATE_INIT );

        // Reset game logic
        emptyGrid( workGrid );

        // Populate initial field
        fillField();

        // Reset scoring
        bonusCount = 0;
        bonusMultiplier = 1;
    }

    /**
     * Marks the gem field as dirty and therefore requiring a redraw. All
     * child gems will be marked dirty as well.
     */
    public void markDirty()
    {
        super.markDirty();

        int row, col;

        // Mark all children dirty
        for ( row = 0; row < NUM_ROWS; row++ )
        {
            for ( col = 0; col < NUM_COLS; col++ )
            {
                gemGrid[row][col].markDirty();
            }
        }
    }

    /**
     * Draws the entire gem field in its current state. All child gems are
     * instructed to draw as well.
     *
     * @param   gc  graphics context used for drawing.
     */
    public void draw( Graphics gc )
    {
        // For states that involve gem overlap, gem field must pre-erase
        if ( getState() == STATE_SWAP )
        {
            eraseTile( gc, cursorPrevRow, cursorPrevCol );
            eraseTile( gc, cursorRow, cursorCol );
        }

        int row, col;

        // Instruct children to draw
        for ( row = 0; row < NUM_ROWS; row++ )
        {
            for ( col = 0; col < NUM_COLS; col++ )
            {
                gemGrid[row][col].draw( gc );
            }
        }

        // Draw either normal or pulsating weapon border
        if ( dirty )
        {
            gc.setColor( COLOR_BORDER );
            gc.drawRect( (x - 1), (y - 1), (wd + 1), (ht + 1) );

            dirty = false;
        }
    }

    /**
     * Handles all time dependent behavior of the gem field, primarily moving
     * through animation states. The heartbeat is passed to all child gems as well.
     *
     * @param   now     current time as milliseconds since unix epoch.
     */
    public void heartbeat( long now )
    {
        long elapsed = (now - stateTime);

        switch ( getState() )
        {
            // Slight delay at start of game before full cascade starts
            case STATE_FULL_CASCADE1:
            {
                if ( elapsed > DURATION_FULL_CASCADE1 )
                {
                    setState( STATE_FULL_CASCADE2 );
                    setGemStates( Gem.STATE_FULL_CASCADE );
                }

                break;
            }
            // Selection cursor is placed once full cascade completes
            case STATE_FULL_CASCADE2:
            {
                if ( elapsed > DURATION_FULL_CASCADE2 )
                {
                    startPlay();
                }

                break;
            }
            // Offer hint if player takes too long to move
            case STATE_MOVE:
            {
                if((elapsed > DURATION_MOVE_HINT) &&
                    (gameEngine.getSettings().isOn(SettingsScreen.FLAG_HINT)))
                {
                    int[] potential = hasPotentialMatch();
                    int result = potential[POTENTIAL_INDEX_RES];

                    // Just in case a match doesn't exist, although this
                    // should not happen according to the rules.
                    if( result != POTENTIAL_MATCH_NONE )
                    {
                        int row = potential[POTENTIAL_INDEX_ROW];
                        int col = potential[POTENTIAL_INDEX_COL];
                        gemGrid[row][col].flash(true, true);
                        switch(result) {
                            case POTENTIAL_MATCH_HORZ:
                                gemGrid[row][col + 1].flash(true, false);
                                break;
                            case POTENTIAL_MATCH_VERT:
                                gemGrid[row + 1][col].flash(true, false);
                                break;
                        }
                    }

                    // Reset state time.
                    setState(STATE_MOVE);
                }
                break;
            }
            // Finalize swap after animation completes
            case STATE_SWAP:
            {
                // Fudge delay needed on slower devices
                int delay = DELAY_SHORT;

                int duration = (DURATION_SWAP + delay);

                if ( elapsed > (validSwap ? duration : (duration * 2)) )
                {
                    // After valid swap, start clear delay before cascading
                    if ( validSwap )
                    {
                        Gem prevGem = gemGrid[ cursorPrevRow ][ cursorPrevCol ];
                        selectedGem = gemGrid[ cursorRow ][ cursorCol ];

                        // Stop swap animation
                        prevGem.setType( workGrid[ cursorPrevRow ][ cursorPrevCol ], Gem.STATE_NORMAL );
                        selectedGem.setType( workGrid[ cursorRow ][ cursorCol ], Gem.STATE_NORMAL );

                        // Mark gems for clearing
                        clearMatches( true );
                    }
                    // After invalid swap, undo cursor movement
                    else
                    {

                        invalidateSwap();

                        cursorRow = cursorPrevRow;
                        cursorCol = cursorPrevCol;
                    }
                }

                break;
            }
            // Slight delay after a match before cascade begins
            case STATE_CLEAR:
            {
                if ( elapsed > (firstCascade ? DURATION_CLEAR : DURATION_CLEAR2) )
                {
                    // Multiplayer local field must wait for swap ack
                    handleCascade();
                }

                break;
            }
            // Slight delay after cascade completes before new gems fill in
            case STATE_CASCADE:
            {
                if(paused || elapsed > ((maxDropCount * DURATION_CASCADE) + DURATION_FILL) ) {
                    paused = false;
                    fillGems();
                }

                break;
            }
            // After fill completes, check for next iteration of cascade
            case STATE_FILL:
            {
                int duration = (maxDropCount * DURATION_CASCADE);

                if ( elapsed > duration )
                {
                    // Add bonus time
                    if(cascadeMatches > 0) {
                        timeBar.addTimeBonus(
                            TIME_BONUS[Math.min(cascadeMatches, TIME_BONUS.length) - 1]);
                    }

                    // More matches, start cascade
                    if ( findAllMatches( true ) > 0 )
                    {
                        clearMatches( false );
                    }
                    // Cascades done (J2ME requires extra delay)
                    else if ( elapsed > (duration + DELAY_SHORT) )
                    {
                        // Add bonus score
                        if(cascadeMatches > 0) {
                            gameBoard.addScore(
                                PTS_BONUS[Math.min(cascadeMatches,
                                    PTS_BONUS.length) - 1]);
                        }

                        // Reset
                        emptyGrid( matchGrid );

                        // If no more matches, show message, cave-in, and restart
                        if ( hasPotentialMatch()[POTENTIAL_INDEX_RES] == POTENTIAL_MATCH_NONE )
                        {
                            gameBoard.p1NoMatches++;

                            gameBoard.showOverlay( StringTable.NO_MOVES );
                            caveInField();
                        }
                        // Check for bonus
                        else if(gameBoard.p1Power > 98)
                        {
                            // Increment bonus multiplier unless maxed
                            if(bonusMultiplier < MAXIMUM_BONUS_MULTIPLIER) {
                                ++bonusMultiplier;
                            }

                            bonusCount = (byte) (bonusMultiplier * CLEAR_BONUS_COUNT);

                            // Show bonus animation
                            AnimatedImage bonusAnimation = bonusAnimations[bonusMultiplier - 1];
                            if(bonusAnimation != null) {
                                gameBoard.showOverlay(bonusAnimation);
                            }

                            // Start bonus gem clearing
                            clearBonusGems();

                            // Play bonus sound
                            gameEngine.playMidi(GameBoard.MIDI_BONUS, false, false);
                            gameEngine.vibrate(1000, false);
                        }
                        // Otherwise, ready for next move
                        else
                        {
                            setState( STATE_MOVE );
                            selectedGem.setFocus( true );
                        }
                    }
                }

                break;
            }
            // After cave-in completes, repopulate field or end game
            case STATE_CAVE_IN:
            {
                if ( elapsed > DURATION_CAVE_IN )
                {
                    // Lowest difficulty level ends when no more matchs are
                    // available.
                    if(gameBoard.getDifficulty() != GameBoard.DIFFICULTY_NORMAL) {
                        fillField();
                    } else {
                        gameBoard.gameOver();
                    }
                }

                break;
            }
            // Remove gems when a bonus is acheived
            case STATE_BONUS:
            {
                if(elapsed > DURATION_CLEAR2) {
                    if(bonusCount > 0) {
                        clearBonusGems();
                    } else {
                        // Demo ends here
                        if(gameEngine.isDemo()) {
                            gameBoard.gameOver();
                        } else {
                            // Reset
                            cascadeMatches = 0;
                            cascadeLength = 0;
                            timeBar.initTime(false);

                            // Hide bonus animation
                            gameBoard.hideOverlay();

                            // Now cascade gems
                            handleCascade();
                        }
                    }
                }
                break;
            }
        }

        int row, col;

        // Pass heartbeat to children
        for ( row = 0; row < NUM_ROWS; row++ )
        {
            for ( col = 0; col < NUM_COLS; col++ )
            {
                gemGrid[row][col].heartbeat( now );
           }
        }
    }

    /**
     * Handles all key press events for the gem field, primarily moving the
     * selection cursor as well as selecting and swapping gems. Note that
     * cursor movement wraps around the edges of the gem field.
     *
     * @param   keyCode     key code being handled.
     */
    public void keyPressed( int keyCode )
    {
        // Translate to game action.
        keyCode = gameEngine.getGameAction(keyCode);

        switch ( getState() )
        {
            // Cursor is being moved around the field
            case STATE_MOVE:
            {
                // Left or reverse left
                if(keyCode == GameEngine.LEFT)
                {
                    cursorCol = (cursorCol == 0) ? (NUM_COLS - 1) : (cursorCol - 1);
                }
                // Right or reverse right
                else if(keyCode == GameEngine.RIGHT)
                {
                    cursorCol = (cursorCol + 1) % NUM_COLS;
                }
                // Up or reverse up
                else if(keyCode == GameEngine.UP)
                {
                    cursorRow = (cursorRow == 0) ? (NUM_ROWS - 1) : (cursorRow - 1);
                }
                // Down or reverse down
                else if(keyCode == GameEngine.DOWN)
                {
                    cursorRow = (cursorRow + 1) % NUM_ROWS;
                }

                // Reset previous gem, highlight new gem
                selectedGem.setFocus( false );
                selectedGem = gemGrid[ cursorRow ][ cursorCol ];
                selectedGem.setFocus( true );

                // Gem selected, switch to select animation
                if(keyCode == GameEngine.FIRE)
                {
                    if ( !selectedGem.isEmpty() )
                    {
                        selectedGem.setState( Gem.STATE_SELECT );
                        setState( STATE_SELECT );
                    }
                }

                break;
            }
            // Selected gem is being swapped
            case STATE_SELECT:
            {
                // User can deselect by pressing select again
                if(keyCode == GameEngine.FIRE)
                {
                    setState( STATE_MOVE );
                    selectedGem.setState( Gem.STATE_NORMAL );
                }
                else
                {
                    cursorPrevRow = cursorRow;
                    cursorPrevCol = cursorCol;

                    selectedGem.setFocus( false );

                    if(keyCode == GameEngine.LEFT)
                    {
                        cursorCol = (cursorCol - 1);
                        swapGems();
                    }
                    // Right
                    else if(keyCode == GameEngine.RIGHT)
                    {
                        cursorCol = (cursorCol + 1);
                        swapGems();
                    }
                    // Up
                    else if(keyCode == GameEngine.UP)
                    {
                        cursorRow = (cursorRow - 1);
                        swapGems();
                    }
                    // Down
                    else if(keyCode == GameEngine.DOWN)
                    {
                        cursorRow = (cursorRow + 1);
                        swapGems();
                    }
                }
            }
        }
    }

    /**
     * Activates (shows) the game object and performs any necessary setup.
     * Default does nothing.
     *
     * @param   init    initial start after screen transition?
     * @param   now     current system time
     */
    public void start(boolean init, long now) {
        super.start(init, now);

        for(int row = 0; row < NUM_ROWS; row++) {
            for(int col = 0; col < NUM_COLS; col++) {
                gemGrid[row][col].start(init, now);
            }
        }
    }

    /**
     * Hides the game object and performs any necessary cleanup. Default
     * does nothing.
     *
     * @param now current system time
     */
    public void pause(long now) {
        super.pause(now);
        paused = true;
        for(int row = 0; row < NUM_ROWS; row++) {
            for(int col = 0; col < NUM_COLS; col++) {
                gemGrid[row][col].pause(now);
            }
        }
    }

}

/**/
