package sims2;



/* Copyright Ideaworks3D Ltd. 2005 */



import java.util.Enumeration;

import java.util.Vector;



import javax.microedition.lcdui.Graphics;

import javax.microedition.lcdui.game.Sprite;





//import com.ideaworks3d.debug.Debug;



public class EdithObject

{	

	/** Loads images used to draw effects */

	private static void LoadEffectImages()

	{

		try

		{

			// Load cloud images

			if (s_CloudStrip == null)

			{

				s_CloudStrip = new ImageStrip(CLOUD_STRIP_GUID, 0, true);

	

			}

			if (s_CloudContentsStrips == null)

			{

				s_CloudContentsStrips = new ImageStrip[AnimationEffect.NUM_CLOUD_EFFECT_TYPES];

				for (int cloudNum = 0; cloudNum < AnimationEffect.NUM_CLOUD_EFFECT_TYPES; cloudNum++)

				{

					s_CloudContentsStrips[cloudNum] = 

						new ImageStrip(CLOUD_CONTENTS_STRIP_GUIDS[cloudNum], 0, true);

				}

			}

			

			// Make sure mosaic image is loaded

			if (s_MosaicStrip == null)

			{

				s_MosaicStrip = new ImageStrip(MOSAIC_STRIP_GUID, 0, true);

			}

		}

		catch (Exception ex)

		{

			//Debug.assertPrintln(false, "EdithObject:LoadEffectImages() failed to load images:" + ex.getMessage());

		}

	}

	

	/**

	 * Construct an instance of EdithObject of type

	 * 

	 * @param guId

	 * @param appearanceGuid	Appearance guid to use, or pass the guId again to use the one linked in the script 

	 * @param x

	 * @param y

	 * @param isDynamic

	 * @param stringId

	 */

	public EdithObject(int guId, int appearanceGuId, int x, int y, boolean isDynamic, int stringId)

	{

		//Debug.println("Creating EdithObject guid=" + guId + " app=" + appearanceGuId, //Debug.CHAN_CORE);

				

		// Retrive the name string

		m_name = SimsCanvas.s_Canvas.getStringById(stringId);

		

		// Store appearance ID

		m_AppearanceGuId 	= appearanceGuId;

		

		m_GuId 				= guId;

		m_InstanceId 		= ++s_objectIDCounter;

		m_orientation 		= Facing.SE;

		m_LastOrientation	= -1;

		m_TilePosX			= x;

		m_TilePosY			= y;

		

		// only load the appearance data if it exists

		if (appearanceGuId != INVALID_GUID_ID)

		{

			//Debug.println("Loading appearance data, appguid=" + appearanceGuId, //Debug.CHAN_EDITH_OBJECT);

			LoadAppearanceData(appearanceGuId);

		}

		

		// create array of static variables

		//Debug.println("Creating static script vars", //Debug.CHAN_EDITH_OBJECT);

		m_staticScriptVars = new int[StaticVariables.NUM_STAT_VARS];

		m_staticScriptVars[StaticVariables.STAT_CURRENT_CONTAINER_SLOT] = INVALID_INSTANCE_ID;

		

		if(HasAppearance())

		{

			//Debug.println("Initialising container slots", //Debug.CHAN_EDITH_OBJECT);

			

			// set up the int array that will store the container slot usage 

			// (Stores the InstanceId of the object contained in it)

			int numContainerSlots = GetNumContainerSlots();

			m_containerSlots = new int[numContainerSlots];

			

			for (int i = 0; i < numContainerSlots; i++)

			{

				this.m_containerSlots[i] = INVALID_INSTANCE_ID;

			}

			

			//Debug.println("Init first anim on menu object", //Debug.CHAN_EDITH_OBJECT);

			

			// Find the first valid animation.

			for(int i = 0; m_CurAnim == null && i < SimObject.NUM_ANIM_IDS; ++i)

			{

				ControlAnimation(i, true, AnimationEffect.NONE);

			}

			

			// Construct billboard

			m_Billboard = new Billboard(isDynamic);

			

			m_CurAnim.DrawToBillBoard(m_Billboard, true);

			

			// Set draw position caching variables

			m_RecalculateDrawPos = true;

		}

		

		// Load cloud and mosaic effect images

		//Debug.println("Loading static animation effects", //Debug.CHAN_EDITH_OBJECT);

		LoadEffectImages();

		

		// Only needed if this is never called ever!

		UpdateBillboards();

		

		//Debug.println("Edith object constructed", //Debug.CHAN_EDITH_OBJECT);

		

	} // end EdithObject constructor

	

	/**

	 * Make a debug string for this object.

	 * @return string

	 */

	public String toString() 

	{

		//return this.m_name;

		return "\"" + m_name + "\"(guid=" + m_GuId + " instance=" + m_InstanceId + ")"; 

	} // end toString



	public void PrintInfo()

	{

//		if (!//Debug.ENABLED)

//			return;

		int[] size = GetDimensions();

//START:print info#

		//Debug.println(m_name + ": " + size[0] + " " + size[1] + " " + m_TilePosX + " " + m_TilePosY, //Debug.CHAN_EDITH_OBJECT);

//END

		int len = GetNumContainerSlots();

		for (int i = 0; i < len; i++)

		{

			int[] slot = GetContainerSlotOffset(i);

			//Debug.println(m_name + ": slot " + i + ": " + slot[0] + " " + size[1], //Debug.CHAN_EDITH_OBJECT);

		}

	}

	

	public int GetXPos()

	{

		return m_TilePosX;

	}

	

	public int GetYPos()

	{

		return m_TilePosY;

	}

	

	public void SetXPos(int xPos)

	{

		SetPos(xPos, m_TilePosY);

	}

	

	public void SetYPos(int yPos)

	{

		SetPos(m_TilePosX, yPos);

	}

	

	public int GetFacing()

	{

		return m_orientation;

	}

	

	public void ForceSetFacing(int facing)

	{

		// Show the correct graphic.

		// There is a bug in the object code

		// which resets the facing direction.

		SetFacing(m_orientation);

		SetFacing(facing);

		UpdateBillboards();

		

		// Force redraw.

		Refresh();

	}

	

	public void SetFacing(int facing)

	{

		m_orientation = facing % Facing.NUM_DIRECTIONS;

		if (m_orientation < 0)

			m_orientation += Facing.NUM_DIRECTIONS;

		m_RecalculateDrawPos = true;



		if (m_orientation != m_LastOrientation)

		{

			if (m_CurAnim != null)

			{

				m_CurAnim.SetFacing(m_orientation);

			}

			m_LastOrientation = m_orientation;

		}

	}



	public int[] GetTopPos()

	{

//START:gettoppos#

		int[] topPos = { m_TilePosX, m_TilePosY };

		int[] result = new int[] {m_Width, m_Height};

		IsoMap.RotatePosInPlace(result, m_orientation);

		if (result[0] < 0)

			topPos[0] += result[0] + 1; 

		if (result[1] < 0)

			topPos[1] += result[1] + 1;

		return topPos;

//END

	}

	

	/** Obtain dimensions in grid cells at current facing direction */

	public int[] GetDimensions()

	{

//START:getdimensions#

		int[] result = new int[] {m_Width, m_Height};

		IsoMap.RotatePosInPlace(result, m_orientation);

//END

		result[0] = Math.abs(result[0]);

		result[1] = Math.abs(result[1]);

		

		return result;

	}

	

	/**

	 * Test whether this objects position is the same as another's

	 * @param eObject Other object

	 * @return true if objects are in the same position

	 */

	public boolean PositionEquals(EdithObject eObject)

	{

		return (m_TilePosX == eObject.m_TilePosX) && (m_TilePosY == eObject.m_TilePosY);

		

	} // end PositionEquals(EdithObject eObject)

	

	/**

	 * Test whether this objects position is the same the given position

	 * @param targetX, targetY, targetZ position to check against

	 * @return true if the object is at the position

	 */

	public boolean PositionEquals(int targetX, int targetY, int targetZ)

	{

		return (m_TilePosX == targetX) && (m_TilePosY == targetY);

		

	} // end PositionEquals(int targetX, int targetY, int targetZ)

	

	/**

	 * Squared distance of this object from another

	 * @param eObject Other object

	 * @return Square of distance apart in cells

	 */

	public int SquaredDistanceFrom(EdithObject eObject)

	{

		int dx = m_TilePosX - eObject.m_TilePosX;

		int dist = dx * dx;

		dx = m_TilePosY - eObject.m_TilePosY;

		dist += dx * dx;

		return dist;

		

	} // end SquaredDistanceFrom

	

	/**

	 * Test whether this EdithObject lies on one of the four tiles next to eObject

	 * 

	 * @param eObject Target EdithObject

	 * @return true if this EdithObject lies exactly one square north, south, east or west of eObject

	 */

	public boolean IsNextTo(EdithObject eObject)

	{

		if (m_TilePosY == eObject.m_TilePosY)

		{

			return ((m_TilePosX == eObject.m_TilePosX - 1) || (m_TilePosX == eObject.m_TilePosX + 1));

		}

		else if (m_TilePosX == eObject.m_TilePosX)

		{

			return ((m_TilePosY == eObject.m_TilePosY - 1) || (m_TilePosY == eObject.m_TilePosY + 1));

		}

		else

		{

			return false;

		}

		

	} // end IsNextTo

	

	/**

	 * @return	Vector representing direction

	 */

	public int[] GetDirectionVector()

	{

		int[] facing = new int[2];

		

		switch (m_orientation)

		{

		case Facing.NW:

			facing[0] = -1;

			facing[1] = 0;

			break;

			

		case Facing.NE:

			facing[0] = 0;

			facing[1] = -1;

			break;

			

		case Facing.SW:

			facing[0] = 0;

			facing[1] = 1;

			break;

			

		case Facing.SE:

		default:

			facing[0] = 1;

			facing[1] = 0;

			break;

		}

		

		return facing;

		

	} // end GetDirectionVector

	

	/** Get number of container slots */

	public int GetNumContainerSlots()

	{

		return m_ContSlotXOffsets.length;

	}

	

	/**

	 * Get offset of container slot from position of object

	 * 

	 * @param index		Slot number

	 * @return			Array {x, y} of offsets

	 */

	public int[] GetContainerSlotOffset(int index)

	{

		int[] result = new int[2];

		result[0] = m_ContSlotXOffsets[index];

		result[1] = m_ContSlotYOffsets[index];

		

		IsoMap.RotatePosInPlace(result, m_orientation);

		

		return result;

	}

	

	/** Get number of routing slots */

	public int GetNumRoutingSlots()

	{

		return m_RoutingSlotXOffsets.length;

	}

	

	/** Get routing slot container indices */

	public int GetRoutingSlotContainer(int routingSlot)

	{

		return m_RoutingSlotContIndices[routingSlot];

	}

	

	/**

	 * Get offset of routing slot from position of object

	 * 

	 * @param index		Slot number

	 * @return			Array {x, y} of offsets

	 */

	public int[] GetRoutingSlotOffset(int index)

	{

		int[] result = new int[2];

		

		result[0] = m_RoutingSlotXOffsets[index];

		result[1] = m_RoutingSlotYOffsets[index];

		

		IsoMap.RotatePosInPlace(result, m_orientation);

		

		return result;

	}

	

	public int GetFreeRoutingSlotLinkedToContainerSlot(int contIndex, int InstanceId)

	{

		boolean found = false;

		int slotNum = 0;

		int numRoutingSlots = GetNumRoutingSlots();

		

		// first see if the object is already in a routing slot for this container slot

		while (!found && 

			   (slotNum < numRoutingSlots) )

		{			

			// If linked to container

			if (m_RoutingSlotContIndices[slotNum] == contIndex)

			{

				// if free

				// (we can pass a null to CanGo, since we won't try to obtain

				// a routing slot that we're standing on)

				int[] pos = GetRoutingSlotOffset(slotNum);

				

				// add the object position to the offset

				pos[0] += m_TilePosX;

				pos[1] += m_TilePosY;

				

				if(House.s_LotSingleton.IsSpaceInWorld(pos[0], pos[1]))

				{

					// if the tile is contains the target object instance id, we've found the routing slot

					EdithObject eObject= House.s_LotSingleton.IsItem(pos[0], pos[1]);

					if ( (eObject != null) &&

						 (eObject.m_InstanceId == InstanceId) )

					{

						found = true;	

					}	

				}

			}

			

			if (!found)

			{

				slotNum++;

			}

		}

		

		if (!found)

		{

			slotNum = 0;

		}

		

		// if the object wasn't in a routing slot already, find one if there is one available

		while (!found && 

			   (slotNum < numRoutingSlots) )

		{			

			// If linked to container

			if (m_RoutingSlotContIndices[slotNum] == contIndex)

			{

				// if free

				// (we can pass a null to CanGo, since we won't try to obtain

				// a routing slot that we're standing on)

				int[] pos = GetRoutingSlotOffset(slotNum);

				

				// add the object position to the offset

				pos[0] += m_TilePosX;

				pos[1] += m_TilePosY;

				

				// if the tile is free, we've found the routing slot				

				if (!House.s_LotSingleton.IsSpaceOccupied(pos[0], pos[1]))

				{

					found = true;	

				}								

			}



			if (!found)

			{

				slotNum++;

			}

		}

		

		return found ? slotNum : INVALID_SLOT_ID;

	}

	

	/**

	 * Set the static variable flag using the given flag mask value

	 * 

	 * @param flagToSet - mask value of the flag to set

	 */

	public void SetStaticVarFlag(int flagToSet)

	{

		m_staticScriptVars[StaticVariables.STAT_FLAGS] |= flagToSet;

		

	} // end SetStaticVarFlag

	

	/**

	 * Clears the static variable flag using the given flag mask value

	 * 

	 * @param flagToReset - mask value of the flag to set

	 */

	public void ResetStaticVarFlag(int flagToReset)

	{

		m_staticScriptVars[StaticVariables.STAT_FLAGS] &= ~(flagToReset);

		

	} // end ResetStaticVarFlag

	

	/**

	 * Checks the static variable flag using the given flag mask value

	 * 

	 * @param flagToCheck - mask value of the flag to set

	 */

	public boolean CheckStaticVarFlag(int flagToCheck)

	{

		return( (m_staticScriptVars[StaticVariables.STAT_FLAGS] & flagToCheck) != 0);

		

	} // end CheckStaticVarFlag

	

	/**

	 * Finds a route from current position to target

	 * Caches route to enable subsequent calls to GetRouteDirection

	 * 

	 * @param	targetX		Target position

	 * @param	targetY		Target position 

	 */

	public void CalculateRoute(int targetX, int targetY)

	{		

		// Make sure that target position is free

		int[] freePos = House.s_LotSingleton.FindNearestEmptyCell(targetX, targetY, this);

		

		// Determine facing direction

		int[] facing = new int[] { 1, 0 };

		IsoMap.RotatePosInPlace(facing, m_orientation);

		

		// Create search start and end nodes for search

		Node startNode = Node.CreateBoardNode(null, m_TilePosX, m_TilePosY, true, facing[0], facing[1]);

		Node endNode = Node.CreateBoardNode(null, freePos[0], freePos[1], false, 1, 0);

		

		// Do search

		Search pathSearch = new Search(startNode, endNode, 180);



		// Store results

		// Recognised fact that a one node path is no path at all

		if (pathSearch.hasPath())

		{

			m_pathSearch = pathSearch;

		}

		else

		{

			m_pathSearch = null;

		}

	}

	

	/**

	 * Return true if a path has been calculated

	 * @return

	 */

	public boolean HasPath()

	{

		return (m_pathSearch != null);

	}

	

	public void ClearPath()

	{

		// Clear calculated path

		m_pathSearch = null;

	}

	

	/**

	 * Gets next direction to move in for route

	 * @return	deltax, deltay array

	 * @throws Exception	If no route calculated, or no route possible

	 */

	public int[] GetNextRouteDirection() throws Exception

	{

		int[] result;

		boolean pathComplete;	// True if path finished

		Node checkNode;	// Current code position

		Node pathNode;		// A node in the path

		

		// Throw exception on no path

		if (m_pathSearch == null)

		{

			throw new Exception("No path on call to GetNextRouteDirection");

		}

		

		// Find current position in path

		// Direction doesn't matter for the comparison.

		checkNode = Node.CreateBoardNode(null, m_TilePosX, m_TilePosY, false, 1, 0);

		if (checkNode.equals((Node)m_pathSearch.PeekNextStep()))

		{

			m_pathSearch.PopNextStep();

		}

		pathComplete = !m_pathSearch.hasPath();

		

		// If we have finished then throw exception -- no more

		if (pathComplete)

		{

			throw new Exception("Path complete");

		}

		

		// Determine facing direction by delta

		pathNode = (Node)m_pathSearch.PeekNextStep();

		int x = pathNode.getX();

		int y = pathNode.getY();

		

		// If this position is occupied then we have to quit.

		if (House.s_LotSingleton.IsSpaceOccupied(x, y))

			throw new Exception("Failed due to moving obstacle");

		

//START:getnextroutedirection#

		result = new int[] { pathNode.getX(), pathNode.getY() };

//END

		result[0] -= m_TilePosX;

		result[1] -= m_TilePosY;

		

		return result;

	}

	

	/**

	 * Main tick function for all objects

	 * @param int time - time elapsed in milliseconds since last time this function was run

	 */

	public void Tick(int millis)

	{

		//Debug.println("Tick: GUID: " + m_GuId + ", Instance ID: " + m_InstanceId, //Debug.CHAN_EDITH_OBJECT);

		

		// Update animation

		 if (HasAppearance())

		{

			// Pause objects whilst dialogues are active

//			if (Dialogue.s_dialogueActive)

//				return;

			

//			//Debug.assertPrintln(m_CurAnim != null, "no animation data on GUID: " + m_GuId);

			// Dont update animation if there is a dialogue active

			if(	m_CurAnim != null /*&&

				!((m_InstanceId == ScriptVM.s_playerInstanceID) && Dialogue.s_dialogueActive)*/ )

			{

				m_CurAnim = m_CurAnim.Update(millis);

			}

			

			if (m_CloudAnimation != null)

			{

				m_CloudAnimation = m_CloudAnimation.Update(millis);

				

				//Debug.assertPrintln(m_CloudContentsAnimation != null,

//								"Cloud contents destroyed before cloud");

				m_CloudContentsAnimation.Update(millis);

			}			

			

			if (m_MosaicAnimation != null)

			{

				m_MosaicAnimation.Update(millis);

			}



			// Update representation in renderer

			UpdateBillboards();

			

			// Update effects

			UpdateEffectsBillBoards();

			

		}

		

		// update the millis

		// Advance the effect counter by the time used in the game loop

		m_effectCounter += millis;

		m_timeSinceLastTick += millis;

		

	} // end Tick

	

	/** 

	 * Determines if an effect is due to occur.

	 * @param	effectDelay	Time until effect occurs. 

	 * */

	public boolean IsEffectDue(int effectDelay)

	{

		return (m_effectCounter >= effectDelay);

		

	} // end IsEffectDue

	

	/** 

	 * Determines the percentage progress until the next effect tick

	 * @param effectDelay	Time until effect occurs

	 * @return fixed point progress representation

	 */

	public int GetEffectProgress(int effectDelay)

	{

		return (((m_effectCounter % effectDelay) * Utility.FIXED_ONE) / effectDelay);

		

	} // end GetEffectProgress

	

	/** Tells the object that we have handled a due effect */

	public void ResetEffect()

	{

		// We zero the counter (to put equal time spacing between

		// the effect, regardless of delay in handling it)

		m_effectCounter = 0;

		

	} // end ResetEffect



	/**

	 * Checks to see if the object is in use by looking at the container slots.

	 * If they are all full, it is in use, if not, it sets the empty slot as the current slot

	 * 

	 */

	public boolean IsObjectInUse(int targetInstanceId) 

	{

		boolean inUse = true;

		

		if(this.HasAppearance())

		{

			for (int i = 0; i < GetNumContainerSlots(); i++)

			{

				if ( (m_containerSlots[i] == INVALID_INSTANCE_ID) ||

					 (m_containerSlots[i] == targetInstanceId) )

				{

					inUse = false;

					break;

				}

			}

		}

		else

		{

			inUse = false;

		}

		

		return(inUse);

		

	} // end IsObjectInUse

	

	/**

	 * Check if object has an appearance.

	 * @return

	 */

	public boolean HasAppearance()

	{

		return /*m_MenuObject == null &&*/ m_AppearanceGuId != INVALID_GUID_ID;

		

	} // end IsObjectMenu



	/**

	 * Checks to see if the given container slot is in use

	 * 

	 * @param containerSlotToCheck

	 * @return

	 */

	public boolean IsContainerSlotInUse(int containerSlotToCheck, int targetInstanceId) 

	{

		boolean inUse = true;

		

		if ( (m_containerSlots[containerSlotToCheck] == INVALID_INSTANCE_ID) ||

			 (m_containerSlots[containerSlotToCheck] == targetInstanceId) )

		{

			inUse = false;

		}

		

		return(inUse);

		

	} // end IsContinerSlotInUse

	

	/**

	 * Gets the container slot to be used by the target if one is available,

	 * or it returns false.

	 * 

	 * @param targetObject

	 * @return

	 */

	public boolean GetContainerSlotToUse(EdithObject targetObject) 

	{

		boolean containerSlotAvailable = false;

		

		for (int i = 0; i <  GetNumContainerSlots(); i++)

		{

			if ( (m_containerSlots[i] == INVALID_INSTANCE_ID) ||

				 (m_containerSlots[i] == targetObject.m_InstanceId) )

			{

				targetObject.m_staticScriptVars[StaticVariables.STAT_CURRENT_CONTAINER_SLOT] = i;

				containerSlotAvailable = true;

				break;

			}

		}

		

		return(containerSlotAvailable);

		

	} // end GetContainerSlotToUse	

	

	/**

	 * Gets the container slot to be used by the target if one is available,

	 * or it returns false.

	 * 

	 * @param targetObject

	 * @return

	 */

	public int GetContainerThisObjectIsUsing(int targetObjectInstanceId) 

	{

		int containerSlotUsedByObject = INVALID_SLOT_ID;

		

		for (int i = 0; i < GetNumContainerSlots(); i++)

		{

			if (m_containerSlots[i] == targetObjectInstanceId)

			{

				containerSlotUsedByObject = i;

				break;

			}

		}

		

		return(containerSlotUsedByObject);

		

	} // end GetContainerSlotToUse

	

	/**

	 * Clears the object container slot, so it can be used again

	 * 

	 * @param targetInstanceId

	 */

	public void ClearContainerSlot(int targetInstanceId) 

	{

		int numContainerSlots = GetNumContainerSlots();

		

		for (int i = 0; i <  numContainerSlots; i++)

		{			

			if (m_containerSlots[i] == targetInstanceId)

			{

				m_containerSlots[i] = INVALID_INSTANCE_ID;

				m_staticScriptVars[StaticVariables.STAT_CURRENT_CONTAINER_SLOT] = INVALID_INSTANCE_ID;

				break;

			}

		}

		

	} // end ClearContainerSlot

	

	/**

	 * Gets the routing slot associated with the container slot being used by the target.

	 * It assumes that a valid container slot was already found and stored in the target's static vars.

	 * 

	 * @param targetObject

	 * @return

	 */

	public boolean GetRoutingSlotToUse(EdithObject targetObject)

	{		

		int containerSlotIndex;		// Container slot to use on target object

		int routingSlotIndex;		// Routing slot to use on target object

				

		// Get container index from target object

		containerSlotIndex = targetObject.m_staticScriptVars[StaticVariables.STAT_CURRENT_CONTAINER_SLOT];

			

		// Determine which routing slot to use in this object

		routingSlotIndex = GetFreeRoutingSlotLinkedToContainerSlot(containerSlotIndex, targetObject.m_InstanceId);

		

		// Store into target object variable

		targetObject.m_staticScriptVars[StaticVariables.STAT_CURRENT_ROUTING_SLOT] = routingSlotIndex;

		

		return (routingSlotIndex != INVALID_SLOT_ID);

		

	} // end GetContainerSlotToUse	

	

	public void Refresh()

	{

		// We need to redraw on both to ensure that the image is clean.

		m_Engine.removeBillboard(m_Billboard, true);

		m_Engine.addBillboard(m_Billboard, true);

	}



	/**

	 * Adds the object to the isometric map

	 * @param mapEngine	Engine to add to

	 */

	public void AddToMap(Engine mapEngine)

	{

		m_Engine = mapEngine;

		mapEngine.addBillboard(m_Billboard, true);

	}

	

	/**

	 * Removes the object from the isometric map

	 * @param mapEngine	Engine to remove from

	 */

	public void RemoveFromMap(Engine mapEngine)

	{

		m_Engine = null;

		mapEngine.removeBillboard(m_Billboard, true);

	}

	

	public void DrawBillboard(Graphics gfx, int x, int y)

	{

		Billboard b = m_Billboard;

//		/*Billboard*/Image image = b.image;



        Utility.drawRegion(gfx, b.image, b.left, b.top, b.width, b.height,

			(b.mirror ? Sprite.TRANS_MIRROR : Sprite.TRANS_NONE), x, y,

			Graphics.TOP | Graphics.LEFT );

        

	}

	

	// Update representation in renderer

	// Public so that it can be updated independently of Tick()

	public void UpdateBillboards()

	{				

		boolean drawn = false;

		

		// Render animation to bill board

		if (m_CurAnim != null)

		{

			// Set appearance

			drawn = m_CurAnim.DrawToBillBoard(m_Billboard, m_RecalculateDrawPos);

		}

		

		// Recalculate bill board position only if necessary

		if (m_RecalculateDrawPos)

		{

			int[] size = GetDimensions();

			int[] topPos = GetTopPos();



			// now topPos is in world pixel space

			topPos = Utility.GridToWorldCoords(topPos[0], topPos[1]);

		

			// The engine requires that we align our billboard to the bottom

			// of the bottommost tile in the image.

			// Calculate the offset from the top of the tile to the bottom

			// Remove a bias to be sure it's counted as within the correct tile.



			int[] offset = Utility.GridToWorldCoords(m_TilePosX + size[0], m_TilePosY + size[1]);

			m_DrawOffsetX = offset[0] - topPos[0];

			m_DrawOffsetY = offset[1] - (topPos[1] + BIAS);



			// We subtract half tile height from y, since the hotspot

			// stored is relative to the top of the topmost tile (not

			// halfway down it, as returned by GridToWorldCoords()

			m_DrawPosX = topPos[0] + m_DrawOffsetX;

			m_DrawPosY = topPos[1] + m_DrawOffsetY - SKUConsts.TILE_HALF_HEIGHT;



			// Flag that there is no need to recalculate

			m_RecalculateDrawPos = false;

			//Debug.println("UpdateBillboards: " + GetXPos() + " " + GetYPos(), //Debug.CHAN_EDITH_OBJECT);

		}

		

		// Set position of bill board

		if (m_CurAnim != null)

		{			

			// Set position of bill board

			m_Billboard.x = m_DrawPosX;

			m_Billboard.y = m_DrawPosY;

			

			// hack the player when lying down to have a different offset so he will sort above other objects

			// NOTE: Only the player would have these values!

			if ( (m_CurAnim.m_AnimId == SimObject.ANIM_ID_LAY_DOWN) ||

				 (m_CurAnim.m_AnimId == SimObject.ANIM_ID_LAY_DOWN_TO_STAND) )

			{

				m_Billboard.y += SORT_BIAS;

			}

	

			// Increase hotspot by offset

			// (it is currently relative to top of object grid)

			if (drawn)

			{

				m_Billboard.anchor_x += m_DrawOffsetX;

				m_Billboard.anchor_y += m_DrawOffsetY;

				

				// hack the player when lying down to have a different offset so he will sort above other objects

				// NOTE: Only the player would have these values!

				if ( (m_CurAnim.m_AnimId == SimObject.ANIM_ID_LAY_DOWN) ||

					 (m_CurAnim.m_AnimId == SimObject.ANIM_ID_LAY_DOWN_TO_STAND) )

				{

					m_Billboard.anchor_y += SORT_BIAS;

				}

			}

		}

		else // Animation is null, so set bill board to far far away

		{

			// Move billboard from map

			m_Billboard.x = Integer.MAX_VALUE;

			m_Billboard.y = Integer.MAX_VALUE;

		}

	}



	/** 

	 * Update bill boards for effects

	 * Seperate, so can be done after update of subclassed SimObject.

	 */

	private void UpdateEffectsBillBoards()

	{

	

		// Update bill boards that display animation effects

		if (m_CloudBillBoard != null)

		{			

			//Debug.assertPrintln(m_CloudAnimation != null, "Error: Cloud animation not loaded");

			//Debug.assertPrintln(m_CloudContentsAnimation != null, "Cloud billboard initialised without cloud contents");

			

			boolean cloudDrawn = m_CloudAnimation.DrawToBillBoard(m_CloudBillBoard, false);

			boolean cloudContentsDrawn = m_CloudContentsAnimation.DrawToBillBoard(m_CloudContentsBillBoard, false);



			// Add only half draw offset, so that it goes halfway across object

			int halfDrawOffsetX = m_DrawOffsetX >> 1;

			int halfDrawOffsetY = m_DrawOffsetY >> 1;

			

			m_CloudBillBoard.x = m_DrawPosX + halfDrawOffsetX;

			m_CloudBillBoard.y = m_DrawPosY + halfDrawOffsetY + BIAS + SORT_BIAS;



			m_CloudContentsBillBoard.x = m_DrawPosX + halfDrawOffsetX;

			m_CloudContentsBillBoard.y = m_DrawPosY + halfDrawOffsetY + BIAS + SORT_BIAS;

			

			if (cloudDrawn)

			{

				m_CloudBillBoard.anchor_x += m_DrawOffsetX;

				m_CloudBillBoard.anchor_y += m_DrawOffsetY + CLOUD_HEIGHT + SORT_BIAS;

			}

			if (cloudContentsDrawn)

			{

				m_CloudContentsBillBoard.anchor_x += m_DrawOffsetX;

				m_CloudContentsBillBoard.anchor_y += m_DrawOffsetY + CLOUD_HEIGHT + SORT_BIAS;

			}

		}

		

		if (m_MosaicAnimation != null)

		{

			// If on map

			UpdateMosaic();

		}

	}



/**

	 * Controls the animation

	 * @param animId		Animation ID to run (only used if second parameter is true)

	 * @param startNotStop	True to start/unpause, false to pause/stop

	 * @param effectId		Enumerated type from enums.AnimationEffect

	 * @return	True if the animation is not paused

	 */

	public boolean ControlAnimation(int animId, boolean startNotStop, int effectId)

	{

		boolean isPausedOrFinished;	// True if finished

		

		// HACK for player to disappear during cloud animation

		if ( (effectId == AnimationEffect.CLOUD_BATHING) ||

			 (effectId == AnimationEffect.CLOUD_EXERCISING) )

		{

			animId = NULL_ANIMATION;

		}



		// If we want to stop the animation, do it no matter which animation ID

		// is passed in

		if (!startNotStop)

		{

			//Debug.assertPrintln(m_CurAnim != null, "Stopped a null animation");

			m_CurAnim.Stop();

		}

		else	// Else we want to start/continue the animation

		{

			// Are we currently playing the identified animation?

			if ((m_CurAnim != null) && (m_CurAnim.GetId() == animId))

			{			

				// Resume

				m_CurAnim.Start();

			}

			else	// We are not running the identified animation

			{

				// If this is the null animation

				if (animId == NULL_ANIMATION)

				{

					m_CurAnim = null;

				}

				else	// Normal animation

				{

					// Load animation

					m_CurAnim = Utility.getAnimation(/*m_GuId, */m_InstanceId, m_AnimData2, animId, m_Symmetry, m_PaletteGuId);

					

					// Flag that orientation changed

					m_LastOrientation = -1;

					SetFacing(m_orientation);

				}

			}

			

			// Start effect

//			EnableAnimationEffect(effectId);

			

			EdithObject cloudTarget = this;

			EdithObject simObject = this;

			

			// Run any animation effects on target object

			if (effectId == AnimationEffect.DISABLE || effectId == AnimationEffect.NONE)

			{					

				cloudTarget.SetCloudEffect(false, 

					effectId - AnimationEffect.FIRST_CLOUD_EFFECT);

				simObject.SetMosaicEffect(false);

			}

			else if ((effectId >= AnimationEffect.FIRST_CLOUD_EFFECT) &&

					(effectId < AnimationEffect.FIRST_CLOUD_EFFECT + AnimationEffect.NUM_CLOUD_EFFECT_TYPES))

			{

				

				cloudTarget.SetCloudEffect(true, 

							effectId - AnimationEffect.FIRST_CLOUD_EFFECT);

				

			}

			else if (effectId == AnimationEffect.MOSAIC)

			{

				simObject.SetMosaicEffect(true);

			}

		}

		

		isPausedOrFinished = (m_CurAnim == null) || !m_CurAnim.GetIsRunning();

		

		if (isPausedOrFinished)

		{

//			DisableAnimationEffect(effectId);

			

//			EdithObject cloudTarget = this;

//			EdithObject simObject = this;

//			

//			cloudTarget.SetCloudEffect(false, 

//				effectId - AnimationEffect.FIRST_CLOUD_EFFECT);

//			simObject.SetMosaicEffect(false);

		}

		

		return !isPausedOrFinished;

		

	} // end ControlAnimation

	

	/**

	 * Check if animation is still running

	 * 

	 * @return	True if running

	 */

	public boolean GetIsAnimRunning()

	{

		return (m_CurAnim != null) && m_CurAnim.GetIsRunning();

		

	} // end GetIsAnimRunning



	public void SetCloudEffect(boolean isEnabled, int contentsType)

	{

		// If cloud needs switching on

		if (isEnabled && (m_CloudBillBoard == null))

		{

			// Make sure cloud frames are loaded			

			//Debug.assertPrintln(s_CloudStrip != null, "Error: Cloud strip not loaded.");

			//Debug.assertPrintln(s_CloudContentsStrips != null, "Error: Cloud contents strip not loaded.");

		

			// Create a bill board to display cloud			

			m_CloudBillBoard = new Billboard(true); 

			//Debug.println("Created cloud billboard " + m_CloudBillBoard.toString(), //Debug.CHAN_CORE);

			m_Engine.addBillboard(m_CloudBillBoard, false);

			

			// Create a bill board to display cloud contents

			m_CloudContentsBillBoard = new Billboard(true);

			//Debug.println("Created cloud contents billboard " + m_CloudContentsBillBoard.toString(), //Debug.CHAN_CORE);			

			m_Engine.addBillboard(m_CloudContentsBillBoard, false);

		

			// Create animation of cloud

			m_CloudAnimation = Utility.GetAnimation(/*m_GuId, */m_InstanceId,

							s_CloudStrip, CLOUD_ANIM_DELAY, true, 0);

			

			// Create animation of cloud contents

			m_CloudContentsAnimation = Utility.GetAnimation(/*m_GuId, */m_InstanceId,

					s_CloudContentsStrips[contentsType], CLOUD_ANIM_DELAY, true, 0);



			// Store type of contents

//			m_CloudContentsType = contentsType;

		}

		// Else if cloud needs switching off

		else if (!isEnabled && (m_CloudBillBoard != null))

		{

			// Disable cloud bill board and contents bill board

			m_Engine.removeBillboard(m_CloudBillBoard, false);

			m_Engine.removeBillboard(m_CloudContentsBillBoard, false);

			

			// Destroy cloud and animation and cloud contents bill board

			m_CloudBillBoard = null;

			m_CloudAnimation = null;

			m_CloudContentsAnimation = null;

			m_CloudContentsBillBoard = null;

		}

	}

	

	public void SetMosaicEffect(boolean enableNotDisable)

	{

		// Don't enable if not on map

		if (m_Engine == null)

		{

			enableNotDisable = false;

		}

		

		// If enabled

		if (enableNotDisable)

		{

			if (m_MosaicAnimation == null)

			{

				//Debug.println("started mosaic effect", //Debug.CHAN_CORE);

				

				// Make sure mosaic images are loaded

				//Debug.assertPrintln(s_MosaicStrip != null,

//								"Mosaic image strip not loaded");

				

				// Create new vector for mosaic bill boards pool

				m_MosaicBillBoardPool = new Vector();

				

				// Create mosaic aniamtion

				m_MosaicAnimation = Utility.GetAnimation(/*m_ScriptGuId,*/ m_InstanceId,

						s_MosaicStrip, MOSAIC_ANIM_DELAY, true, 0);							

			}

		}

		else	// Is disabled

		{			

			if (m_MosaicAnimation != null)

			{

//				if (s_debugBillboards) //Debug.println("killed mosaic effect", //Debug.CHAN_CORE);

				

				// Kill all mosaic billboards

				Enumeration boards = m_MosaicBillBoardPool.elements();

				while (boards.hasMoreElements())

				{

					m_Engine.removeBillboard((Billboard)boards.nextElement(), false);

				}

				m_MosaicBillBoardPool = null;

				

				// Kill mosaic animation

				m_MosaicAnimation = null;

			}

		}

	}

	

	public boolean IsCloudAnimationPlaying(int id)

	{

		return m_CloudAnimation != null && m_CloudContentsType == id-AnimationEffect.FIRST_CLOUD_EFFECT;

	}

	

	/*

	 * Methods for ItemObjects which no longer exist

	 */

	

	/**

	 * Finds the centre screen position of the item in its grid cells 

	 * 

	 * @param item	Item to get centre position

	 */

	public int[] GetCentreWorldPos(EdithObject item)

	{

		// get the centre position of the main tile

		int[] itemScreenPos = Utility.GridToWorldCoords(item.GetXPos(), item.GetYPos());

		

		// TODO  Perhaps this should be reinstated?

//		// if the item has more than one tile, then update the position

//		// by averaging the m_positions of the tiles it is on

//		if (item.m_positions.size() > 1)

//		{

//			int sumX = 0;

//			int sumY = 0;

//						

//			Enumeration curPos = m_positions.elements();

//			

//			while (curPos.hasMoreElements() )

//			{

//				int[] pos = (int[])curPos.nextElement();

//				int[] tempScreenPos = IsoMap.GridToWorldCoords(pos[0], pos[1]);

//				sumX += tempScreenPos[0];

//				sumY += tempScreenPos[1];

//			}

//			

//			itemScreenPos[0] = (sumX / item.m_positions.size() );

//			itemScreenPos[1] = (sumY / item.m_positions.size() );

//		}

		

		return (itemScreenPos);

		

	} // end GetCentreScreenPos

	

	/**

	 * Read a load of ItemObjects from a file

	 * 

	 * @param fileName	File containing objects

	 * @return Array of objects

	 */

	public static Vector ReadFromFile(byte[] data)

	{

		Vector newObjects = new Vector();

		

		// Read stream fully into StringBuffer

		try

		{

//			data = Utility.GetGlobalResource(guid);

			//Debug.println("Type: Object placement", //Debug.CHAN_GLOBAL_RESOURCE);

			// Spawn objects

			ReadFromString(new String(data), newObjects);

		}

		catch (Exception ex)

		{

			//Debug.assertPrintln(false, "ItemObject:ReadFromFile(): " + ex.getMessage());

		}		

		

		return newObjects;

		

	} // end ReadFromFile

		

	/** Read a load of ItemObjects from a String

	 * 

	 * @param	data		Text to create from

	 * @param	objects	Vector to add objects to 	

	 */

	private static void ReadFromString(String data, Vector objects)

	{

		// Parse info

		int position = 0;							// Position at start of current line

		Vector args = new Vector();					// Arguments for command

		StringBuffer curArg = new StringBuffer();	// Current argument

		char curChar;								// Current character being read

		int lineNum = 1;



		// Current ItemObject info

		EdithObject curItem = null;

		

		// While still lines to edit

		while (position < data.length())

		{

			// Keep adding arguments until end of line

			args.removeAllElements();

			

			while ((position < data.length()) && ((curChar = data.charAt(position)) != '\n') && (curChar  != '\r'))

			{

				// Increase parse position

				position++;

				

				// Read argument until space encountered

				if (curChar != ' ')

				{

					curArg.append(curChar);

				}

				else

				{

					// Add and then empty current argument

					args.addElement(curArg.toString());

					curArg.delete(0, curArg.length());

				}

			}

			

			// Append final argument

			if (curArg.length() != 0)

			{

				args.addElement(curArg.toString());

				curArg.delete(0, curArg.length());

			}

			

			// Switch on first argument

			int argNum = 0;

			String firstArg = (args.size() > 0) ? ((String)args.elementAt(argNum++)) : "";

			

			try

			{

			

				if (firstArg.equals("NEWOBJECT"))

				{

					// Add any existing item to objects list

					if (curItem != null)

					{

						objects.addElement(curItem);

					}

					

					// Create new

				    int itemId = Integer.valueOf((String)args.elementAt(argNum++)).intValue();

					int appearanceId = SimsApp.s_MIDlet.ITEM_APP_DATA[(itemId - WorldData.ITEM_FRIDGE)];

					int x = Integer.valueOf((String)args.elementAt(argNum++)).intValue();

					int y = Integer.valueOf((String)args.elementAt(argNum++)).intValue();

					int stringId = (StringId.OBJECT_NAME_0 + (itemId - WorldData.ITEM_FRIDGE) == StringId.MONDAY ? 

									StringId.FRONT_DOOR : StringId.OBJECT_NAME_0 + (itemId - WorldData.ITEM_FRIDGE) );

					

					// Interactive items.

					if ((itemId >= WorldData.FIRST_ITEM) &&

						(itemId <= WorldData.LAST_ITEM))

					{

						if(itemId == WorldData.ITEM_COFFEE_MACHINE || itemId == WorldData.ITEM_ALARM_CLOCK)

						{

							curItem = new EdithObject(itemId, appearanceId, x, y, false, stringId);

							

							// Ignore facing for 250k.

							if(!SimsApp.s_MIDlet.IsItemAvailable(itemId))

								curItem.SetFacing(0);

							else

								curItem.SetFacing(1);

							

							curItem.UpdateBillboards();

						}

						else if(SimsApp.s_MIDlet.IsItemAvailable(itemId))

						{

							curItem = new EdithObject(itemId, appearanceId, x, y, false, stringId);

							

							// hack to make the chess table face the correct way

							if(itemId == WorldData.ITEM_CHESS_TABLE)

								curItem.SetFacing(2);

							else

								curItem.SetFacing(0);

			

							curItem.UpdateBillboards();

						}

					}

					

					// Non-interactive items.

					if ((itemId >= WorldData.FIRST_GRAPHICAL_ITEM) && 

						(itemId <= WorldData.LAST_GRAPHICAL_ITEM))

					{

						curItem = new EdithObject(itemId, appearanceId, x, y, false, stringId);

						

						// Ignore facing for 250k.

						curItem.SetFacing(0);

						curItem.UpdateBillboards();

					}

				}

			}

			catch (Exception ex)

			{

				//Debug.assertPrintln(false, "Error parsing ItemObjects on line " + lineNum);

				curItem = null;

			}

			

			// Move start past end of line

			position++;

			

			lineNum++;

		}

		

		// Add any remaining object

		if (curItem != null)

		{

			objects.addElement(curItem);

		}

	

	} // end ReadFromString

	

	/** Load appearance data */

	private void LoadAppearanceData(int guid)

	{

        int count;

		// Obtain appearance data block

		byte[] data = null;

		try

		{

			data = Utility.GetGlobalResource(guid);

			 

			//Debug.println("Type: Appearance data", //Debug.CHAN_GLOBAL_RESOURCE);

			

			int dataPos = 0;

			

			// Skip first 4 bytes -- these are the UI icon guid

			dataPos += 4;

			

			// Load symmetry, width and height

			m_Symmetry = Utility.ByteSerializeType(0, data, dataPos++, 1, false);

			m_Width = Utility.ByteSerializeType(0, data, dataPos++, 1, false);

			m_Height = Utility.ByteSerializeType(0, data, dataPos++, 1, false);

			

			// Load container slots

			int numContainer = Utility.ByteSerializeType(0, data, dataPos++, 1, false);

			m_ContSlotXOffsets = new int[numContainer];

			m_ContSlotYOffsets = new int[numContainer];

			m_ContSlotPixelHeights = new int[numContainer];

			for (count = 0; count < numContainer; count++)

			{

				int gridX = data[dataPos++];

				int gridY = data[dataPos++];

				int pixelHeight = Utility.ByteSerializeType(0, data, dataPos++, 1, false);

				

				// Add container slot to object

				m_ContSlotXOffsets[count] = gridX;

				m_ContSlotYOffsets[count] = gridY;

				m_ContSlotPixelHeights[count] = pixelHeight;

			}

			

			// Load routing slots

			int numRouting = Utility.ByteSerializeType(0, data, dataPos++, 1, false);

			m_RoutingSlotXOffsets = new int[numRouting];

			m_RoutingSlotYOffsets = new int[numRouting];

			m_RoutingSlotContIndices = new int[numRouting];

			for (count = 0; count < numRouting; count++)

			{

				int gridX = data[dataPos++];

				int gridY = data[dataPos++];

				int containerIndex = Utility.ByteSerializeType(0, data, dataPos++, 1, false);

				

				// Add routing slot to object

				m_RoutingSlotXOffsets[count] = gridX;

				m_RoutingSlotYOffsets[count] = gridY;

				m_RoutingSlotContIndices[count] = containerIndex;

			}

			

			// Load animations

			int numAnims = Utility.ByteSerializeType(0, data, dataPos++, 1, false);

			m_AnimData2= new byte[numAnims][];

			for (count = 0; count < numAnims; count++)

			{

				int animSize = 0;

				

				// Get number of image strips

				int numImageStrips = Utility.ByteSerializeType(0, data, dataPos + animSize++, 1, false);

				

				// Skip strip guids

				// (each guid stored as 4 bytes)

				animSize += numImageStrips * 4;

				

				// Get number of control frames

				int numControlFrames = Utility.ByteSerializeType(0, data, dataPos + animSize++, 1, false);

				

				// Skip control frames

				// (each control frame stored as 4 bytes)

				animSize += numControlFrames * 4;			

				

				// Copy into new byte array

				m_AnimData2[count] = new byte[animSize];

				System.arraycopy(data, dataPos, m_AnimData2[count], 0, animSize);

				

				dataPos += animSize;

			}

			

			m_PaletteGuId = 0;

			

			if (dataPos < data.length)

			{

				int overridePalettes = Utility.ByteSerializeType(0, data, dataPos++, 1, false);

				if ((overridePalettes & 1)==1)

				{

					m_PaletteGuId = Utility.ByteSerializeType(0, data, dataPos, 4, false);

					dataPos += 4;

				}

			}

		}

		catch (Exception ex)

		{

			// Failed to load

			//Debug.println("EdithObject(" + m_GuId + "):LoadAppearanceData(): " + ex.getMessage(), //Debug.CHAN_GLOBAL_RESOURCE);

		}

	}

	

	private void SetPos(int xPos, int yPos)

	{

		// If changed position

		if ((m_TilePosX != xPos) || (m_TilePosY != yPos))

		{

			m_TilePosX = xPos;

			m_TilePosY = yPos;

			m_RecalculateDrawPos = true;

		}

	}



	/** Create/destroy sufficient mosaic billboards */

	private void UpdateMosaic()

	{

//		Debug.assertPrintln(m_Billboard != null,

//						"No billboard present on mosaic update");

//		Debug.assertPrintln(m_Billboard.image != null,

//						"No billboard image present on mosaic update");

				

		// Determine dimensions to cover		

//START:IGNORE:size#

		int size[] = new int[2];

//END

		size[0] = (m_CurAnim != null) ? m_CurAnim.GetWidth() : 0;

		size[1] = (m_CurAnim != null) ? m_CurAnim.GetHeight() : 0;

		

//START:IGNORE:topLeft#

		int topLeft[] = new int[2];

//END

		topLeft[0] = m_Billboard.x - /*size[0] / 2;*/m_Billboard.anchor_x;

		topLeft[1] = m_Billboard.y - m_Billboard.anchor_y;

		

		// Determine number of required mosaic boards

		int billWidth = s_MosaicStrip.GetWidth(0);

		int billHeight = s_MosaicStrip.GetHeight(0);

		int boardsAcross = 1;//Math.max(1, (size[0] / billWidth));// + 1;

		int boardsDown = 1;//Math.max(1, (size[1] / billHeight));// + 1;

		int boardsNeeded = boardsAcross * boardsDown;

		

		// Adjust top left position so that when drawn, billboards

		// are centred over character

		topLeft[0] = topLeft[0] + (size[0] / 2) - ((billWidth * boardsAcross) / 2);

		topLeft[1] = topLeft[1] + (size[1] / 2) - ((billHeight * boardsDown) / 2);

		

		// Create/destroy to get to correct amount

		while (m_MosaicBillBoardPool.size() < boardsNeeded)

		{

			Billboard newBill = new Billboard(true);

			m_MosaicBillBoardPool.addElement(newBill);

			m_Engine.addBillboard(newBill, false);						

		}

		while (m_MosaicBillBoardPool.size() > boardsNeeded)

		{

			Billboard delBill = (Billboard)m_MosaicBillBoardPool.firstElement();

			m_MosaicBillBoardPool.removeElement(delBill);

			m_Engine.removeBillboard(delBill, false);

		}

		

		// Position each and draw animation to it		

		int yPos = topLeft[1];

		Enumeration boards = m_MosaicBillBoardPool.elements();

		for (int billY = 0; billY < boardsDown; billY++)

		{

			int xPos = topLeft[0];

			for (int billX = 0; billX < boardsAcross; billX++)

			{

//				if (s_debugBillboards) //Debug.println("board at " + xPos + 

//					"," + yPos, //Debug.CHAN_CORE);

				

				Billboard board = (Billboard)boards.nextElement();

				m_MosaicAnimation.DrawToBillBoard(board, true);

				

				// The anchor point (position of board) must be directly below

				// that of the sim to get the z-ordering correct

				board.x = m_Billboard.x;

				board.y = m_Billboard.y + 1;				

				

				// The anchor offset is used to position the bill board on the

				// screen.  It must be increased by the player's anchor location

				// minus the intended position

				board.anchor_x += m_Billboard.x - xPos;

				board.anchor_y += m_Billboard.y - yPos;

				

				xPos += billWidth;				

			}

			yPos += billHeight;

		}

		

		// output number of boards alive

//		if (s_debugBillboards) //Debug.println(m_MosaicBillBoardPool.size() + 

//			" boards alive", //Debug.CHAN_CORE);

	}



	public static final byte BIAS = 2;

	

	/** A value to indicate an invalid slot (returned from methods) */

	public static final int INVALID_SLOT_ID							= -1; 

	

	/** A value to determine that an object instance id is invalid */

	public static final int INVALID_INSTANCE_ID 					= -1;

	

	/** A value to determine that an object guId is invalid */

	public static final int INVALID_GUID_ID 						= -1;

	

	/** Edith Object Flag flags */

	public static final int STATIC_VAR_FLAG_CANCEL_SCRIPT		= (1 << 0);

	public static final int STATIC_VAR_FLAG_WALKING	            = (1 << 1);

	

	/** Edith Object save mask for the flags that will need to be saved */

	public static final int STATIC_VARS_BF_SAVE_MASK 				= (0);



	private static final boolean s_debugBillboards = false;

	

	/** A number to take as meaning "display no animation" when passed to ControlAnimation */

	private static final int NULL_ANIMATION = -1;

	

	/** Guid of the cloud strip */

	private static final int CLOUD_STRIP_GUID = 34858;

	

	/** Guid of mosaic strip */

	private static final int MOSAIC_STRIP_GUID = 34540;

	

	// moved to seperate images

//	/** Guid of the cloud contents strip */

//	private static final int CLOUD_CONTENTS_STRIP_GUID = 33460;

	

	

	private static final int[] CLOUD_CONTENTS_STRIP_GUIDS =

	{

		34850,	//CLOUD_BATHING

		34852,	//CLOUD_COOKING

		34857,	//CLOUD_MUSIC

//		34064,	//CLOUD_CLEANING

//		34855,	//CLOUD_EXTINGUISHER (replaces Exercise GUID in compile)

		34855,	//CLOUD_FIRE

		34853,	//CLOUD_EXERCISING

		34856,	//CLOUD_LOGIC

		34849,	//CLOUD_ARCADE

	};

		

	/** Time between frames in cloud animation */

	private static final int CLOUD_ANIM_DELAY = 500;

	

	/** Height of cloud off ground */

	private static final int CLOUD_HEIGHT = 24;

	

	/** Time between frames in mosaic animation */

	private static final int MOSAIC_ANIM_DELAY = 150;

	private static final int SORT_BIAS = 100;

	

	/** object instance ID counter */

	public static int 		s_objectIDCounter;

	

//	/** Image used to mosaic a sim */

//	private static Image s_MosaicImage;

	

	/** Images used to animate cloud */

	private static ImageStrip s_CloudStrip;

	

	/** Images used to animate cloud */

	private static ImageStrip[] s_CloudContentsStrips;



	/** Images used to animation mosaic */

	private static ImageStrip s_MosaicStrip;

	

	// class variables

	public String 		m_name;

	public int[]		m_staticScriptVars;

	public int[]		m_containerSlots;			/** store the instance ids of the objects that are contained in this object */

	public int 			m_GuId;						/** object type: i.e.: "toliet" */

	public int 			m_InstanceId;				/** actual occurence of the object in the game */

	

	public Billboard	m_Billboard;				/** Used to render to engine */

//	protected Billboard	m_DynamicBillboard;			/** Used to render to engine */

	protected Animation	m_CurAnim;					/** Current animation running */

	protected Engine    m_Engine;

	

	private int			m_AppearanceGuId;			/** Guid of appearance data */

	public int 			m_TilePosX, m_TilePosY;		/** World Tile Position */

	private int 		m_LastOrientation;			/** Facing direction */

	private int 		m_orientation;				/** Facing direction */

	

//	private Point2 		m_targetPos;				/** The target location to walk to */

	private Search 		m_pathSearch;				/** The path found to the target */

//	private int 		m_pathIndex;				/** The index of the current position along the path */

	private int 		m_effectCounter;			/** Count down timer to when the action will have an effect on the world. */





	private int			m_Symmetry;					/** The symmetrical property of the object */

	private int			m_Width;					/** Width in tiles under no rotation */

	private int			m_Height;					/** Height in tiles under no rotation */

	private byte[][]	m_AnimData2;				/** The animation data for each animation in object */

	private int			m_PaletteGuId;				/** The override palette to use (0 for default palette) */

	

	private int			m_timeSinceLastTick = 0; 	/** How long since we last did objects ticks. */ 

	

	private int			m_DrawPosX;					/** Draw position of billboard */

	private int			m_DrawPosY;					/** Draw position of billboard */

	private int			m_DrawOffsetX;				/** Draw offset (to hotspot) */

	private int			m_DrawOffsetY;				/** Draw offset (to hotspot) */	

	private boolean 	m_RecalculateDrawPos;		/** True if we need to recalculate draw pos */

	

	private int[]		m_ContSlotXOffsets;			/** Position of slots */

	private int[]		m_ContSlotYOffsets;			/** Position of slots */

	private int[]		m_ContSlotPixelHeights;		/** Position of slots */

	

	private int[]		m_RoutingSlotXOffsets;		/** Position of slots */

	private int[]		m_RoutingSlotYOffsets;		/** Position of slots */

	private int[]		m_RoutingSlotContIndices;	/** Container index */

	

	private Billboard	m_CloudBillBoard;			/** Billboard used to display cloud */

	private Billboard	m_CloudContentsBillBoard;	/** Billboard used to display cloud contents */

	private Animation	m_CloudAnimation;			/** Animation used to display cloud */

	private Animation	m_CloudContentsAnimation;	/** Animation used to display cloud contents */

	private int			m_CloudContentsType;		/** Type of cloud contents to display */

		

	private Animation	m_MosaicAnimation;				/** Animation of mosaic */

	private Vector		m_MosaicBillBoardPool;			/** Bill boards used to display mosaic */



} // End EdithObject

