package sims2;



/* Copyright Ideaworks3D Ltd. 2005 */

/*

 * Created on 02-Jun-2005

 */







import javax.microedition.lcdui.Graphics;

import javax.microedition.lcdui.Image;

import javax.microedition.lcdui.game.Sprite;



//import com.ideaworks3d.debug.Debug;







/**

 * Handles loading of an ImageStrip data structure

 */

public class ImageStrip

{

	

	/**

	 * Create image strip from Png byte array

	 * 

	 * @param buffer

	 */

    public ImageStrip(byte[] buffer)

    {

    	m_Guid = -5; // this can be anything, we really don't care

    	m_Image = Image.createImage(buffer, 0, buffer.length);

    	m_ImageMetrics = new byte[METRICS_SIZE];

		Utility.ByteSerializeType(0, m_ImageMetrics, 0, 2, true);

		Utility.ByteSerializeType(0, m_ImageMetrics, 2, 2, true);

		Utility.ByteSerializeType(m_Image.getWidth(), m_ImageMetrics, 4, 1, true);

		Utility.ByteSerializeType(m_Image.getHeight(), m_ImageMetrics, 5, 1, true);

		Utility.ByteSerializeType(0, m_ImageMetrics, 6, 2, true);

		Utility.ByteSerializeType(0, m_ImageMetrics, 8, 2, true);

	}

    

	/**

	 * Reads an image strip

	 * @param guid			Guid of image strip to load

	 * @param paletteGuId 	Guid of palette to use with image (0 means don't override default palette)

	 */

	public ImageStrip(int guid, int paletteGuId, boolean cacheImage)

	{

		try

		{

	       	int curCol;

			// Obtain global data for guid

			byte[] data = null;

			

			 data = Utility.GetGlobalResource(guid);	

				//Debug.println("Type: ImageStrip", //Debug.CHAN_GLOBAL_RESOURCE);

			 

			int dataPos = 0;

			

			// Read image

			int imageSize = Utility.ByteSerializeType(0, data, dataPos, 2, false);

			dataPos += 2;



			if (paletteGuId != 0)

			{

				OverwritePalette(data, dataPos, paletteGuId);

			}

			m_Image = Utility.createImage(paletteGuId, guid, data, dataPos, imageSize, cacheImage);

			dataPos += imageSize;	



			// Read positioning data

			int numCols = Utility.ByteSerializeType(0, data, dataPos++, 1, false);

			////Debug.println("new ImageStrip len=" + data.length + " size=" + imageSize + " numCols=" + numCols);

//START:IGNORE:tempData#

			byte tempData[][][] = new byte[numCols][][];

//END

			int xPos = 0;

			int numFrames = 0;

			for (curCol = 0; curCol < numCols; curCol++)

			{

				int width = Utility.ByteSerializeType(0, data, dataPos++, 1, false);

				

				int numImages = Utility.ByteSerializeType(0, data, dataPos++, 1, false);

//START:double array#

				tempData[curCol] = new byte[numImages][METRICS_SIZE];

//END

				

				int yPos = 0;

				

				for (int curImg = 0; curImg < numImages; curImg++)

				{

					int height = Utility.ByteSerializeType(0, data, dataPos++, 1, false);

					int numHotSpots = Utility.ByteSerializeType(0, data, dataPos++, 1, false);

					

					byte[] myData = tempData[curCol][curImg];

					Utility.ByteSerializeType(xPos, myData, 0, 2, true);

					Utility.ByteSerializeType(yPos, myData, 2, 2, true);

					Utility.ByteSerializeType(width, myData, 4, 1, true);

					Utility.ByteSerializeType(height, myData, 5, 1, true);

					for (int hotSpotNum = 0; hotSpotNum < MAX_HOT_SPOTS; hotSpotNum++)

					{		

						byte type = HotSpotTypes.NULL;

						short xHs = 0;

						short yHs = 0;

						if (numHotSpots > hotSpotNum)

						{

							type = data[dataPos++];

							xHs = (short)Utility.ByteSerializeType(0, data, dataPos, 2, false);

							dataPos += 2;						

							yHs = (short)Utility.ByteSerializeType(0, data, dataPos, 2, false);

							dataPos += 2;						

						}

						

						int offset = PRE_HOT_SPOT_DATA_LENGTH + hotSpotNum * HOT_SPOT_DATA_LENGTH;

						myData[offset++] = type;

						Utility.ByteSerializeType(xHs, myData, offset, 2, true);

						offset += 2;

						Utility.ByteSerializeType(yHs, myData, offset, 2, true);

						offset += 2;

					}

					dataPos += Math.max(0, numHotSpots - MAX_HOT_SPOTS) * 3;

					

					yPos += height;

					numFrames++;

				}

				

				// Increase x position in strip

				xPos += width;

			}

			

			// Output to metrics array

			m_ImageMetrics = new byte[numFrames * METRICS_SIZE];

			int offset = 0;

			for (curCol = 0; curCol < numCols; curCol++)

			{

				

				for (int curImg = 0; curImg < tempData[curCol].length; curImg++)

				{

					System.arraycopy(tempData[curCol][curImg], 0, m_ImageMetrics, offset, METRICS_SIZE);

					offset += METRICS_SIZE;

				}

			}

			

			// Store guid

//				if (//Debug.ENABLED)

//				{

				m_Guid = guid;

//				}

		}

		catch (Exception ex)

		{

			// Failed to load

//			//Debug.println("ImageStrip(): " + ex.getMessage(), //Debug.CHAN_GLOBAL_RESOURCE);

			

			// This may happen with deliberately removed animations.

			// We throw here which will be caught in getAnimation.

			throw new RuntimeException();

		}

	}

	

	public static boolean OverwritePalette(byte[] pngData, int pngDataPos, int paletteGuId)

	{

		byte[] palette = null;

		try

		{

			palette = Utility.GetGlobalResource(paletteGuId);

			//Debug.println("Type: Palette", //Debug.CHAN_GLOBAL_RESOURCE);

			//Debug.assertPrintln(palette[0]=='P' && palette[1]=='A' && palette[2]=='L', "invalid palette format on GUID "+paletteGuId);

		}

		catch (Exception ex)

		{

			// Failed to load

			//Debug.println("ImageStrip.OverwritePalette(): " + ex.getMessage());

			return false;

		}

		

		// Size is stored as number of entries minus 1. The size per entry is 3 bytes.

		int size = ((palette[3] & 0xff) + 1) * 3;



		//Debug.println("Palette size: " + size, //Debug.CHAN_GLOBAL_RESOURCE);

		

		// Find palette block in PNG (start looking 33 bytes in since there's that much known header)

		for (int i=pngDataPos+33; i<pngData.length-12; i++)

		{

			if (pngData[i]=='P' && pngData[i+1]=='L' && pngData[i+2]=='T' && pngData[i+3]=='E')

			{

				int oldsize = (pngData[i-1] & 0xff) + ((pngData[i-2] & 0xff) << 8);

				

				if (oldsize != size)	// Don't copy anything if the palette's a different size!

				{

//START:debug print 1#

					//Debug.println("ImageStrip.OverwritePalette: mismatched palette size from palette " + paletteGuId + " (" + size + " bytes, image was "+oldsize+" bytes)");

//END

//					return false;

				}

				

				System.arraycopy(palette, 4, pngData, i+4, Math.min(size, oldsize) );	

				

				// calc crc on pngData from i to oldsize + 4

				// put crc at oldsize + 4

				CRCCalculator crcCalc = new CRCCalculator();

				

				byte[] plteChunk = new byte[4 + oldsize];

				System.arraycopy(pngData, i, plteChunk, 0, plteChunk.length);

				int checksum = crcCalc.crc(plteChunk);



//				byte[] bytes = new byte[4];

//				int pos = 3;

////				while (checksum != 0)

////				{

////					bytes[pos--] = (byte)(checksum & 0xff);

////					checksum >>>= 8;

////				}

				

				int oldchecksum = Utility.ByteSerializeType(0, pngData, i + 4 + oldsize, 4, false);

				//Debug.println("Old checksum: " + Integer.toHexString(oldchecksum) + " New checksum: " + Integer.toHexString(checksum), //Debug.CHAN_CORE);

//				System.arraycopy(bytes, 0, pngData, i + oldsize + 4, 4);

				Utility.ByteSerializeType(checksum, pngData, i + 4 + oldsize, 4, true);

				return true;

			}

		}

		

		// No palette block found

		return false;

	}

	

	/**

	 * @param	frameNum 	Index of frame to get width of

	 * @return	Width of image frame

	 */

	public int GetWidth(int frameNum)

	{

		int offset = frameNum * METRICS_SIZE + 4;

		int width = Utility.ByteSerializeType(0, m_ImageMetrics, offset, 1, false);

		

		return width;

	}

	

	/**

	 * @param	frameNum 	Index of frame to get height of 

	 * @return	Height of image frame

	 */

	public int GetHeight(int frameNum)

	{

		int offset = frameNum * METRICS_SIZE + 5;

		int height = Utility.ByteSerializeType(0, m_ImageMetrics, offset++, 1, false);

		

		return height;

	}

	

	/**

	 * Obtain hot spot offset by type

	 * 

	 * @param frameNum			Frame index to pull hot spot data for

	 * @param type				Type of hot spot to return position of

	 * @return					Position of first hot spot that matches type, or [0, 0] 

	 * 							if type does not exist in hot spots data for current frame

	 */

	public int[] GetHotSpot(int frameNum, int type)

	{

		return GetHotSpot(frameNum, true, type);

	}

	

	/**

	 * Obtain hot spot offset by type

	 * 

	 * @param frameNum			Frame index to pull hot spot data for

	 * @return					Position of first hot spot, or [0, 0] 

	 * 							if all hot spots are null

	 */

	public int[] GetHotSpot(int frameNum)

	{

		return GetHotSpot(frameNum, false, 0);

	}

	

	/** Determine number of frames in animation */

	public int GetNumFrames()

	{

		return m_ImageMetrics.length / METRICS_SIZE;

	}

	

	/**

	 * Render to a graphics context

	 * 

 	 * @param context		Graphics context to render to

	 * @param frameNum		Frame number to render to 

	 * @param xTrg			X position in context to draw to

	 * @param yTrg			Y position in context to draw to

	 * @param useHotSpot	True if image should be anchored at hot spot

	 * @param anchor		Normal graphics anchor

	 */

	public void Draw(Graphics context, int frameNum, int xTrg, int yTrg, boolean useHotSpot, int anchor)

	{

		int offset = frameNum * METRICS_SIZE;

		

		// Assert for a crash occurring on the emulator 15/05/05 - 20:55

//START:debug print 2#

		//Debug.assertPrintln(offset < m_ImageMetrics.length, 

						//	"ImageStrip(" + m_Guid + "):Draw offset " + offset + " >= " + m_ImageMetrics.length);

//END

		

		int xPos = (m_ImageMetrics[offset++] << 8) | (m_ImageMetrics[offset++] & 0xff);

		int yPos = (m_ImageMetrics[offset++] << 8) | (m_ImageMetrics[offset++] & 0xff);

		int width = (m_ImageMetrics[offset++] & 0xff);

		int height = (m_ImageMetrics[offset++] & 0xff);

		

		// Assert that billboard portion of image exists

//		if (Debug.ENABLED)

//		{

//			int imageWidth = m_Image.getWidth();

//			int imageHeight = m_Image.getHeight();

			

//START:debug print 3#

			//Debug.assertPrintln(xPos >= 0, "Draw() guid=" + m_Guid + ",  xPos(" + xPos + ")<0");

			//Debug.assertPrintln(yPos >= 0, "Draw() guid=" + m_Guid + ", yPos(" + yPos + ")<0");

			//Debug.assertPrintln(xPos < m_Image.getWidth(), "Draw() guid=" + m_Guid + ", xPos(" + xPos + ")>=width");

			//Debug.assertPrintln(yPos < m_Image.getHeight(), "Draw() guid=" + m_Guid + ", yPos(" + yPos + ")>=height");

			//Debug.assertPrintln((xPos + width) <= imageWidth, "Draw() guid=" + m_Guid + ", xPos+width(" + (xPos + width) + ")>width(" + imageWidth + ")");

			//Debug.assertPrintln((yPos + height) <= imageHeight, "Draw() guid=" + m_Guid + ", yPos+height(" + (yPos + height) + ")>height(" + imageHeight + ")");

//END

		//}

		

		// Hard code that first hotspot is taken

		if (useHotSpot)

		{

			offset++;

			short hsX = (short)((m_ImageMetrics[offset++] << 8) | (m_ImageMetrics[offset++] & 0xFF));

			short hsY = (short)((m_ImageMetrics[offset++] << 8) | (m_ImageMetrics[offset++] & 0xFF));

			

			xTrg -= (int)hsX;

			yTrg -= (int)hsY;

		}

		

		// Draw image

		context.drawRegion(m_Image, xPos, yPos, width, height, Sprite.TRANS_NONE,

						xTrg, yTrg, anchor);

	}

	

	/**

	 * Make billboard display the image strip

	 * @param billImage		Billboard to display the image strip

	 * @param frameNum		Index of frame to draw

	 * @param isMirrored	True if bill board should be mirrored

	 * @throws ArrayIndexOutOfBoundsException 	If frame number out of range

	 */

	public void DrawToBillBoard(Billboard billboard, int frameNum,

		boolean isMirrored) throws ArrayIndexOutOfBoundsException

	{

		int offset = frameNum * METRICS_SIZE;



		// Assert for a crash occurring on the emulator 15/05/05 - 20:55

		if (offset + 5 >= m_ImageMetrics.length)

		{

			//Debug.println("ImageStrip:DrawToBillBoard offset " + offset

			//	+ " too small for image metrics of length "

			//	+ m_ImageMetrics.length);

			return;

		}



		int xPos = (m_ImageMetrics[offset++] << 8)

			| (m_ImageMetrics[offset++] & 0xff);

		int yPos = (m_ImageMetrics[offset++] << 8)

			| (m_ImageMetrics[offset++] & 0xff);

		int width = (m_ImageMetrics[offset++] & 0xff);

		int height = (m_ImageMetrics[offset++] & 0xff);



		// Assert that billboard portion of image exists

//		if (Debug.ENABLED)

//		{

//			int imageWidth = m_Image.getWidth();

//			int imageHeight = m_Image.getHeight();

//

//			//Debug.assertPrintln(xPos >= 0, "DrawToBillBoard() guid=" + m_Guid

//				+ ",  xPos(" + xPos + ")<0");

//			//Debug.assertPrintln(yPos >= 0, "DrawToBillBoard() guid=" + m_Guid

//				+ ", yPos(" + yPos + ")<0");

//			//Debug.assertPrintln(xPos < m_Image.getWidth(),

//				"DrawToBillBoard() guid=" + m_Guid + ", xPos(" + xPos

//					+ ")>=width");

//			//Debug.assertPrintln(yPos < m_Image.getHeight(),

//				"DrawToBillBoard() guid=" + m_Guid + ", yPos(" + yPos

//					+ ")>=height");

//			//Debug.assertPrintln((xPos + width) <= imageWidth,

//				"DrawToBillBoard() guid=" + m_Guid + ", xPos+width("

//					+ (xPos + width) + ")>width(" + imageWidth + ")");

//			//Debug.assertPrintln((yPos + height) <= imageHeight,

//				"DrawToBillBoard() guid=" + m_Guid + ", yPos+height("

//					+ (yPos + height) + ")>height(" + imageHeight + ")");

//		}

//		else

//		// If not in debug mode, don't draw if it's out of bounds

		{

			boolean valid = (xPos >= 0) && (yPos >= 0)

				&& ((xPos + width) <= m_Image.getWidth())

				&& ((yPos + height) <= m_Image.getHeight());



			if (!valid)

			{

				return;

			}

		}



		// Hard code that first hotspot is taken

		offset++;

		short hsX = (short) ((m_ImageMetrics[offset++] << 8) | (m_ImageMetrics[offset++] & 0xFF));

		short hsY = (short) ((m_ImageMetrics[offset++] << 8) | (m_ImageMetrics[offset++] & 0xFF));



		// Set image

//		if (SKUConsts.FLIPPING_MEMORY_LEAKS && isMirrored)

//		{

//			if (m_MirroredImage == null)

//				CreateMirroredStrip();

//			billboard.image = m_MirroredImage;

//

//			// Set frame to display

//			int image_w = m_MirroredImage.getWidth();

//			billboard.left = image_w - xPos - width;

//

//			// Set anchor point

//			billboard.anchor_x = image_w - hsX;

//

//			// Set mirroring

//			billboard.mirror = false;

//		}

//		else

		{

			billboard.image = m_Image;

			

			// Set frame to display

			billboard.left = xPos;					



			// Set mirroring

			billboard.mirror = isMirrored;

			

			// Set anchor point

			// If mirrored, then reverse x hotspot			

			if (isMirrored)

			{

				billboard.anchor_x = (width - 1) - hsX;

			}

			else

			{

				billboard.anchor_x = hsX;

			}

		}



		// Set frame to display

		billboard.top = yPos;

		billboard.width = width;

		billboard.height = height;



		// Set anchor point

		billboard.anchor_y = hsY;

	}

	

	/**

	 * Obtain hot spot offset by type

	 * 

	 * @param frameNum			Frame index to pull hot spot data for

	 * @param typeImportant		True if type needs to match

	 * @param type				Type of hot spot to return position of

	 * @return					Position of first hot spot that matches type, or [0, 0] 

	 * 							if type does not exist in hot spots data for current frame

	 */

	private int[] GetHotSpot(int frameNum, boolean typeImportant, int type)

	{

		int[] result = new int[2];	// Value to return

		int dataPos;				// Position in metrics data

		boolean found;				// True if hot spot type found

		int hotSpotIndex;			// Current hot spot being processed

		

		// Check validity of arguments

		//Debug.assertPrintln((frameNum >= 0) && (frameNum < GetNumFrames()),

						//"GetHotSpot: Invalid frame number !(0<=" + frameNum 

							//+ "<" + GetNumFrames() + ")");

		

		if (typeImportant)

		{

//START:debug print 6#

			//Debug.assertPrintln((type >= 0) && (type < HotSpotTypes.NUM_TYPES),

						//	"GetHotSpot: Invalid type !(0<=" + type 

						//	+ "<" + HotSpotTypes.NUM_TYPES + ")");

//END

		}

		

		// Skip to correct position in metrics data for this frame

		// Also skip past the non-hot spot data

		dataPos = frameNum * METRICS_SIZE;

		dataPos += PRE_HOT_SPOT_DATA_LENGTH;

		

		// Do search for correct hot spot type		

		found = false;

		hotSpotIndex = 0;

		while (!found && (hotSpotIndex < MAX_HOT_SPOTS))

		{

			int curType = (int)m_ImageMetrics[dataPos];

			

			if ((!typeImportant && (curType != HotSpotTypes.NULL)) ||

				(typeImportant && (type == curType)) )

			{

				found = true;

				dataPos++;

				result[0] = ((m_ImageMetrics[dataPos++] << 8) | (m_ImageMetrics[dataPos++] & 0xFF));

				result[1] = ((m_ImageMetrics[dataPos++] << 8) | (m_ImageMetrics[dataPos++] & 0xFF));

			}

			else

			{

				hotSpotIndex++;

				dataPos += HOT_SPOT_DATA_LENGTH;

			}

		}

		

		// If not found, fill result with 0's

		if (!found)

		{

			result[0] = 0;

			result[1] = 0;

		}

		

		return result;

	}

	

    /*private void CreateMirroredStrip()

    {

//    	int w = m_Image.getWidth();

//    	int h = m_Image.getHeight();

//    	int rgb[] = new int[w*h];  

//    	m_Image.getRGB(rgb, 0, w, 0, 0, w, h);

//    	int pos = 0, pos_mirror=w-1;

//    	int w2 = w/2;

//    	for (int y=0; y<h; y++, pos+=w-w2, pos_mirror+=w+w2)

//    		for (int x=0; x<w2; x++, pos++, pos_mirror--)

//    		{

//    			int t = rgb[pos];

//    			rgb[pos] = rgb[pos_mirror];

//    			rgb[pos_mirror] = t;

//    		}

//       	m_MirroredImage = Image.createRGBImage(rgb, w, h, true);

       	m_MirroredImage = m_Image;

    }*/

	

	public boolean SetHotSpot(int frameNum, int type, int[] pos)

	{		

//		int[] result = new int[2];	// Value to return

		int dataPos;				// Position in metrics data

		boolean found;				// True if hot spot type found

		int hotSpotIndex;			// Current hot spot being processed

		

		// Check validity of arguments

		//Debug.assertPrintln((frameNum >= 0) && (frameNum < GetNumFrames()),

						//"SetHotSpot: Invalid frame number !(0<=" + frameNum 

							//+ "<" + GetNumFrames() + ")");

		

		// Skip to correct position in metrics data for this frame

		// Also skip past the non-hot spot data

		dataPos = frameNum * METRICS_SIZE;

		dataPos += PRE_HOT_SPOT_DATA_LENGTH;

		

		// Do search for correct hot spot type		

		found = false;

		hotSpotIndex = 0;

		while (!found && (hotSpotIndex < MAX_HOT_SPOTS))

		{

			int curType = (int)m_ImageMetrics[dataPos];

			

			if (type == curType)

			{

				dataPos++;

				m_ImageMetrics[dataPos++] = (byte)(pos[0] >> 8);

				m_ImageMetrics[dataPos++] = (byte)(pos[0]);

				m_ImageMetrics[dataPos++] = (byte)(pos[1] >> 8);

				m_ImageMetrics[dataPos++] = (byte)(pos[1]);

				return true;

			}

		}

		return false;

		// If not found, fill result with 0's

	}



	/** Maximum loaded hot spots */

	private static final int MAX_HOT_SPOTS = 2;

	

	/** Size of non-hotspot data, that appears before hotspot data in each metric */

	private static final int PRE_HOT_SPOT_DATA_LENGTH = 6;

	

	/** Data size per hot spot */

	private static final int HOT_SPOT_DATA_LENGTH = 5;

	

	/** Size of each metric record */

	private static final int METRICS_SIZE = PRE_HOT_SPOT_DATA_LENGTH 

											+ MAX_HOT_SPOTS * HOT_SPOT_DATA_LENGTH;

	

	private int m_Guid;				/** Guid of this strip (DEBUG ONLY) */

	private Image m_Image;			/** Image strip */

//	private Image m_MirroredImage;	/** Mirrored copy of image strip for crappy S60 devices*/

	private byte[] m_ImageMetrics;	/** Location of each image within strip */

}

