/*
 *     Copyright (c)2001-2003 DemiVision, LLC. All Rights Reserved.
 *
 * The information contained herein is the CONFIDENTIAL and PROPRIETARY
 *                  information of DemiVision, LLC.
 */

package com.bejeweled2_j2me;

import javax.microedition.lcdui.*;
import javax.microedition.midlet.MIDlet;
import java.io.*;
import java.util.Random;
import javax.microedition.media.*;

/**
 * The <tt>GameEngine</tt> controls the top-level flow between system screens within
 * the application interface. A screen life cycle (init, start, pause, destroy)
 * is defined. Though multiple instances of <tt>BaseScreen</tt> are in existence
 * at the same time, there is only ever one screen in focus. The GameEngine also
 * maintains the application offscreen buffer for drawing as well as the master
 * game loop for all game logic. The GameEngine supplies the following functionality
 * to all instances of <tt>GameObject</tt>:
 *
 * <ul>
 *   <li>Game heartbeat</li>
 *   <li>Double buffer</li>
 *   <li>Key events</li>
 *   <li>Network events</li>
 * </ul>
 *
 * The GameEngine also provides API methods for accessing network functionality
 * neccessary to communicate with the remote <i>Kraken</i> server. Font handling and
 * general utility methods are provided as well. The GameEngine is a singleton object.
 *
 * @see         BaseScreen
 * @see         GameObject
 *
 * @author      Barry Sohl
 * @version     1.1.0
 */
final class GameEngine extends Canvas implements Runnable
{
    /* Constants */

    /** Major version component of <i>Dioskilos</i> library in use by this application. */
    public static final byte VERSION_MAJOR = 1;

    /** Minor version component of <i>Dioskilos</i> library in use by this application. */
    public static final byte VERSION_MINOR = 1;

    /** Micro version component of <i>Dioskilos</i> library in use by this application. */
    public static final byte VERSION_MICRO = 0;

    /** Build version component of <i>Dioskilos</i> library in use by this application. */
    public static final int VERSION_BUILD = 8;

    /** String representation of <i>Dioskilos</i> library version in the format: 'major.minor.micro.build'. */
    public static String VERSION_LABEL;

    // Screen identifiers
    private final byte NUM_SCREENS = 4;

    /** Identifier for the <tt>MainScreen</tt>, which contains the main application menu. */
    public static final byte SCREEN_MAIN = 0;

    /** Identifier for the <tt>SettingsScreen</tt>, which allows users to set application parameters. */
    public static final byte SCREEN_SETTINGS = 1;

    /** Identifier for the <tt>GameScreen</tt>, which contains the core game logic. */
    public static final byte SCREEN_GAME = 2;
    
    public static final byte SCREEN_GAME_MODE = 3;

    // Fonts

    /** Shorthand reference for the Small, Plain, System <tt>Font</tt>. */
    public static Font FONT_PLAIN;
    /** Shorthand reference for the Small, Bold, System <tt>Font</tt>. */
    public static Font FONT_BOLD;
    /** Shorthand reference for the Medium, Plain, System <tt>Font</tt> used internally for title bar text. */
    public static Font FONT_TITLE;

    /** Pixel height of the small, plain, proportional font. */
    public static int HT_PLAIN;

    // Game action codes (should be treated as constants)

    /** Game action code for the 'Left' key. */
    public static int KEY_LEFT;

    /** Game action code for the 'Right' key. */
    public static int KEY_RIGHT;

    /** Game action code for the 'Up' key. */
    public static int KEY_UP;

    /** Game action code for the 'Down' key. */
    public static int KEY_DOWN;

    /** Game action code for the 'Fire' key. */
    public static int KEY_FIRE;

    /** Game action code for the 'Game A' key. */
    public static int KEY_GAME_A;

    /** Game action code for the 'Game B' key. */
    public static int KEY_GAME_B;

    /** Game action code for the 'Game C' key. */
    public static int KEY_GAME_C;

    /** Game action code for the 'Game D' key. */
    public static int KEY_GAME_D;

    // Alert sound support
    /** Identifier for the simple sound associated with an Alarm alert. */
    public static final AlertType SOUND_ALARM = AlertType.ALARM;

    /** Identifier for the simple sound associated with a Confirmation alert. */
    public static final AlertType SOUND_CONFIRM = AlertType.CONFIRMATION;

    /** Identifier for the simple sound associated with an Error alert. */
    public static final AlertType SOUND_ERROR = AlertType.ERROR;

    /** Identifier for the simple sound associated with an Info alert. */
    public static final AlertType SOUND_INFO = AlertType.INFO;

    /** Identifier for the simple sound associated with a Warning alert. */
    public static final AlertType SOUND_WARNING = AlertType.WARNING;

    /** Number of sounds to load. */
    public static final int NUMBER_OF_SOUNDS = 6;

    /** Shorthand reference for <tt>(Graphics.TOP | Graphics.HCENTER)</tt>. */
    public static final int CENTERED = (Graphics.TOP | Graphics.HCENTER);

    /** Shorthand reference for <tt>(Graphics.TOP | Graphics.LEFT)</tt>. */
    public static final int TOP_LEFT = (Graphics.TOP | Graphics.LEFT);

    /** Shorthand reference for <tt>(Graphics.TOP | Graphics.RIGHT)</tt>. */
    public static final int TOP_RIGHT = (Graphics.TOP | Graphics.RIGHT);

    /** Shorthand reference for <tt>(Graphics.BOTTOM | Graphics.LEFT)</tt>. */
    public static final int BOTTOM_LEFT = (Graphics.BOTTOM | Graphics.LEFT);

    /** Shorthand reference for <tt>(Graphics.BOTTOM | Graphics.RIGHT)</tt>. */
    public static final int BOTTOM_RIGHT = (Graphics.BOTTOM | Graphics.RIGHT);

    // Text resource handling
    private final int MAX_TEXT_SIZE = 2048;

    private final char DELIM_TEXT = '~';
    private final char DELIM_LIST = ',';

    /* Data Fields */
    private static GameEngine gameEngine;   // Singleton instance

    private DioskilosMIDlet parent;                 // Helper references
    private Display display;
    private Random random;

    private boolean isDemo;                 // Is this a demo version

    private int frameDelay;               // Frame rate

    String username;                        // Local player's username

    private BaseScreen[] screenList;        // Screen handling
    private byte curScreen;
    public int wd, ht;

    private boolean paused;                 // Maintains game heartbeat

    private Image offscreenImg;             // Maintains double buffer
    private Graphics offscreenGc;
    private boolean systemDoubleBuffer;

    private char[] textBuffer;              // Text resource handling

    private javax.microedition.media.Player[] midiPlayers;
    private javax.microedition.media.Player midiPlayer;

    /** Is the 'Sound On' setting currently enabled? */
    public boolean soundOn;

    /** Is the 'Vibrate On' setting currently enabled? */
    public boolean vibrateOn;

    /** Is the 'Funlights On' setting currently enabled? */
    public boolean funlightsOn;

    // Globally shared resources

    /** Reference to a preloaded "bullet" image suitable for use in <tt>List</tt> widgets. */
    public Image bulletImg;

    /**
     * Private constructor protects against instantiation.
     */
    private GameEngine()
    {setFullScreenMode(true);
        }

    /**
     * Returns a reference to the singleton <tt>GameEngine</tt>.
     *
     * @return  reference to singleton instance.
     */
    public static GameEngine getInstance()
    {
        // Lazy instantiation
        if ( gameEngine == null )
        {
            gameEngine = new GameEngine();
        }

        return gameEngine;
    }

    /**
     * Get the MIDlet associated with this game engine instance.
     *
     * @return MIDlet instance
     */
    public DioskilosMIDlet getMIDlet() {
        return(parent);
    }

    /**
     * Returns an application specific property value matching the
     * specified key as defined in the MIDlet .jad file.
     *
     * @param   key     return property value matching this key, null if no match.
     * @return  property value as a <tt>String</tt>.
     */
    public String getProperty( String key )
    {
        return parent.getAppProperty( key );
    }

    /**
     * Returns an application specific property value matching the
     * specified key as defined in the MIDlet .jad file.
     *
     * @param   key     return property value matching this key, null if no match.
     * @return  property value as an <i>boolean</i>.
     */
    public boolean getBooleanProperty( String key )
    {
        String value = getProperty(key).toLowerCase();
        if(value.equals(StringTable.TRUE) || value.equals(StringTable.YES) ||
            value.equals(String.valueOf(true).toLowerCase())) {
            return(true);
        }
        return(false);
    }

    /**
     * Returns an application specific property value matching the
     * specified key as defined in the MIDlet .jad file.
     *
     * @param   key     return property value matching this key, null if no match.
     * @return  property value as an <i>int</i>.
     */
    public int getIntProperty( String key )
    {
        return Integer.parseInt( getProperty( key ) );
    }

    /**
     * Returns an application specific property value represented as a comma
     * delimited list of items. The format for the property is:<p>
     *
     * <tt>Prop-Name: numItems,item1,item2,item3</tt>
     *
     * A newly allocated array of strings will be returned containing each item
     * in the list.
     *
     * @param   property    application property being parsed.
     * @return  array of strings containing each item in property list.
     */
    public String[] getPropertyList( String property )
    {
        String propList = getProperty( property );

        int beginIndex = 0;
        int endIndex = propList.indexOf( DELIM_LIST );

        // Determine how many items in list
        int numItems = Integer.parseInt( propList.substring( beginIndex, endIndex ) );
        beginIndex = (endIndex + 1);

        String[] listItems = new String[ numItems ];

        // Parse each item
        for ( int i = 0; i < numItems; i++ )
        {
            endIndex = propList.indexOf( DELIM_LIST, beginIndex );
            endIndex = (endIndex > 0) ? endIndex : propList.length();

            listItems[i] = propList.substring( beginIndex, endIndex );
            beginIndex = (endIndex + 1);
        }

        return listItems;
    }

    /**
     * Returns a text string read from an external resource file. The resource file
     * must be a plain text file with no formatting. For efficiency, the file can be
     * subdivided into resource subitems. Subitems are separated by the '<tt>~</tt>'
     * character placed on a line by itself. The first subitem, and files with only a
     * single item, must include this separator. There should be no separator at the
     * end of the file.
     *
     * @param   file        path to plain text resource file.
     * @param   offset      subitem offset within resource file.
     *
     * @return  text resource at specified offset within resource file.
     */
    public String getTextResource( String file, int offset )
    {
        String text = null;

        try
        {
            // Open external file
            DataInputStream in = new DataInputStream( StringTable.getResource( file, true ) );

            char nextByte = 0;

            // Advance to offset of resource subitem
            for ( int i = 0; i < (offset + 1); i++ )
            {
                do
                {
                    nextByte = readTextByte(in);
                }
                while ( (nextByte != -1) && (nextByte != DELIM_TEXT) );
            }

            int i = 0;

            // Clear buffer
            for ( i = 0; i < MAX_TEXT_SIZE; i++ )
            {
                textBuffer[i] = 0;
            }

            // Skip CR/LF
            readTextByte(in);

            i = 0;
            boolean reading = true;

            // Read all text for item
            while ( reading )
            {
                try
                {
                    nextByte = readTextByte(in);

                    switch ( nextByte )
                    {
                        // End of subitem
                        case DELIM_TEXT:
                        {
                            reading = false;
                            break;
                        }
                        // Otherwise add to string
                        default:
                        {
                            textBuffer[ i++ ] = nextByte;
                            break;
                        }
                    }
                }
                catch ( EOFException ex )
                {
                    reading = false;
                }
            }

            // Final result
            text = new String( textBuffer ).trim();

            // Cleanup
            in.close();

            // Replace placeholders with version numbers.
            int versionPosition = text.indexOf(StringTable.VERSION_PLACEHOLDER);
            while(versionPosition != -1) {
                text = text.substring(0, versionPosition) +
                        gameEngine.getProperty(StringTable.PROP_VERSION) +
                        text.substring(versionPosition + 1);
                versionPosition = text.indexOf(StringTable.VERSION_PLACEHOLDER);
            }
            int libraryPosition = text.indexOf(StringTable.LIBRARY_PLACEHOLDER);
            while(libraryPosition != -1) {
                text = text.substring(0, libraryPosition) +
                        GameEngine.VERSION_LABEL +
                        text.substring(libraryPosition + 1);
                libraryPosition = text.indexOf(StringTable.LIBRARY_PLACEHOLDER);
            }
        }
        catch ( Exception ex )
        {
            if (Build.DEBUG) gameEngine.writeDebug( StringTable.DBG_GEGTR1 , ex );
        }

        return text;
    }

    /**
     * Read the next byte from a text file. Ignores windows linefeeds.
     *
     * @param in text file stream
     * @return next character
     * @throws IOException if reading fails or EOF is reached
     */
    private char readTextByte(DataInputStream in) throws IOException {
        char character = StringTable.readUnicode(in);
        while(character == 0x0d) {
            character = StringTable.readUnicode(in);
        }
        return (character);
    }

    /**
     * Is this a demo version of the game? Same code, just different
     * flow.
     *
     * @return true if demo version
     */
    public boolean isDemo() {
        return isDemo;
    }

    /**
     * Allows <tt>BaseScreen</tt> implementations to access the master display.
     *
     * @return  reference to master display.
     */
    public Display getMasterDisplay()
    {
        return display;
    }

    /**
     * Returns a reference to the specified screen from within the list of
     * system screens. Only valid after initialization.
     *
     * @param   screenId    unique identifier of desired screen.
     * @return  reference to specified screen.
     */
    public BaseScreen getScreenInstance( byte screenId )
    {
        return screenList[ screenId ];
    }

    /**
     * Returns a reference to the <tt>SettingsScreen</tt> used to retrieve
     * user defined settings. Call {@link SettingsScreen#isOn(byte)}
     * to determine whether a particular setting is turned on or off.
     *
     * @return  reference to <tt>SettingsScreen</tt>.
     */
    public SettingsScreen getSettings()
    {
        return (SettingsScreen) screenList[ SCREEN_SETTINGS ];
    }

    /**
     * Reads a string from the specified input stream at the current read
     * pointer position. Developers <i>must</i> use this method so that the
     * implementation can be changed to/from Unicode as necessary. Note that
     * the returned string may be empty or null. This method supports strings
     * up to 128 characters in length.
     *
     * @param   in      input stream ready for reading.
     * @return  string read from stream.
     * @throws  IOException if error occurs while reading.
     */
    public String readString( DataInput in ) throws IOException
    {
        return internalReadString( in, in.readByte() );
    }

    /**
     * This method is identical to {@link readString(DataInput)} except that strings
     * 32,767 characters in length are supported. Note that the returned string may
     * be empty or null.
     *
     * @param   in      input stream ready for reading.
     * @return  string read from stream.
     * @throws  IOException if error occurs while reading.
     */
    public String readLongString( DataInput in ) throws IOException
    {
        return internalReadString( in, in.readShort() );
    }

    /**
     * Used internally to read a string from the input stream. Supports both 128
     * character strings as well as 32,767 character strings.<p>
     *
     * <b>WARNING</b>: This method allocates bytes equal to the leading
     * length indicator of the string for each call.<p>
     *
     * @param   in      input stream ready for reading.
     * @param   len     number of bytes string occupies on stream.
     *
     * @return  string read from stream.
     * @throws  IOException if error occurs while reading.
     */
    private String internalReadString( DataInput in, int len ) throws IOException
    {
        String str = null;

        if ( len >= 0 )
        {
            // Use temp buffer
            byte[] buffer = new byte[ len ];
            in.readFully( buffer );

            str = new String( buffer );
        }

        return str;
    }

    /**
     * Writes the given string to the specified output stream at the current
     * write pointer position. Developers <i>must</i> use this method so that the
     * implementation can be changed to/from Unicode as necessary. This method
     * supports strings 128 characters in length. Null and empty strings are valid.
     *
     * @param   out     output stream ready for writing.
     * @param   str     string to be written.
     *
     * @throws  IOException if error occurs while writing.
     */
    public void writeString( DataOutput out, String str ) throws IOException
    {
        if ( str == null )
        {
            out.writeByte( (byte)-1 );
        }
        else
        {
            byte len = (byte) str.length();

            // Strings are a length indicator followed by content
            out.writeByte( len );
            out.write( str.getBytes(), 0, len );
        }
    }

    /**
     * This method is identical to {@link writeString(DataOutput)} except that strings
     * 32,767 characters in length are supported. Null and empty strings are valid.
     *
     * @param   out     output stream ready for writing.
     * @param   str     string to be written.
     *
     * @throws  IOException if error occurs while writing.
     */
    public void writeLongString( DataOutput out, String str ) throws IOException
    {
        if ( str == null )
        {
            out.writeShort( (short)-1 );
        }
        else
        {
            short len = (short) str.length();

            // Strings are a length indicator followed by content
            out.writeShort( len );
            out.write( str.getBytes(), 0, len );
        }
    }

    /**
     * Returns the total number of bytes that the specified string will occupy
     * on an input or output stream including any header information. Developers
     * <i>must</i> use this method because the internal implementation may be
     * changed to/from double-byte Unicode at any time.
     *
     * @param   str     string being measured.
     * @return  total number bytes string will occupy, 0 for empty string, -1 for null.
     */
    public int getStringSize( String str )
    {
        int size = -1;

        if ( str != null )
        {
            int len = str.length();
            size = (len + ((len > 255) ? 2 : 1));
        }

        return size;
    }

    /**
     * Transitions the application to a new screen. The current screen will
     * be hidden before the new one is shown. Only valid for transitioning
     * between system screens.
     *
     * @param   screenId    unique identifier of destination screen.
     */
    public void transition( byte screenId )
    {
        long now = System.currentTimeMillis();
        Dialog dlg = Dialog.getInstance();

        // Hide old
        screenList[ curScreen ].pause( now );
        dlg.hide();

        curScreen = screenId;

        // Show new
        screenList[ curScreen ].start( true, now );

        // Support screens that start with dialog
        if ( !dlg.isShowing() )
        {
            setCurrent(curScreen);
        }
    }

    public void transition( byte screenId, boolean init )
    {
        long now = System.currentTimeMillis();
        Dialog dlg = Dialog.getInstance();

        // Hide old
        screenList[ curScreen ].pause( now );
        dlg.hide();

        curScreen = screenId;

        // Show new
        screenList[ curScreen ].start( init, now );

        // Support screens that start with dialog
        if ( !dlg.isShowing() )
        {
            setCurrent(curScreen);
        }
    }
    
    /**
     * Restarts the specified screen without re-initializing as is done
     * from {@link transition(byte)}.
     *
     * @param   screenId    unique identifier of screen being restarted.
     */
    public void restart( byte screenId )
    {
        Dialog.getInstance().hide();

        // Reshow same screen
        setCurrent(screenId);
        screenList[ screenId ].start( false, System.currentTimeMillis() );
    }

    /**
     * Registers a system screen in the master screen container. There is space
     * reserved for each of the five system screen types:
     *
     *   {@link GameEngine#SCREEN_MAIN},
     *   {@link GameEngine#SCREEN_SETTINGS},
     *   {@link GameEngine#SCREEN_GAME},
     *
     * In most cases, an instance of the default screen should be provided. If
     * the developer has created an extended version of the screen, then that
     * instance should be provided instead. Typically this is only necessary
     * for the <tt>GameScreen</tt>.
     *
     * @param   screenId    unique identifier of screen being registered.
     * @param   screen      reference to screen instance.
     *
     * @throws  IOException if error occurs initializing screen.
     */
    public void registerScreen( byte screenId, BaseScreen screen ) throws IOException
    {
        screen.setBounds( 0, 0, wd, ht );
        screen.setBgndColor( BaseScreen.COLOR_BGND );
        screen.init();

        screenList[ screenId ] = screen;
    }

    /**
     * Returns a psudeo-random number between 0 and (range - 1).
     *
     * @param   range   number of possibilities.
     * @return  randomly generated number.
     */
    public int genRandomNumber( int range )
    {
        return Math.abs( random.nextInt() % range );
    }

    /**
     * Plays the simple alert sound specified by the given identifier. This is
     * the minimal sound capability available on most MIDP devices. <i>Not
     * available on all devices</i>.
     *
     * @param   type  {@link GameEngine#SOUND_ALARM identifier} of sound to play.
     * @param force
     */
    public void playAlertSound(AlertType type, boolean force)
    {
    }

    /**
     * Stop all MIDP2 midi players and free resources for those players.
     *
     * @throws MediaException if unable to stop all players
     */
    private void stopMidiForMotorolaI730() throws MediaException {
        if(midiPlayer != null) {
            midiPlayer.stop();
            midiPlayer.deallocate();
            midiPlayer.close();
            midiPlayer = null;
        }
    }


    /**
     * Loads all MIDI resources to be played by the application. This method
     * must be called prior to playing any MIDI sound. External MIDI files must
     * use the following naming convention: <tt>sndX.mid</tt>, where <tt>X</tt>
     * is a numeric index starting at 0 and increasing sequentially. This index
     * is used to identify the file for playback in {@link playMidi(int,boolean)}.
     * The external JAD file must contain a property <tt>Num-MIDIs</tt>, which
     * indicates the number of MIDI files to be loaded by this method. Note that
     * the sound at index 0 will be used internally as the lobby "match made" sound.
     * Therefore index 0 is always required for phones supporting MIDI, and should
     * be chosen accordingly. <i>Not available on all devices</i>.
     */
    public void loadMidi()
    {
        int numFiles = NUMBER_OF_SOUNDS;

        // Allocate memory
        midiPlayers = new javax.microedition.media.Player[ numFiles ];

        // Preload MIDI resources
        try
        {
            for ( int i = 0; i < numFiles; i++ )
            {
                // Files must follow 'sndX.mid' naming convention
                midiPlayers[i] =
                    Manager.createPlayer(getClass().getResourceAsStream(
                        StringTable.MIDI_FILE_ROOT +
                        StringTable.MIDI_FILE_PREFIX + i +
                        StringTable.MIDI_FILE_SUFFIX
                        ), StringTable.MIDI_MIME_TYPE);
            }
        }
        catch ( Exception ex )
        {
            if (Build.DEBUG) writeDebug( StringTable.DBG_GELM1 , ex );
        }
    }

    /**
     * Plays the MIDI sound corresponding to the specified index. All MIDI files
     * must be already preloaded using {@link loadMidi()}. This method has no
     * effect if sound is currently turned off. The MIDI will be automatically
     * stopped if the MIDlet is paused. It will not automatically be restarted.
     * The MIDI sound can be played once or in a continuous loop. <i>Not available
     * on all devices</i>.
     *
     * @param   index   numeric index of preloaded MIDI file.
     * @param   loop    play in a continuous loop or just once?
     * @param force
     */
    public void playMidi(int index, boolean loop, boolean force)
    {
        // Only play if sound is on
        if ( soundOn || force )
        {
            try
            {
                // MIDP 2.0 playback
                stopMidi(index);
                midiPlayers[index].start();

                // MIDP 2.0 playback for the Motorola i730
                //stopMidiForMotorolaI730();
                //midiPlayer = Manager.createPlayer(
                //    StringTable.MIDI_URI_PREFIX + StringTable.MIDI_FILE_ROOT + 
                //    StringTable.MIDI_FILE_PREFIX +
                //    index + StringTable.MIDI_FILE_SUFFIX);
                //midiPlayer.start();
            }
            catch ( Exception ex )
            {
                if (Build.DEBUG) writeDebug( StringTable.DBG_GEPM1 , ex );
            }
        }
    }

    /**
     * Stops any currently playing MIDI sound. This method has no effect if no
     * sound is currently playing. Playback will not be restarted until {@link
     * playMidi(boolean)} is called again. <i>Not available on all devices</i>.
     */
    public void stopMidi()
    {
        stopMidi(-1);
    }

    /**
     * Stops any currently playing MIDI sound. This method has no effect if no
     * sound is currently playing. Playback will not be restarted until {@link
     * playMidi(boolean)} is called again. <i>Not available on all devices</i>.
     *
     * @param nextSound that will be played (used for optimization)
     */
    public void stopMidi(int nextSound)
    {
        for(int index = 0; index < midiPlayers.length; ++index) {
            try {
                midiPlayers[index].stop();
                if(index != nextSound) {
                    midiPlayers[index].deallocate();
                }
            } catch(MediaException ex) {
                if (Build.DEBUG) writeDebug( StringTable.DBG_GESM1 , ex );
            }
        }
        try {
            stopMidiForMotorolaI730();
        } catch(MediaException ex) {
            if (Build.DEBUG) writeDebug( StringTable.DBG_GESM1 , ex );
        }
    }

    /**
     * Calls the native vibrate functionality of the device. The vibration will
     * last for the specified duration. A duration of 0 will stop the current
     * vibration, if any. <i>Not available on all devices</i>.
     *
     * @param   duration    desired length of vibration in milliseconds.
     * @param force
     */
    public void vibrate(int duration, boolean force)
    {
        if(gameEngine.vibrateOn || force) {
            display.vibrate(duration);
        }
    }

    /**
     * Loads an image with the specified filename from the MIDlet JAR file. If
     * the image is not at the root level of the JAR, specifiy a relative path.
     * This method will block until the image is fully loaded.
     *
     * @param   filename    image to be loaded.
     * @param isLocalized
     * @return  successfully loaded image.
     * @throws IOException if error occurs loading image.
     */
    public Image loadImage(String filename, boolean isLocalized) throws IOException
    {
        Thread.yield();         // Allow loading display some time.
        Image image = null;
        if(isLocalized) {
            try {
                image = Image.createImage(StringTable.FILE_ROOT +
                        StringTable.getLocale() + filename);
            } catch(IOException exception) {
                image = null;
            }
        }
        if(image == null) {
            try {
                image = Image.createImage(StringTable.FILE_ROOT + filename);
            }
            catch (IOException ex) {
                System.out.println("Couldn't load: " + filename);
                throw ex;
            }
        }
        return image;
    }

    /**
     * Draws a clipped subimage extracted from a larger image strip. This is
     * typically used to efficiently display individual frames of an animation
     * sequence. The 'src' parameters specify the offset to the subimage within
     * the source image strip. The 'dest' parameters specify the location where
     * the subimage is to be drawn on the destination graphics context. The
     * clipping reqion of the destination graphics context will be returned to
     * its original state when this method completes.
     *
     * @param   gc      destination graphics context used for drawing.
     * @param   image   source image from which to extract clipped subimage.
     *
     * @param   destX   x location of destination drawing area.
     * @param   destY   y location of destination drawing area.
     * @param   destWd  width of destination drawing area.
     * @param   destHt  height of destination drawing area.
     *
     * @param   srcX    x offset to desired subimage in source image.
     * @param   srcY    y offset to desired subimage in source image.
     */
    public void drawClippedImage( Graphics gc, Image image,
                                  int destX, int destY, int destWd, int destHt,
                                  int srcX, int srcY )
    {
        // Save current clipping region
        int oldX  = gc.getClipX();
        int oldY  = gc.getClipY();
        int oldWd = gc.getClipWidth();
        int oldHt = gc.getClipHeight();

        // Set new clipping region, draw clipped image
        gc.setClip( destX, destY, destWd, destHt );
        gc.drawImage( image, (destX - srcX), (destY - srcY), TOP_LEFT);

        // Reset clipping region
        gc.setClip( oldX, oldY, oldWd, oldHt );
    }

    /**
     * Enables quiting the application from any screen. All resources will
     * be released and control will return to the OS.
     */
    public void exit()
    {
        // Cleanup
        try
        {
            destroy();
        }
        catch ( Exception ex )
        {
            if (Build.DEBUG) writeDebug( StringTable.DBG_GEE1 , ex );
        }

        // System exit
        parent.notifyDestroyed();
    }

    /**
     * Outputs debug content such as error messages or trace statements. If RMS
     * debugging is enabled, the output will be persisted to RMS for later viewing.
     * Otherwise, output is directed at stderr. Additional debug information will
     * be output if an exception is specified.
     *
     * @param   str     debug string being written.
     * @param   ex      exception causing error, null if no exception.
     */
    public void writeDebug(String str, Exception ex) {
        String msg = str;

        // Additionl debug info
        if(ex != null) {
            msg = (str + StringTable.DBG_SEPARATOR + ex +
                StringTable.DBG_SEPARATOR + ex.getMessage());
            ex.printStackTrace();
        }

        System.err.println(msg);
    }

    /**
     * Performs initialization including allocation of the offscreen buffer.
     * A double buffer will be allocated only if the system does not double
     * buffer by default. Should be called as close to instantiation as
     * possible given that significant memory is allocated here.
     *
     * @param   parent      reference to parent <tt>MIDlet</tt>.
     * @throws  IOException if error occurs loading resources.
     */
    void init( DioskilosMIDlet parent ) throws IOException
    {
        this.parent = parent;

        // Check if this is a demo version
        isDemo = true;//gameEngine.getBooleanProperty(StringTable.PROP_DEMO);

        // Library version
        String buildStr = (StringTable.BUILD_PAD + VERSION_BUILD);
        buildStr = buildStr.substring( buildStr.length() - StringTable.BUILD_PAD.length() );

        VERSION_LABEL = (VERSION_MAJOR + StringTable.VERSION_DELIM + VERSION_MINOR + StringTable.VERSION_DELIM +
                                                         VERSION_MICRO + StringTable.VERSION_DELIM + buildStr);
        // Seed random number generator
        random = new Random();

        // Lookup and save game action codes
        KEY_LEFT   = getKeyCode( LEFT );
        KEY_RIGHT  = getKeyCode( RIGHT );
        KEY_UP     = getKeyCode( UP );
        KEY_DOWN   = getKeyCode( DOWN );
        KEY_FIRE   = getKeyCode( FIRE );
        KEY_GAME_A = getKeyCode( GAME_A );
        KEY_GAME_B = getKeyCode( GAME_B );
        KEY_GAME_C = getKeyCode( GAME_C );
        KEY_GAME_D = getKeyCode( GAME_D );

        // Save device specific font constants
        FONT_PLAIN     = Font.getFont( Font.FACE_SYSTEM, Font.STYLE_PLAIN,
            DeviceSpecific.PLAIN_FONT_SIZE );
        FONT_BOLD      = Font.getFont( Font.FACE_SYSTEM, Font.STYLE_BOLD, 
            DeviceSpecific.PLAIN_FONT_SIZE ); 
        FONT_TITLE     = Font.getFont( Font.FACE_SYSTEM, Font.STYLE_BOLD,  Font.SIZE_LARGE );

        HT_PLAIN = FONT_PLAIN.getHeight();

        // Allocate double buffer (if necessary)
        systemDoubleBuffer = isDoubleBuffered();

        if ( !systemDoubleBuffer )
        {
            offscreenImg = Image.createImage( getWidth(), getHeight() );
            offscreenGc = offscreenImg.getGraphics();
        }

        // Calculate app specific frame delay
        frameDelay = DeviceSpecific.FRAME_DELAY;

        // Prep for text resource handling
        textBuffer = new char[ MAX_TEXT_SIZE ];

        wd = getWidth();
        ht = getHeight();

        // Allocate screen container
        screenList = new BaseScreen[ NUM_SCREENS ];

        // Start with the main menu
        curScreen = SCREEN_MAIN;

        // Initialize menu and dialog statics
        //MainMenu.init();
        DialogCanvas.init();
    }

    /**
     * Starts, or restarts, the current screen. If the current screen is
     * drawing to the <tt>GameEngine</tt>, the game heartbeat is also restarted.
     * Safe to call multiple times.
     *
     * @param   init    initial start after launch?
     * @param   show    tell the display to make this visible?
     */
    public void start( boolean init, boolean show)
    {
        // Save reference to master display, not valid until startApp()
        if ( init )
        {
            display = parent.getDisplay( );

            // Initialize global dialog box
            Dialog.getInstance().init( display );
        }

        // Refocus current screen
        screenList[ curScreen ].start( init, System.currentTimeMillis() );
        screenList[ curScreen ].markDirty();
        if(show) {
            // If the dialog thinks it should be showing then it probably
            // should be.
            if(Dialog.getInstance().isShowing()) {
                display.setCurrent(Dialog.getInstance().getDisplayable());
            } else {
                setCurrent(curScreen);
            }
        }

        // Unpause
        paused = false;
    }

    private void setCurrent(int id) {
        Displayable displayable = screenList[ id ].getDisplayable();
        if (displayable != null) {
            display.setCurrent(displayable);
        }
    }

    /**
     * Pauses game heartbeat and draw/paint cycle, but does not release
     * significant resources. Can also be used to pause the network
     * listening threads.
     */
    public void pause()
    {
        paused = true;
        screenList[ curScreen ].pause( System.currentTimeMillis() );

        // Stops sounds
        stopMidi();
    }

    /**
     * Insures proper cleanup upon exit and helps garbage collector.
     */
    void destroy()
    {
        // Cleanup all preallocated screens
        if ( screenList != null )
        {
            for ( int i = 0; i < NUM_SCREENS; i++ )
            {
                screenList[i].destroy();
            }
        }
    }

    /* Canvas */

    /**
     * Called when the game engine regains focus following a call to {@link
     * hideNotify()}. The game heartbeat is restarted and gameplay continues.
     */
    protected void showNotify()
    {
        start(false, false);
        repaint();

        // Restart game loop
        display.callSerially( this );
    }

    /**
     * Called when the game engine loses focus, typically following a screen
     * transition or display of a modal dialog. The game heartbeat and all
     * gameplay will be paused. Note that network activity <i>does</i> continue.
     */
    protected void hideNotify()
    {
        pause();
    }

    /**
     * Redraws the entire current game state and then blits to the drawing
     * surface. Drawing is either directly to the underlying <tt>Canvas</tt> or
     * to the offscreen buffer depending on whether or not the system is
     * automatically double buffering. <i>For internal use only</i>.
     *
     * @param   gc  graphics context being painted.
     */
    public void paint( Graphics gc )
    {
        // Auto double buffering, draw directly to canvas
        if ( systemDoubleBuffer )
        {
            screenList[ curScreen ].draw( gc );
        }
        // Manual double buffering, draw to offscreen buffer then page flip
        else
        {
            screenList[ curScreen ].draw( offscreenGc );
            gc.drawImage( offscreenImg, 0, 0, TOP_LEFT );
        }
    }

    /**
     * Translate key presses to a limited set of keys, allowing consistency
     * across phones and ease of handling for phone specific keys. If they
     * are not mapped to game actions by the phone, then the following
     * keys will be mapped:
     *
     * KEY_NUM5 -> FIRE
     * KEY_NUM2 -> UP
     * KEY_NUM8 -> DOWN
     * KEY_NUM4 -> LEFT
     * KEY_NUM6 -> RIGHT
     *
     * @param keyCode MIDP key value
     * @return translated key value
     */
    public int translateKey(int keyCode) {
        if(keyCode == DeviceSpecific.KEY_CODE_OK) {
            return(getKeyCode(FIRE));
        }

        // Translate number keys
        switch(keyCode) {
            case Canvas.KEY_NUM5:
                return (getKeyCode(FIRE));
            case Canvas.KEY_NUM2:
                return (getKeyCode(UP));
            case Canvas.KEY_NUM8:
                return (getKeyCode(DOWN));
            case Canvas.KEY_NUM4:
                return (getKeyCode(LEFT));
            case Canvas.KEY_NUM6:
                return (getKeyCode(RIGHT));
        }

        return(keyCode);
    }

    /**
     * Top-level event handler for key presses, which are supplied to the
     * current screen. Only the <tt>GameScreen</tt> handles key events.
     * <i>For internal use only</i>.
     *
     * @param   keyCode     key event being handled.
     */
    public void keyPressed( int keyCode )
    {
        //if (Build.DEBUG) 
        //{
        //    String screenName = screenList[ curScreen ].getClass().getName();
        //    int index = screenName.lastIndexOf('.');
        //    if (index != -1) {
        //        screenName = screenName.substring(index + 1);
        //    }
        //    System.out.println("GameEngine:" + screenName + ". keyCode = " + keyCode + 
        //        " = " + getKeyName(keyCode) + " = " + getGameAction(keyCode));
        //}
        
        keyCode = translateKey(keyCode);
        if(keyCode == 0) {
            return;
        }

        screenList[ curScreen ].keyPressed( keyCode );
    }

    /**
     * Top-level event handler for key releases, which are supplied to the
     * current screen. Only the <tt>GameScreen</tt> handles key events.
     * <i>For internal use only</i>.
     *
     * @param   keyCode     key event being handled.
     */
    public void keyReleased( int keyCode )
    {
        screenList[ curScreen ].keyReleased( keyCode );
    }

    /* Runnable */

    /**
     * Main game loop. Provides a heartbeat to the object tree. Redraws and blits
     * at the currently specified frame rate. If paused, the cycle will break and must
     * be restarted with a call to {@link start(boolean)} or {@link showNotify()}.
     * <i>For internal use only</i>.
     */
    public void run()
    {
        if ( !paused )
        {
            // Wait a beat according to frame rate
            if ( frameDelay > 0 )
            {
                try
                {
                    Thread.currentThread().sleep( frameDelay );
                }
                catch ( Exception ex )
                {}
            }
            else
            {
                Thread.yield();
            }

            try
            {
                long now = System.currentTimeMillis();

                // Execute time-dependent game functionality
                screenList[ curScreen ].heartbeat( now );

                // Request a repaint
                repaint();

                // Execute redraw and event handling serially
                display.callSerially( this );
            }
            catch ( Exception ex )
            {
                if (Build.DEBUG) writeDebug( StringTable.DBG_GER1 , ex );
            }

            Thread.yield();
        }
    }

}

