/*
 *     Copyright (c)2001-2002 DemiVision, LLC. All Rights Reserved.
 *
 * The information contained herein is the CONFIDENTIAL and PROPRIETARY
 *                  information of DemiVision, LLC.
 */

package com.bejeweled2_j2me;

import javax.microedition.lcdui.*;
import javax.microedition.rms.*;
import java.io.*;
import java.util.Vector;


/**
 * The <tt>SettingsScreen</tt> encapsulates all user interface activity for
 * setting the various options applicable to the application. The SettingsScreen
 * also manages the underlying RMS record store where all local settings
 * are persisted. A number of internal settings are stored as well as custom
 * settings that can be defined by the developer. Currently only custom boolean
 * settings are supported. These settings are saved to persistent storage as a
 * series of on/off flags. The following list of standard settings are
 * automatically displayed in the interface and managed by the SettingsScreen:
 *
 * <ul>
 *   <li>Obscure password on/off</li>
 *   <li>Logout</li>
 *   <li>Sound on/off</li>
 * </ul>
 *
 * Additionally, there is a debug setting provided that allows debug content such
 * as error messages and trace statements to be persisted to RMS on devices where
 * console output is not available. Developers can add their own application specific
 * settings by calling {add()}. This is typically best done from within
 * {@link DioskilosMIDlet#registerScreens()} in the extended version of <tt>DioskilosMIDlet</tt>.
 * In most cases, no further customization of the SettingsScreen is required. <i>It
 * is most efficient to use only the default SettingsScreen whenever possible</i>.
 * However, developers can provide custom functionality when necessary by extending
 * SettingsScreen. The custom screen must be registered with the GameEngine by overriding
 * {@link DioskilosMIDlet#registerScreens()}.
 *
 * @see         DioskilosMIDlet
 *
 * @author      Barry Sohl
 * @version     1.0.3
 */
class SettingsScreen extends BaseScreen implements CommandListener
{

    // Record constants.
    private final byte RECORD_HIGH_SCORE    = 1;
    private final byte RECORD_FLAGS         = 2;

    // Standard flags
    static byte FLAG_HINT = -1;
    static byte FLAG_SOUND = -1;
    static byte FLAG_VIBRATE = -1;
    static byte FLAG_FUNLIGHTS = -1;
    static byte FLAG_RESET = -1;

    /* Data Fields */
    private RecordStore     recordStore;        // Underlying RMS
    private Vector          flags;
    private DialogCanvas    settingsForm;       // Interface widgets

    /** High score. */
    private int highScore;

    /**
     * Default constructor allocates internal memory.
     */
    public SettingsScreen()
    {
        // Memory to store N Byte flags
        flags = new Vector();
    }

    /**
     * Determines whether or not the specified settings flag is currently
     * turned on.
     *
     * @param   index   setting flag in question.
     * @return  true if currently set to true, else false.
     */
    public boolean isOn( byte index )
    {
        if(index < 0) {
            return true;
        }
        else if (index >= flags.size()) {
            return false;
        }
        return (((Byte)flags.elementAt( index )).byteValue() == 1);
    }

    /**
     * Sets the application setting at the specified flag index to the
     * specified boolean value. The change is reflected in the interface.
     *
     * @param   index   flag being set.
     * @param   value   set flag true or false?
     */
    public void set( byte index, boolean value )
    {
        if(index < 0) {
            return;
        }
        flags.setElementAt( new Byte( (byte) (value ? 1 : 0) ), index );
        settingsForm.setCheckState(index, value);
    }

    /**
     * Adds an application specific setting to the interface <tt>ChoiceGroup</tt>
     * used to display options to the user. Only boolean settings are supported.
     * <i>Must be called before calling </i>{read()}. If a record
     * store has not yet been created, <tt>defaultValue</tt> will specify the
     * initial value of the setting. Otherwise, the value currently saved in the
     * record store will be used.
     *
     * @param   defaultValue    initial value of setting.
     */
    private byte add(boolean defaultValue )
    {
        flags.addElement( new Byte( (byte) (defaultValue ? 1 : 0) ) );
        return(byte) (flags.size() - 1);
    }

    /**
     * Reads in all setting values from the persistent record store. If a record
     * store does not yet exist, one will be created. The interface is updated
     * to reflect the stored setting values.
     *
     * @throws  RecordStoreException if error occurs creating or reading settings.
     */
    public void read() throws RecordStoreException
    {
        // Open the record store, create one if necessary
        recordStore = RecordStore.openRecordStore( StringTable.RECORD_STORE, true );

        // If newly created, set defaults
        if ( recordStore.getNumRecords() == 0 )
        {
            // All fields except for flags start empty
            for ( int i = 1; i < RECORD_FLAGS; i++ )
            {
                recordStore.addRecord( null, 0, 0 );
            }

            recordStore.addRecord( getFlagArray(), 0, flags.size() );
        }
        Thread.yield();         // Allow loading display some time.

        // Read high score.
        if(recordStore.getRecordSize(RECORD_HIGH_SCORE) > 0) {
            byte[] highScoreData = recordStore.getRecord(RECORD_HIGH_SCORE);
            ByteArrayInputStream highScore =
                new ByteArrayInputStream(highScoreData);
            try {
                this.highScore = new DataInputStream(highScore).readInt();
            } catch(IOException e) {
                //System.out.println(e.toString());
            }
        } else {
            highScore = 0;
        }
        Thread.yield();         // Allow loading display some time.

        // Flags
        byte[] record = recordStore.getRecord( RECORD_FLAGS );

        // Update interface to reflect flag settings
        for ( byte i = 0; i < record.length; i++ )
        {
            set( i, (record[i] == 1) );
        }
        Thread.yield();         // Allow loading display some time.

        recordStore.closeRecordStore();
        recordStore = null;
    }

    /**
     * Saves the entire set of settings flags, both standard and custom, to the
     * persistent record store. Because writing to the RMS is expensive, this
     * method should only be called if a setting has indeed changed.
     *
     * @throws  RecordStoreException if error occurs saving flags.
     */
    private void save() throws RecordStoreException
    {
        // Open the record store
        recordStore = RecordStore.openRecordStore(StringTable.RECORD_STORE, false);

        // Write high score.
        ByteArrayOutputStream highScore = new ByteArrayOutputStream();
        try {
            new DataOutputStream(highScore).writeInt(this.highScore);
            byte[] highScoreData = highScore.toByteArray();
            recordStore.setRecord(
                RECORD_HIGH_SCORE, highScoreData, 0, highScoreData.length);
        } catch(IOException e) {
            //System.out.println(e.toString());
        }

        // Update internal array
        for ( byte i = 0; i < flags.size(); i++ )
        {
            set( i, settingsForm.getCheckState( i ) );
        }

        // Write to RMS
        recordStore.setRecord( RECORD_FLAGS, getFlagArray(), 0, flags.size() );

        recordStore.closeRecordStore();
        recordStore = null;
    }

    /**
     * Attempt to restore the RMS to the correct state.
     *
     * @param reset true if default values should be used instead of current
     *          values
     * @return true if values were reset
     */
    public boolean restore(boolean reset)
    {
        // Determine if the info was reset.
        boolean wasReset = false;

        try {
            // Store current settings if not resetting.
            int highScore = 0;
            boolean sound       = true;
            boolean vibrate     = true;
            boolean funlights   = true;
            boolean hint        = true;
            if(!reset) {
                highScore = this.highScore;
                sound       = isOn(FLAG_SOUND);
                vibrate     = isOn(FLAG_VIBRATE);
                funlights   = isOn(FLAG_FUNLIGHTS);
                hint        = isOn(FLAG_HINT);
            }

            // Delete record store if it exists.
            if(recordStore != null) {
                try {
                    // Make sure record store is closed first.
                    recordStore.closeRecordStore();
                } catch(RecordStoreException exception) {
                }
            }
            try {
                RecordStore.deleteRecordStore(StringTable.RECORD_STORE);
            } catch(RecordStoreNotFoundException exception) {
            }

            // Recreate.
            read();

            // Restore and save settings is not resetting.
            if(reset) {
                wasReset = true;
            } else {
                this.highScore = highScore;
                set(FLAG_SOUND,     sound);
                set(FLAG_VIBRATE,   vibrate);
                set(FLAG_FUNLIGHTS, funlights);
                set(FLAG_HINT,      hint);
                save();
            }
        } catch(RecordStoreException exception) {
            // Just proceed with defaults.
            highScore = 0;
            set(FLAG_SOUND,     true);
            set(FLAG_VIBRATE,   true);
            set(FLAG_FUNLIGHTS, true);
            set(FLAG_HINT,      true);
            wasReset = true;
        }

        return(wasReset);
    }

    /**
     * Generates a byte array from the flag values in the underlying
     * <tt>Vector</tt>. Since this method allocates new memory, it should
     * not be called frequently.
     *
     * @return  byte array constructed from flag values
     */
    private byte[] getFlagArray()
    {
        int numFlags = flags.size();
        byte[] array = new byte[ numFlags ];

        // Fill byte array from each flag in vector
        for ( int i = 0; i < numFlags; i++ )
        {
            array[i] = ((Byte)flags.elementAt( i )).byteValue();
        }

        return array;
    }

    /* BaseScreen */

    /**
     * Returns a reference to the <tt>Displayable</tt> object that should be
     * given focus for the current screen.
     *
     * @return  reference to current <tt>Displayable</tt> object.
     */
    public Displayable getDisplayable()
    {
        return settingsForm;
    }

    /**
     * Performs all once-only initialization for the screen. Most memory
     * allocation and interface setup occurs here. Prepares the standard
     * settings interface.
     *
     * @throws  IOException if error occurs creating or loading settings
     */
    public void init() throws IOException
    {
        // Add setting flags.
        Vector labels = new Vector(0, 2);
        labels.addElement(StringTable.OPTION_SOUND);
        FLAG_SOUND = add(true);
        labels.addElement(StringTable.OPTION_VIBRATE);
        FLAG_VIBRATE = add(true);
        labels.addElement(StringTable.ERASE_RECORDS);
        FLAG_RESET = add(false);
        
        //if(labels.size() == 0) {
        //    labels.addElement(StringTable.OPTION_HINT);
        //    FLAG_HINT = add(true);
        //}

        // Temporary arrays for set-up of dialog.
        String labelStrings[] = new String[labels.size()];
        boolean defaultValues[] = new boolean[labels.size()];
        for(byte index = 0; index < labelStrings.length; ++index) {
            labelStrings[index] = (String) labels.elementAt(index);
            defaultValues[index] = isOn(index);
        }

        // Layout interface
        settingsForm = new DialogCanvas(this, DialogCanvas.CHECKLIST, StringTable.OPTION_TITLE );
        settingsForm.setList(labelStrings, defaultValues);
        settingsForm.setDialogListener( this );
    }

    /**
     * Performs general cleanup at shutdown, closes the underlying record store.
     */
    public void destroy()
    {
        // Do nothing
    }

    /**
     * Handles all soft key commands for this screen. Primarily the
     * Save command, which saves settings to persistent storage then
     * returns to the main menu.
     *
     * @param   cmd     command being handled.
     * @param   dis     displayable containing command.
     */
    public void commandAction( Command cmd, Displayable dis )
    {
        if (cmd.getCommandType() == Command.BACK) {
            safeSave();
            gameEngine.soundOn = isOn(SettingsScreen.FLAG_SOUND);
            gameEngine.vibrateOn = isOn(SettingsScreen.FLAG_VIBRATE);
            gameEngine.funlightsOn = isOn(SettingsScreen.FLAG_FUNLIGHTS);
                
            if (GameBoard.isPaused) 
            {
                GameBoard gameBoard = (GameBoard)
                    gameEngine.getScreenInstance(GameEngine.SCREEN_GAME);
                gameEngine.transition( GameEngine.SCREEN_GAME, false );
                gameBoard.handlePause();
            } 
            else
            {
                MainScreen.showTrialOver = false;
                gameEngine.transition( GameEngine.SCREEN_MAIN );
            }
        }
    }

    /**
     * Handle dialog item toggling with appropriate event response.
     *
     * @param type of event, should be ON or OFF from DialogCanvas
     * @param index of item toggled
     */
    public void dialogEvent(byte type, int index) {
        super.dialogEvent(type, index);
        if(type == DialogCanvas.ON) {
            if (index == FLAG_SOUND) {
                gameEngine.playMidi(GameBoard.MIDI_SWAP, false, true);
                gameEngine.playAlertSound(GameBoard.ALERT_SWAP, true);
            }
            else if (index == FLAG_VIBRATE) {
                gameEngine.vibrate(500, true);
            }
            else if (index == FLAG_RESET) {
                // Reset high scores
                set((byte)index, false);
                
                if (gameEngine.isDemo()) {
                    showUnavilableScreen();
                }
                else {
                    dlg.show(this, Dialog.DLG_YESNO, StringTable.RESET_SOFTKEY,
                        StringTable.RESET_MESSAGE);
                }
            }
        }
    }
    
    private void showUnavilableScreen() {
        DialogListener listener = new DialogListener() {
            public void dialogDismissed( byte result ) {
                if (result == Dialog.CMD_OK) {
                    Build.showBuyURIAndExit(gameEngine);
                }
                else if (result == Dialog.CMD_CANCEL) {
                    gameEngine.restart( GameEngine.SCREEN_SETTINGS );
                }
            }
        };
        
        String message = StringTable.VODAFONE_UNAVAIL;
        if (Build.VODAFONE) {
            message += "|" + StringTable.VODAFONE_GET_GAME + "|" +
                StringTable.VODAFONE_DOWNLOAD;
        }
        
        dlg.show(listener, Dialog.DLG_VODA_GET_BACK, "BEJEWELED", message);        
    }

    /**
     * Get the current high score.
     *
     * @return  Current highest score.
     */
    public int getHighScore() {
        return highScore;
    }

    /**
     * Report the game's score in case it should be stored, such as for
     * a high score.
     *
     * @param score for the latest game
     */
    public void reportScore(int score) {
        if(score > highScore) {
            setHighScroe(score);
        }
    }

    /**
     * Clear high score.
     */
    public void resetHighScore() {
        setHighScroe(0);
    }

    /**
     * Change high score, save, and update display value.
     *
     * @param score new value
     */
    private void setHighScroe(int score) {
        highScore = score;
        safeSave();
        ((MainScreen)gameEngine.getScreenInstance(
            GameEngine.SCREEN_MAIN)).setHighScore(highScore);
    }

    /**
     * Save data with special handling in case of corruption.
     */
    private void safeSave() {
        try {
            save();
        } catch(RecordStoreException exception) {
            if(restore(false)) {
                dlg.showError(
                    gameEngine.getScreenInstance(GameEngine.SCREEN_MAIN),
                    StringTable.RMS_FAILURE);
            }
        }
    }

    /**
     * Force record store to close.
     *
     * @throws RecordStoreException if fails
     */
    public void forceClose() throws RecordStoreException {
        if(recordStore != null) {
            recordStore.closeRecordStore();
        }
    }
    
    
    public void dialogDismissed( byte result )
    {
        // From the score reset screen
        if (result == Dialog.CMD_YES) {
            gameEngine.getSettings().resetHighScore();
        }
        
        gameEngine.transition( GameEngine.SCREEN_SETTINGS );
     
    }

}
