package shell.core;

import java.util.Vector;
import javax.microedition.lcdui.Image;
import javax.microedition.lcdui.Graphics;
import javax.microedition.lcdui.game.GameCanvas;

public abstract class Scene {
    
    private Image backgroundImage;
    private int backgroundColor;
    private Vector visuals = new Vector();
    
    private Visual[] softKeys = new Visual[2];
    
    // Dirty rect
    private boolean needsFullRepaint;
    private int minX;
    private int maxX;
    private int minY;
    private int maxY;
    
    
    public abstract void load();
    
    
    public void unload() {
        visuals = new Vector();
        backgroundImage = null;
        softKeys[0] = null;
        softKeys[1] = null;
    }
    
    
    private void setSoftKey(int index, Visual softKey, int x, int y, int anchor) {
        if (softKeys[index] != softKey) {
            if (softKeys[index] != null) {
                remove(softKeys[index]);
            }
            if (softKey != null) {
                softKey.setLocation(x, y);
                //softKey.moveTo(x, Device.HEIGHT - visual.getHeight(), 0, 500, Visual.EASE_OUT);
                softKey.setAnchor(anchor);
                softKey.setFrame(0);
                softKeys[index] = softKey;
                add(softKeys[index]);
            }
        }
    }
    
    
    public final void setSoftKey1(Visual softKey) {
        setSoftKey(0, softKey, Device.SOFT1_X, Device.SOFT1_Y, Device.SOFT1_ANCHOR);
    }
    
    
    public final void setSoftKey2(Visual softKey) {
        setSoftKey(1, softKey, Device.SOFT2_X, Device.SOFT2_Y, Device.SOFT2_ANCHOR);
    }
    
    
    public final void setSoftKey1Highlight(boolean highlight) {
        if (softKeys[0] != null) {
            softKeys[0].setFrame(highlight?1:0);
        }
    }
    
    
    public final void setSoftKey2Highlight(boolean highlight) {
        if (softKeys[1] != null) {
            softKeys[1].setFrame(highlight?1:0);
        }
    }

    
    
    /**
        Sets the background image. If the image is smaller than the display dimensions the image
        is tiled.
    */
    public final void setBackground(Image image) {
        backgroundImage = image;
        needsFullRepaint = true;
    }
    
    
    /**
        Sets the background color. The background color is only drawn if the background image is
        null.
    */
    public final void setBackground(int color) {
        backgroundColor = color;
        needsFullRepaint = true;
    }
    
    
    public Image getBackgroundImage() {
        return backgroundImage;
    }
    
    
    public final void add(Visual visual) {
        if (visual != null) {
            visuals.addElement(visual);
            // Note, full repaint is not optimal for scenes that add/remove Visuals often
            needsFullRepaint = true;
        }
    }
    
    
    public final boolean remove(Visual visual) {
        if (visual == null) {
            return false;
        }
        
        boolean removed = visuals.removeElement(visual);
        if (removed) {
            // Note, full repaint is not optimal for scenes that add/remove Visuals often
            needsFullRepaint = true;
        }
        return removed;   
    }
    
    
    public final void updateScene(int elapsedTime) {
        update(elapsedTime);
        
        minX = Device.WIDTH;
        maxX = 0;
        minY = Device.HEIGHT;
        maxY = 0;
        
        for (int i = 0; i < visuals.size(); i++) {
            Visual visual = (Visual)(visuals.elementAt(i));
            
            int oldX = visual.getScreenX();
            int oldY = visual.getScreenY();
            
            boolean changed = visual.update(elapsedTime);
            if (changed) {
                addDirtyRect(oldX, oldY, visual.getWidth(), visual.getHeight());
                addDirtyRect(visual.getScreenX(), visual.getScreenY(), 
                    visual.getWidth(), visual.getHeight());
            }
        }
    }
    
    
    public void update(int elapsedTime) {
        // Do nothing
    }
    
    
    private final void addDirtyRect(int x, int y, int w, int h) {
        int x1 = x;
        int y1 = y;
        int x2 = x + w - 1;
        int y2 = y + h - 1;
        
        // Off-by-one error in the emulator?
        x2++;
        y2++;

        if (x1 < minX) {
            minX = Math.max(0, x1);
        }
        
        if (y1 < minY) {
            minY = Math.max(0, y1);
        }
        
        if (x2 > maxX) {
            maxX = Math.min(Device.WIDTH - 1, x2);
        }
        
        if (y2 > maxY) {
            maxY = Math.min(Device.HEIGHT - 1, y2);
        }
    }
    
    
    /**
        @return true if something was painted.
    */
    public final boolean draw(GameCanvas canvas, Graphics g, boolean forceFullRepaint) {
        
        forceFullRepaint |= needsFullRepaint;
        needsFullRepaint = false;
        
        if (forceFullRepaint) {
            g.setClip(0, 0, Device.WIDTH, Device.HEIGHT);
        }
        else {
            if (minX >= maxX || minY >= maxY) {
                // Nothing to draw
                return false;
            }
            int w = maxX - minX + 1;
            int h = maxY - minY + 1;
            g.setClip(minX, minY, w, h);
        }
        
        // Draw the background - either a solid color or a tiled image
        if (backgroundImage == null) {
            g.setColor(backgroundColor);
            g.fillRect(0, 0, Device.WIDTH, Device.HEIGHT);
        }
        else {
            int w = backgroundImage.getWidth();
            int h = backgroundImage.getHeight();
            
            int numAcross = Math.max(1, (Device.WIDTH + w - 1) / w);
            int numDown = Math.max(1, (Device.HEIGHT + h - 1) / h);
            
            int x = 0;
            for (int i = 0; i < numAcross; i++) {
                int y = 0;
                for (int j = 0; j < numDown; j++) {
                    g.drawImage(backgroundImage, x, y, Graphics.TOP | Graphics.LEFT);
                    y += h;
                }
                x += w;
            }
        }
        
        // Draw the visual elements
        for (int i = 0; i < visuals.size(); i++) {
            Visual visual = (Visual)(visuals.elementAt(i));
            visual.draw(g);
        }
        
        // Flush the buffer
        if (forceFullRepaint) {
            canvas.flushGraphics();
        }
        else {
            int w = maxX - minX + 1;
            int h = maxY - minY + 1;
            canvas.flushGraphics(minX, minY, w, h);
        }
        
        return true;
    }
}