package sims2;



/* Copyright Ideaworks3D Ltd. 2005 */

/*

 * Node.java

 */







import java.util.Enumeration;

import java.util.Vector;



//import com.ideaworks3d.debug.Debug;





/**

 * <p>Interface that all Node types in A Star Searches should implement.</p>

 * 

 * <p>To optimise the best path function <code>Node</code>s require equal

 * alternatives.</p>

 */

public class Node

{

	/** To express that a cost is truly infinite (not just big!) */

	public static final int	INFINITE_COST	= Integer.MAX_VALUE;

	/** 

	 * Create a new node.

	 * After object construction evaluateCost() should be called.

	 * 

	 * @param	parent 	that this node comes from

	 */

	private Node(Node parent, int nodeType)

	{

		this.m_Type = nodeType;

		this.parent = parent;

		children = new Vector();

	}

	

	/** 

	 * Add a child node

	 * 

	 * @param newChild	Child to add

	 */

	public void addChild(Node newChild, boolean hasPreferredDirection, int preferredDirectionX, int preferredDirectionY)

	{

		children.addElement(newChild);

		newChild.setParent(this, hasPreferredDirection, preferredDirectionX, preferredDirectionY);

	}



	/**

	 * Estimate cost to travel to a given node from here

	 * 

	 * @param	toNode	Node travelled to

	 * @return	estimated cost to end node (admissible -- best case)

	 */

	public int calculateHeuristicTo(Node toNode)

	{

		// Calculate from manhattan distance.  

		return	Math.abs(toNode.m_X - m_X) + 

				Math.abs(toNode.m_Y - m_Y);

	}

	

	/** Create and return all successor nodes.

	 * 

	 * @param start					Start node

	 * @param end					End node

	 * @param hasPreferredDirection	Is there a preferred direction?

	 * @param preferredDirectionX	Preferred delta x

	 * @param preferredDirectionY	Preferred delta y 

	 */

	public Node[] createSuccessors(Node start, Node end, boolean hasPreferredDirection, int preferredDirectionX, int preferredDirectionY)

	{

		if (m_Type == TYPE_BOARD)

		{

			return BoardCreateSuccessors(start, end, hasPreferredDirection, preferredDirectionX, preferredDirectionY);

		}

		else

		{

			return PortalCreateSuccessors(start, end, hasPreferredDirection, preferredDirectionX, preferredDirectionY);

		}

	}



	/** Compare with another node for equality */

	public boolean equals(Node other)

	{

		return	(m_X == other.m_X &&

				 m_Y == other.m_Y );

	}

	

	/**

	 * Get nodes equivalent to this, self included (allowing multiple destinations).

	 * 

	 * @return	Vector of equivalent nodes

	 */

	public Vector getEquivalentNodes()

	{

		if (m_Type == TYPE_BOARD)

		{

			return BoardGetEquivalentNodes();

		}

		else

		{

			return PortalGetEquivalentNodes();

		}

	}

	

	/**

	 *  Obtain the fitness level (cost + heuristic)

	 *

	 * @param	endNode	Fitness we are estimating to 

	 */

	public int getFitness(Node endNode) 

	{ 		

		// Calculate fitness level

		int fitness = calculateHeuristicTo(endNode);

		if (totalCost == INFINITE_COST)

			fitness = INFINITE_COST;

		else

			fitness = totalCost + fitness;

		

		return fitness;

	}

	

	/** Determine which node is the parent of this (null for none) */

	public Node getParent() 

	{ 

		return parent; 

	}



	/** @return	X Position */

	public int getX() 

	{ 

		return m_X; 

	}



	/** @return	Y Position */

	public int getY() 

	{ 

		return m_Y; 

	}



	/** Provide a unique hash code */

	public int hashCode() 

	{

		int hash;

		if (m_Type == TYPE_BOARD)

		{

			hash = (m_X << 16) + m_Y;

		}

		else			

		{

			hash = m_Id;

		}

		

		return hash;

	}

	

	/** Redefine who is the parent */

	public void setParent(Node parent, boolean hasPreferredDirection, int preferredDirectionX, int preferredDirectionY)

	{

		// Reset the parent of this and recalculate.

		this.parent = parent;

		evaluateCost(hasPreferredDirection, preferredDirectionX, preferredDirectionY);

		

		// Evaluate cost of children.

		Enumeration childrenEnum = children.elements();

		while (childrenEnum.hasMoreElements())

			((Node)childrenEnum.nextElement()).setParent(this, hasPreferredDirection, preferredDirectionX, preferredDirectionY);

	}

	

	/**

	 * Calculate the cost of the path from a start node.

	 * 

	 * @param fromNode					Start node

	 * @param hasPreferredDirection		Does this have a preferred direction?

	 * @param preferredDirectionX		Preferred delta x

	 * @param preferredDirectionY		Preferred delta y

	 * @return

	 */

	protected int calculateCostFrom(Node fromNode, boolean hasPreferredDirection, int preferredDirectionX, int preferredDirectionY)

	{

			int cost;

			

			// If boards are not the same then there is no route

			if (this.equals(fromNode))

			{

				cost = 0;

			}

			else

			{

				// Assume that we only stepping one square along (by method

				// of generating children.

				

				// Check if possible to walk between squares on board

				if (!House.s_LotSingleton.IsSpaceOccupied(m_X, m_Y))

					cost = 	fromNode.calculateHeuristicTo(this);

				else

					cost = INFINITE_COST;

			}

			

			return cost;

		}



	/**

	 * Called to evaluate the cost of this node

	 */

	protected void evaluateCost(boolean hasPreferredDirection, int preferredDirectionX, int preferredDirectionY)

	{

		int parentCost;

		

		// Determine cost of parent (if any)

		if (parent == null)

			parentCost = 0;

		else

			parentCost = parent.totalCost;

		

		// Calculate total cost so far on this path

		if (parentCost == INFINITE_COST)

		{

			totalCost = INFINITE_COST;

		}

		else

		{

			// Determine cost of getting to this location from last

			int thisCost;

			if (parent != null)

			{

				thisCost = calculateCostFrom(

					parent, 

					hasPreferredDirection,

					preferredDirectionX,

					preferredDirectionY );

			}

			else

			{

				thisCost = 0;

			}

			

			// Add to total

			if (thisCost == INFINITE_COST)

				totalCost = INFINITE_COST;

			else

				totalCost = parentCost + thisCost;

		}

	}

	

	private static final int TYPE_BOARD = 0;

	private static final int TYPE_PORTAL = 1;

	

	/** Slower path finding, but shorter routes */

	private static final boolean	FIND_BEST_PATH	= true;	

	

	/** Type of this node (board or portal) */

	private int m_Type;

	

	/** X position in the map */

	protected int m_X;

	/** Y position in the map */

	protected int m_Y;

	

	/** Children nodes */

	private Vector children;

	/** Parent node of this node */

	private Node parent;

	/** Cost so far on this path */

	private int totalCost;

	

	

	/*-------------------- MERGED FROM BOARDNODE.JAVA ------------------*/

	

	/**

	 * Special constructor for if the board has a preferred direction

	 * @param parent	Parent node in search -- needed to pass to Node constructor

	 * @param	xPos	X position of this grid location

	 * @param	yPos	Y position of this grid location

	 */

	public static Node CreateBoardNode(Node parent, int xPos, int yPos, boolean hasPreferredDirection, int preferredDirectionX, int preferredDirectionY)

	{

		Node node = new Node(parent, TYPE_BOARD);

		

		node.m_X = xPos;

		node.m_Y = yPos;

		

		// Needs to be called after variables are set

		// (otherwise parent constructor evaluates the cost as infinite)

		node.evaluateCost(hasPreferredDirection, preferredDirectionX, preferredDirectionY);

		

		return node;

	}

	

	/** 

	 * Create and return all successor nodes 

	 *

	 * @return	List of nodes

	 */

	/** Create and return all successor nodes */

	public Node[] BoardCreateSuccessors(Node start, Node end, boolean hasPreferredDirection, int preferredDirectionX, int preferredDirectionY)

	{

		Node[] children;

		

		// We can currently go up, down, left or right.

		// Ensure that the direction moved is the direction preferred.

		children = new Node[4];

		children[0] = CreateBoardNode(this, m_X - 1, m_Y, hasPreferredDirection, preferredDirectionX, preferredDirectionY);

		children[1] = CreateBoardNode(this, m_X + 1, m_Y, hasPreferredDirection, preferredDirectionX, preferredDirectionY);

		children[2] = CreateBoardNode(this, m_X, m_Y - 1, hasPreferredDirection, preferredDirectionX, preferredDirectionY);

		children[3] = CreateBoardNode(this, m_X, m_Y + 1, hasPreferredDirection, preferredDirectionX, preferredDirectionY);

		

		return children;

	}

	

	/* (non-Javadoc)

	 * @see astar.Node#getEquivalentNodes()()

	 */

	public Vector BoardGetEquivalentNodes()

	{

		Vector nodes = new Vector();



		// Only portal tiles have equivalents.

		byte sector = House.s_LotSingleton.getSector(m_X, m_Y);

		if((sector & 0x80) != 0)

		{

			byte portalId = (byte) (sector & ~0x80);

			// Fill out the portals vector.

			House.s_LotSingleton.FindPortals(portalId, m_X, m_Y, nodes);

		}

		else

		{

			// This is not a real portal.

			nodes.addElement(this);

		}

		return nodes;

	}

	

	/* (non-Javadoc)

	 * @see com.ideaworks3d.sims.path_finding.Node#calculateCostFrom(com.ideaworks3d.sims.path_finding.Node, boolean, int, int)

	 */

	protected int BoardCalculateCostFrom(Node fromNode, boolean hasPreferredDirection, int preferredDirectionX, int preferredDirectionY)

	{

		int cost;

	

		if (this.equals(fromNode))

		{

			cost = 0;

		}

		else

		{

			// Assume that we only stepping one square along (by method

			// of generating children.

			

			// Check if possible to walk between squares on board

			if (!House.s_LotSingleton.IsSpaceOccupied(m_X, m_Y))

			{

				// Calculate from manhattan distance.  

				cost = 	fromNode.calculateHeuristicTo(this);

				

				// BW - This biases towards already calculated paths.

				// This ensures that the best node will be the closest node to the destination.

				if (!FIND_BEST_PATH)

					cost >>= 1;

				

				// Make it more costly if we're not preferred in the direction stakes!

				if( hasPreferredDirection &&

					fromNode.m_X + preferredDirectionX != m_X || 

					fromNode.m_Y + preferredDirectionY != m_Y )

				{

					cost++;

				}

			}

			else	// Not walkable

			{

				cost = INFINITE_COST;

			}

		}

		

		return cost;

	}

	

	/*-------------------- MERGED FROM PORTALNODE.JAVA ------------------*/

	

	/**

	 * Constructor

	 * @param parent	Parent to which it's linked

	 * @param id		Portal id (or END_ID, START_ID)

	 * @param x			x position on the map

	 * @param y			y position on the map

	 */

	public static Node CreatePortalNode(Node parent, byte id, int x, int y)

	{

		// Construct the superclass Node.

		Node node = new Node(parent, TYPE_PORTAL);

		

		// Set the members.

		node.m_Id = id;

		node.m_X = x;

		node.m_Y = y;

		

		// Evaluate the traversal cost.

		node.evaluateCost(false, 0, 0);

		

		return node;

	}

	

	/**

	 * Constructor

	 * @param parent	Parent to which it's linked

	 * @param xPos		x position in the map

	 * @param yPos		y position in the map

	 */

	public static Node CreatePortalNode(Node parent, int xPos, int yPos)

	{

		Node node = new Node(parent, TYPE_PORTAL);

		

		node.m_X = xPos;

		node.m_Y = yPos;

		

		// m_X & m_Y must be set.

		node.m_Id = node.PortalGetSectorId();

		

		node.evaluateCost(false, 0, 0);

		

		return node;

	}



	/* (non-Javadoc)

	 * @see astar.Node#createSuccessors(astar.Node, astar.Node)

	 */

	public Node[] PortalCreateSuccessors(Node start, Node end, boolean hasPreferredDirection, int preferredDirectionX, int preferredDirectionY)

	{

		// start and end must be portal nodes.

//		Node startPortal	= start;

		Node endPortal		= end;

		

		Node[] children;

		

		// Get the links.

		byte[] links = PortalGetPortalLinks(endPortal);

		children = new Node[links.length];

		

		// Create the objects.

		for(int i = 0; i < children.length; ++i)

		{

			// Fill in the positions.

			int[] p;

			

			// We should never link to the start.

			//Debug.debugAssert(links[i] != PORTAL_START_ID);

			

			// The end node is generated.

			if(links[i] != PORTAL_END_ID)

				p = House.s_LotSingleton.getPortalLocation(links[i]);

			else

				p = endPortal.PortalGetPortalLocation();

			

			// Create this child.

			children[i] = CreatePortalNode(this, links[i], p[0], p[1]);

		}

		

		return children;

	}



	/* (non-Javadoc)

	 * @see astar.Node#getEquivalentNodes()

	 */

	public Vector PortalGetEquivalentNodes()

	{

		// Portal nodes are unique.

		Vector ret = new Vector();

		ret.addElement(this);

		return ret;

	}

	

	/** Get the internal portal id */

	public byte PortalGetId() 			{ return m_Id; }

	

	/**

	 * <p>

	 * Get the portal id for this location

	 * </p>

	 * <p>

	 * This will return a different result to GetId() for start and end nodes.

	 * </p>

	 */

	public byte PortalGetPortalId()	

	{ 

		byte sector = PortalGetSectorId(); 

		

		// Can't be a wall.

		//Debug.debugAssert(sector != -1);

		

		// Return the existing portal id if there is one.

		if ((sector & 0x80) != 0)

			return (byte) (sector & ~0x80);

		else

			return VIRTUAL_PORTAL;

	}

	

	/**

	 * Get all the portals linked to this node.

	 * 

	 * @param end	The end portal node (to which this may link)

	 * @return		Array of portal ids

	 */

	public byte[] PortalGetPortalLinks(Node end)

	{

		byte[] links;

		

		// The end node should never be checked.

		//Debug.debugAssert(m_Id != PORTAL_END_ID);

		

		// Check whether this is a portal or sector tile.

		byte portalId = PortalGetPortalId();

		if(portalId != VIRTUAL_PORTAL)	// portal

		{

			// Get the standard portal links.

			links = House.s_LotSingleton.getPortalLinks(portalId);

		}

		else								// sector

		{

			// Find the real portals adjoining this sector.

			links = House.s_LotSingleton.getPortals( PortalGetSectorId() );

		}

		

		// Check to see if we should add the end portal.

		if( PortalLinksTo(end) )

		{

			//Debug.debugAssert(end.PortalGetId() == PORTAL_END_ID);

			

			// Add end to the links array.

			byte[] temp = new byte[links.length +1];

			System.arraycopy(links, 0, temp, 0, links.length);

			temp[links.length] = end.PortalGetId();

			links = temp;

		}

		

		return links;

	}

	

	/**

	 * Get the portal map location.

	 * 

	 * @return	position

	 */

	public int[] PortalGetPortalLocation()

	{

		return new int[] {m_X, m_Y};

	}

	

	/**

	 * Get the sector id at the node location.

	 * 

	 * @return	Sector id

	 */

	public byte PortalGetSectorId()

	{

		return House.s_LotSingleton.getSector(m_X, m_Y);

	}

	

	// Currently this could be made more efficient by 

	// inlining it in GetPortalLinks().

	/**

	 * Find whether this links to another node.

	 * 

	 * @param other		The other node

	 * @return			Links?

	 */

	public boolean PortalLinksTo(Node other)

	{

		// Work out the portal ids.

		final byte thisPortalId		= PortalGetPortalId();

		final byte otherPortalId	= other.PortalGetPortalId();

		

		// I hope the compiler can optimise this.

		// I've laid it out for legibility.

		if( thisPortalId != VIRTUAL_PORTAL &&

			otherPortalId != VIRTUAL_PORTAL )

		{

			// Both are portals //

			// Check for equality.

			if(thisPortalId == otherPortalId)

				return true;

			

			// Search the portal links.

			byte[] links = House.s_LotSingleton.getPortalLinks(thisPortalId);

			for(int i = 0; i < links.length; ++i)

				if(links[i] == otherPortalId)

					return true;

		}

		else 

		if( thisPortalId != VIRTUAL_PORTAL &&

			otherPortalId == VIRTUAL_PORTAL )

		{

			// Only this is a portal //

			// Check other sector.

			byte[] portals = House.s_LotSingleton.getPortals( other.PortalGetSectorId() );

			for(int i = 0; i < portals.length; ++i)

				if(portals[i] == thisPortalId)

					return true;

		}

		else

		if( thisPortalId == VIRTUAL_PORTAL &&

			otherPortalId != VIRTUAL_PORTAL )

		{

			// Only other is a portal //

			// Check this sector.

			byte[] portals = House.s_LotSingleton.getPortals( PortalGetSectorId() );

			for(int i = 0; i < portals.length; ++i)

				if(portals[i] == otherPortalId)

					return true;

		}

		else

		if( thisPortalId == VIRTUAL_PORTAL &&

			otherPortalId == VIRTUAL_PORTAL )

		{

			// Neither are portals //

			// See if the sectors match.

			if( PortalGetSectorId() == other.PortalGetSectorId() )

				return true;

		}

		

		// We failed to find a link.

		return false;

	}

	

	/** Marker id for the end node */

	static public final byte PORTAL_END_ID			= -2;

	/** Marker id for the start node */

	static public final byte PORTAL_START_ID		= -1;

	/** Marker for a virtual (generated) portal */

	static public final byte VIRTUAL_PORTAL	= -1;

	

	/** The id number for this portal */

	private byte m_Id;

}

