






    





    



/*
 * PropertyReader.java
 * Created on Oct 22, 2004
 * 
 * Copyright Zaxis Technologies, Inc. 2004
 * 10401 Roselle Street, Suite A202
 * San Diego, CA 92121
 * USA
 * 858-623-0666
 *
 * The copyright to the computer program(s) 
 * is the property of Zaxis Technologies, Inc. 2004.
 * The program(s) may be used and/or copied only with
 * the written permission of Zaxis Technologies, Inc.
 * or in accordance with the terms and conditions
 * stipulated in the agreement/contract under which
 * the program(s) have been supplied.
 */

import java.io.*;

/**
 * Class for parsing properties.
 * 
 * @author glewis
 */
public class PropertyReader {
   public static final int MAX_PROPERTY_NAME_LENGTH = 10;
   public static final int MAX_PROPERTY_VALUE_LENGTH = 200;
   public static final byte ESCAPE_CHAR = '/';
   public static final byte SEPARATOR_CHAR = '=';
   public static final byte[] QUOTE_CHARS = new byte[] {'\"', '\''};
   public static final boolean CASE_SENSITIVE = false;

   private DataInputStream in;
   private boolean eof;
   private boolean eol;
   private IOException error;
   private int propertyNameIndex;
   private String[] nameStrings;

   public byte[] propertyName;
   public byte[] propertyValue;    
   public int propertyNameLength;
   public int propertyValueLength;    

   public PropertyReader() {
      propertyName = new byte[MAX_PROPERTY_NAME_LENGTH];
      propertyValue = new byte[MAX_PROPERTY_VALUE_LENGTH];
      reset(null);
   }

   public void reset() {
      reset(in);
   }

   public String getDebugString() {
      return new String(propertyName, 0, propertyNameLength)
      + "=" + new String(propertyValue, 0, propertyValueLength);
   }

   public void reset(DataInputStream in) {
      this.in = in;
      propertyNameLength = 0;
      propertyValueLength = 0;
      eof = (in == null);
      eol = eof;
      error = null;
      propertyNameIndex = -1;
   }

   public void setNameStrings(String[] names) {
      this.nameStrings = names;
      propertyNameIndex = -2;
   }

   private int readToken(byte[] buf, boolean isName, boolean newlineOK) 
   throws IOException {
      int len = 0;
      int val = 0;
      byte quote = 0;
      boolean escape = false;
      int maxlen = buf.length;

      if (eof || (eol && ((!isName) || (!newlineOK)))) {
         return 0; 
      }

      val = in.read();
      if (isName) {
         while ((val <= ' ') && (newlineOK || (val != '\n')) && (val >= 0)) {
            val = in.read();        
         }
      }

      if (isName && (val != '\n')) {
         quote = SEPARATOR_CHAR;
         eol = eof;
      }
      else {
         for (int i=QUOTE_CHARS.length; --i>=0;) {
            if (val == QUOTE_CHARS[i]) {
               quote = QUOTE_CHARS[i];
               val = in.read();
            }
         }
      }

      while ((quote == 0) ? (val > ' ') 
             : ((val != quote) && (val >= 0))) {
         if (val == ESCAPE_CHAR) {
            val = in.read();
         }
         if (val >= 0) {
            if (len < maxlen) {
               buf[len++] = (byte)val;
            }
            val = in.read();
         }
      }

      if (val < 0) {
         eol = true;
         eof = true;
      }
      else if (val == '\n') {
         eol = true;
      }

      return len;
   }

   /** 
    * reads in the next property.
    * @param newlineOK true if the property does not have to appear on the
    *        current line.
    * @return true if there was a property to read, false otherwise.
    * @throws IOException if there was an error reading in data.
    */
   public boolean next(boolean newlineOK) throws IOException {
      if (error != null) {
         throw error;
      }
      try {
         while (true) {        
            propertyNameLength = readToken(propertyName, true, newlineOK);
            if (propertyNameLength == 0) {
               propertyValueLength = 0;
               propertyNameIndex = -1;
               return false;
            }
            propertyValueLength = readToken(propertyValue, false, false);
            if (propertyValueLength != 0) {
               propertyNameIndex = -2;
               return true;
            }
         }
      }
      catch (IOException e) {
         ;
         propertyNameLength = 0;
         propertyValueLength = 0;
         eol = true;
         eof = true;
         error = e;
         propertyNameIndex = -1;
         throw e;
      }
   }

   /** 
    * returns the index of the property name within this PropertyReader's
    * nameStrings.
    * 
    * @return the index of the name or -1 if the name does not appear in
    *         this PropertyReaders's nameStrings. 
    */
   public int getNameIndex() {
      if (propertyNameIndex == -2) {
         propertyNameIndex = getStringIndex(propertyName,
                                            0, propertyNameLength, nameStrings);
      }
      return propertyNameIndex;
   }

   /**
    * returns the index of the property value within values, or -1
    * if the property value is not one of the specified Strings.
    * 
    * @param values
    * @return
    */
   public int getValueIndex(String[] values) {
      return getStringIndex(propertyValue, 0, propertyValueLength, values);
   }

   /** 
    * searches for a match between the first len chars
    * in buf starting at off and each of the Strings in vals.
    * 
    * @param buf the buffer containing the string to match.
    * @param off the offset within buf of the string.
    * @param len the length of the string within buf.
    * @param vals the array of Strings to look for a match in.
    * @return the index of the matched String or -1 if no Strings matched.
    */
   public static int getStringIndex(byte[] buf, int off, int len,
                                    String[] vals) {
      int index = -1;                                         
      if ((len > 0) && (vals != null)) {
         for (index=vals.length; --index>=0;) {
            String s = vals[index];
            if (s.length() == len) {
               int j = len;
               int o = off + len;
               while (--j >= 0) {
                  int c1 = s.charAt(j);
                  int c2 = buf[--o];
                  if ((c1 != c2) && (CASE_SENSITIVE || !(
                                                        ((c1 >= 'A') && (c1 <= 'Z')
                                                         && (c1 == c2 + 'A' - 'a')) ||
                                                        ((c1 >= 'a') && (c1 <= 'Z')
                                                         && (c1 != c2 + 'a' - 'A'))))) {
                     break;
                  }
               }
               if (j < 0) {
                  break;
               }
            }
         }
      }
      return index;
   }

   /** 
    * returns the int value of the property.  If the property starts with
    * 0x then it is treated as hex, otherwise it is treated as decimal.
    * Only non-negative numbers are allowed.
    * 
    * @return the number read, or -1 if the property was not properly formed.
    */
   public int getIntValue() {
      if (propertyValueLength == 0) {
         return -1;
      }
      int val = 0;
      for (int i=0; i<propertyValueLength; i++) {
         int c = propertyValue[i];
         if ((c >= '0') && (c <= '9')) {
            val = (val * 10) + propertyValue[i] - '0';
         }
         else if ((i == 1) && (propertyValue[0] == '0') && (propertyValue[1] == 'x')) {
            return getHexValue();
         }
         else {
            return -1; // should probably throw an exception
         }
      }
      return val;
   }

   /**
    * returns the int value of the property value interpreted as hex.
    * this can be of either a form such as 9C0A or as 0xABCDEF.
    * Only non-negative numbers are allowed.
    * 
    * @return the number read, or -1 if the property was not properly formed.
    */
   public int getHexValue() {
      if (propertyValueLength == 0) {
         return -1;
      }
      int val = 0;
      for (int i=0; i<propertyValueLength; i++) {
         int c = propertyValue[i];
         val <<= 4;
         if ((c >= '0') && (c <= '9')) {
            val += c - '0';
         }
         else if ((c >= 'a') && (c <= 'f')) {
            val += c + 0x0A - 'a';
         }
         else if ((c >= 'A') && (c <= 'F')) {
            val += c + 0x0A - 'A';
         }
         else if ((i == 1) && (propertyValue[0] == '0') && (propertyValue[1] == 'x')) {
            val = 0;
         }
         else {
            return -1; // should probably throw an exception
         }
      }
      return val;
   }

   public static void print(PrintStream out, String name, byte[] val, int valLen) {
      out.print(name);
      out.write('=');
      out.write('\"');
      out.write(val, 0, valLen);
      out.write('\"');        
      out.write(' ');
   }

   public static void print(PrintStream out, String name, int val) {
      out.print(name);
      out.print("=0x0");
      int numDigits = 0;
      int tVal = val;
      while (tVal > 0) {
         numDigits++;
         tVal >>= 4;
      }
      while (numDigits > 0) {
         int dig = (val >> (--numDigits << 2)) & 0x0F;
         if (dig >= 0x0A) {
            out.write('A' - 0x0A + dig);
         }
         else {
            out.write('0' + dig);
         }
      }
      out.write(' ');
   }
}
