






    





    



/*
 * ByteUtils.java
 * Created on Apr 12, 2004
 * 
 * Copyright Zaxis Technologies, Inc. 2004
 * 10401 Roselle Street, Suite A202
 * San Diego, CA 92121
 * USA
 * 858-623-0666
 *
 * The copyright to the computer program(s) 
 * is the property of Zaxis Technologies, Inc. 2004.
 * The program(s) may be used and/or copied only with
 * the written permission of Zaxis Technologies, Inc.
 * or in accordance with the terms and conditions
 * stipulated in the agreement/contract under which
 * the program(s) have been supplied.
 */

import java.io.*;

/**
 * Utility operations for byte arrays.
 *
 * Converts arrays of chars to and from byte arrays in UTF-8 format.
 * This will allow for efficient storage and aid internationalization.
 * Modified versions of DataXStream read/write UTF methods.
 *
 * Reads and Writes integer and long values to and from byte arrays.
 *
 * Calculates Cyclical Redundancy Check (crc) values.
 *
 * Generates an item ids for records
 *
 * @version 1.0
 * @author sgargan
 */
public class ByteUtils {
   /* type of value to read from buffer */
   public static final int SHORT = 2;

   /* type of value to read from buffer */
   public static final int INT = 4;

   /* CRC Lookup table */
   private static int CRCTable[];


   /**
    * Takes an array of characters and encodes them in UTF-8 into the given
    * byte array. This will only convert while there is buffer to place the data
    * into or chars to read. It can result in truncated strings so be careful
    * about the lengths & offsets supplied
    *
    * @param chars the array containing the chars to encode
    * @param charoffset the offset into the char array to start at
    * @param length the no of chars to encode
    * @param bytes the byte array to encode the chars into
    * @param byteoffset the offset into the byte array to start at
    *
    * @return int the number of bytes required to encode all the chars
    */
   public static int charsToBytes(char[] chars, int charoffset, int length,
                                  byte[] bytes, int byteoffset) {        
      int bytecount = byteoffset;        
      int range = charoffset + length;
      for (int i = charoffset; i < range && byteoffset < bytes.length; i++) {            
         if ((chars[i] >= 0x0001) && (chars[i] <= 0x007F)) {
            bytes[byteoffset++] = (byte) chars[i];
         }
         else if (chars[i]> 0x07FF) {
            if (chars[i] >= 0xD800 && chars[i] <= 0xDFFF) {
               bytes[byteoffset++] = (byte)'?';
               continue;
            }
            bytes[byteoffset++] = (byte)(0xE0 | ((chars[i] >> 12) & 0x0F));
            bytes[byteoffset++] = (byte)(0x80 | ((chars[i] >>  6) & 0x3F));
            bytes[byteoffset++] = (byte)(0x80 | ((chars[i] >>  0) & 0x3F));
         }
         else {
            bytes[byteoffset++] = (byte)(0xC0 | ((chars[i] >>  6) & 0x1F));
            bytes[byteoffset++] = (byte)(0x80 | ((chars[i] >>  0) & 0x3F));
         }
      }

      //return bytecount;
      return byteoffset - bytecount;
   }

   /**
    * Converts UTF-8 encoded bytes from and array back into characters
    * and places. This will only convert while there is buffer to place the data
    * into or chars to read. It can result in truncated strings so be careful
    * about the lengths & offsets supplied
    *
    * @param chars the array to hold the decoded chars
    * @param charoffset the offset into the char array to start at
    * @param length the no of chars to decode
    * @param bytes the byte array holding the encoded chars
    * @param byteoffset the offset into the byte array to start at
    *
    * @return int the number of chars decoded
    */
   public static int bytesToChars(char[] chars, int charoffset, int length,
                                  byte[] bytes, int byteoffset)
   throws UTFDataFormatException{
      int charcount = charoffset;
      int c1, c2, c3;
      while (charcount < length + charoffset && charcount< chars.length
             && byteoffset < bytes.length) {

         c1 = (int) bytes[byteoffset++] & 0xff;
         switch (c1 >> 4) {
            case 0:
            case 1:
            case 2:
            case 3:
            case 4:
            case 5:
            case 6:
            case 7:                    
               chars[charcount++] = (char)c1;
               break;
            case 12:
            case 13:                    
               // if no more data return
               if (byteoffset +1 > bytes.length) {
                  return charcount - charoffset;
               }
               c2 = (int) bytes[byteoffset++];                    
               // check that the top two bits read 10
               if ((c2 & 0xC0) != 0x80) {
                  throw new UTFDataFormatException(
                                                  "Illegal UTF-8 encoding trailing byte should "+
                                                  "begin 0x80, byte1="+Integer.toHexString(c1)+
                                                  ", byte2="+Integer.toHexString(c2));
               }
               chars[charcount++]=(char)(((c1 & 0x1F) << 6) |
                                         (c2 & 0x3F));
               break;
            case 14:
               /* 1110 xxxx  10xx xxxx  10xx xxxx */

               if (byteoffset +2 > bytes.length) {
                  return charcount - charoffset;
               }
               c2 = (int) bytes[byteoffset++];
               c3 = (int) bytes[byteoffset++];

               // check both bytes start with 10
               if (((c2 & 0xC0) != 0x80) || ((c3 & 0xC0) != 0x80)) {
                  throw new UTFDataFormatException(
                                                  "Illegal UTF-8 encoding trailing byte should "+
                                                  "begin 0x80, byte1="+Integer.toHexString(c1)+
                                                  ", byte2="+Integer.toHexString(c2)+
                                                  ", byte3="+Integer.toHexString(c3));
               }
               chars[charcount++]=(char)(((c1  & 0x0F) << 12) |
                                         ((c2 & 0x3F) << 6)  |
                                         ((c3 & 0x3F) << 0));
               break;
            default:
               throw new UTFDataFormatException(
                                               "Illegal UTF-8 encoding "+Integer.toHexString(c1));
         }
      }
      return charcount - charoffset;
   }



   /**
    * Takes an array of chars and calculates the number of bytes it will
    * occupy when UTF-8 encoded.
    *
    * @param chars the array containing the chars to encode
    * @param offset the offset into the char array to start at
    * @param length the no of chars to encode
    *
    * @return int the number of bytes required to encode all the chars
    */
   public static int countBytes(char[] chars, int offset, int length){

      int count = 0;
      int range = offset + length;
      for (int i = offset; i < range; i++) {            
         if ((chars[i] >= 0x0001) && (chars[i] <= 0x007F)) {
            count++;
         }
         else if (chars[i] > 0x07FF) {
            if (chars[i]>= 0xD800 && chars[i] <= 0xDFFF){
               count++;
            }
            else {
               count += 3;
            }
         }
         else {
            count += 2;
         }
      }
      return count;
   }

   /**
    * Reads a int value from the buffer using the number of bytes specified.
    * To read a short the size would be two and require a cast from int to short.
    * Reads of ints will be much more common than longs so method returns in and
    * in long case makes 2 calls to this.
    *
    * @param buffer the array containing the data
    * @param the offset to begin reading at
    * @param size the number of bytes to use
    */
   public static final int readValue(byte[] buffer, int offset, int size) {
      int result = 0;
      for (int i = size; --i >= 0;) {
         result <<= 8;
         result |= (buffer[offset++]&0xFF);
      }
      return result;
   }

   /**
    * Reads a long from the buffer at the specified offset
    *
    * @param buffer the array containing the data
    * @param the offset to begin reading at
    * @param size the number of bytes to use
    */
   public static final long readValue(byte[] buffer, int offset){
      return((long)(readValue(buffer, offset, 4)) << 32)
      + (readValue(buffer, offset + 4, 4) & 0xFFFFFFFFL);
   }

   /**
    * Write an int to the buffer at the specified offset
    *
    * @param buffer the array to write the value to
    * @param the offset to begin writing at
    * @param val the int value to write out
    */
   public static final void writeValue(byte[] buffer, int offset, int val,
                                       int size) {        
      for (int i = size; --i >= 0;) {
         buffer[offset+i] = (byte)val;
         val >>= 8;
      }
   }

   /**
    * Write an long to the buffer at the specified offset.
    * Consists of two int writes.
    *
    * @param buffer the array to write the value to
    * @param the offset to begin writing at
    * @param val the long value to write out
    */
   public static final void writeValue(byte[] buffer, int offset, long value) {
      writeValue(buffer, offset, (int)(value >> 32), 4);
      writeValue(buffer, offset+4, (int) value, 4);
   }

   /**
    * General CRC generation function. To begin, use a lastcrc value of -1
    *
    * @param buffer byte array to generate the CRC32
    * @param start byte start position
    * @param count number of byte's to include in CRC calculation
    * @param crc previously generated CRC32.
    *
    * @return 32 bit CRC
    */
   //*/
   public static int crc32(byte buffer[], int offset, int length, int lastcrc){
      int i, j;
      int crc;

      // build the table (first time only)
      if (CRCTable == null){
         CRCTable = new int[256];
         final int CRC32_POLYNOMIAL = 0xEDB88320;
         for (i = 0; i <= 255; i++) {
            crc = i;
            for (j = 8; j > 0; j--){
               if ((crc & 1) == 1){
                  crc = (crc >>> 1) ^ CRC32_POLYNOMIAL;
               }
               else{
                  crc >>>= 1;
               }
            }
            CRCTable[i] = crc;
         }
      }

      // calculate the crc
      crc = lastcrc;
      while (length-- != 0) {
         i = crc >>> 8;
         j = CRCTable[(crc ^ buffer[offset++]) & 0xFF];
         crc = i ^ j;
      }

      return crc;
   }//*/

   /**
    * Takse and integer, converts it ot a string and places it in the
    * char array at the given offset
    *
    * @param value the int value to convert
    * @param arr the array to place the stringified value in
    * @param offset the offset to place the string at
    *
    * @return the length in chars of the string
    *
    */
   public static int itoa(int value, char[] arr, int offset){

      int count = 0;
      int num = value;
      do {
         num = num / 10;
         count = count + 1;
      } while (num != 0);

      num = count;

      if (value < 0) {
         arr[offset] = '-';
         offset++; 
         num++;
         value = value * -1;
      }

      while (count > 0) {
         arr[--count + offset] = (char)((value % 10) + 48);
         value /= 10;
      }        
      return num;        
   }

   /** 
    * converts a sequence of characters into the corresponding integer
    * value and returns it.
    */
   public static int atoi(char[] arr, int offset, int len) {
      int total = 0;
      int neg = arr[offset] == '-' ? 1 : 0;
      for (int i = neg; i< len; i++) {
         if (!Character.isDigit(arr[offset+i])) {
            throw new IllegalArgumentException("Char not digit.");
         }
         total *= 10;
         total += (int)(arr[offset+i]- '0');
      }        
      if (neg == 1) {
         total *= -1;
      }
      return total;
   }

   /**
    * Tests an array size and converts it if necessary
    */
   public static byte[] testSize(byte[] buffer, int length) {
      if (buffer.length < length) {
         byte[] replace = new byte[length + (buffer.length/10)];
         System.arraycopy(buffer, 0, replace, 0, buffer.length);
         return replace;
      }
      else {
         return buffer;
      }
   }

   /**
    * Tests an array size and converts it if necessary
    */
   public static char[] testSize(char[] buffer, int length) {
      if (buffer.length < length) {
         char[] replace = new char[length + (buffer.length/10)];
         System.arraycopy(buffer, 0, replace, 0, buffer.length);
         return replace;
      }
      else {
         return buffer;
      }
   }
}
