






    





    



/*
 * ProgressMonitor.java
 * Created on Oct 5, 2004
 * 
 * Copyright Zaxis Technologies, Inc. 2004
 * 10401 Roselle Street, Suite A202
 * San Diego, CA 92121
 * USA
 * 858-623-0666
 *
 * The copyright to the computer program(s) 
 * is the property of Zaxis Technologies, Inc. 2004.
 * The program(s) may be used and/or copied only with
 * the written permission of Zaxis Technologies, Inc.
 * or in accordance with the terms and conditions
 * stipulated in the agreement/contract under which
 * the program(s) have been supplied.
 */

/**
 * A ProgressMonitor contains the state of progress of one or more operations.
 * 
 * @author glewis
 */
public class ProgressMonitor implements ProgressListener {
   private static final int    PRECISION_MULTIPLIER = 100;

   private ProgressListener[]  listeners;
   private ProgressMonitor[]   children;
   private int[]               weights;
   private int                 progress;
   private int                 maxProgress;
   private Throwable           error;

   /** creates a new ProgressMonitor with specified maxProgress. */
   public ProgressMonitor(int maxProgress) {
      this.maxProgress = maxProgress;
   }

   /** creates a new ProgressMonitor with initial maxProgress of 100. */
   public ProgressMonitor() {
      this(100);
   }

   /** returns the maximum progress value for this ProgressMonitor. */
   public int getMaxProgress() {
      return maxProgress;
   }

   /** sets the maximum progress value for this ProgressMonitor. */
   public void setMaxProgress(int maxProgress) {
      synchronized (this) {
         if (maxProgress != this.maxProgress) {
            this.maxProgress = maxProgress;
            progressChanged(null);
         }
      }
   }

   /**
    * public as an implementation side effect - notifies this ProgressMonitor
    * that one of its children's progress has changed.
    */
   public void progressChanged(ProgressMonitor monitor) {
      synchronized (this) {
         if (error != null) {
            try {
               int newProgress = 0;
               int totalWeight = 0;
               for (int i=children.length; --i>=0;) {
                  if (children[i] != null) {
                     totalWeight += weights[i];
                     newProgress += 
                     (children[i].getProgress()
                      * weights[i]
                      * maxProgress
                      * PRECISION_MULTIPLIER)
                     / children[i].maxProgress;
                  }
               }
               if (totalWeight > 0) {
                  setProgress(newProgress / 
                              (totalWeight * PRECISION_MULTIPLIER));
               }
               else {
                  setProgress(++progress - 1);
               }
            }
            catch (Throwable t) {
               ;
               setError(t);
            }
         }
      }
   }

   /** Add a ProgressListener to receive notifications of state changes. */
   public void addProgressListener(ProgressListener listener) {
      synchronized (this) {
         if (listeners == null) {
            listeners = new ProgressListener[] {listener};
         }
         else {
            int index = -1;
            for (int i=listeners.length; --i>=0;) {
               if (listeners[i] == listener) {
                  return;
               }
               else if (listeners[i] == null) {
                  index = i;
               }
            }
            if (index < 0) {
               index = listeners.length;
               ProgressListener[] newListeners =
               new ProgressListener[index+1];
               System.arraycopy(listeners, 0, newListeners, 0, index);
               listeners = newListeners;
            }
            listeners[index] = listener;
         }
      }
   }

   /**
    * adds a ProgressMonitor that represents the specified weight
    * of total progress for this ProgressMonitor.
    * Calling this method for a ProgressMonitor that has already been
    * added will result in changing the weight for that child.
    */
   public void addChild(ProgressMonitor child, int weight) {
      synchronized (this) {
         if (children == null) {
            children = new ProgressMonitor[] {child};
            weights = new int[] {weight};
         }
         else {
            int index = -1;
            for (int i=children.length; --i>=0;) {
               if (children[i] == child) {
                  weights[i] = weight;
                  return;
               }
               else if (children[i] == null) {
                  index = i;
               }
            }
            if (index < 0) {
               index = children.length;
               int[] newWeights = new int[index+1];
               System.arraycopy(weights, 0, newWeights, 0, index);
               weights = newWeights;

               ProgressMonitor[] newChildren =
               new ProgressMonitor[index+1];
               System.arraycopy(children, 0, newChildren, 0, index);
               children = newChildren;
            }
            weights[index] = weight;
            children[index] = child;
         }
         child.addProgressListener(this);
         progressChanged(child);
      }
   }

   /** removes a previously added ProgressListener. */
   public void removeProgressListener(ProgressListener listener) {
      synchronized (this) {
         if (listeners != null) {
            for (int i=listeners.length; --i>=0;) {
               if (listeners[i] == listener) {
                  listeners[i] = null;
                  return;
               }
            }
         }
      }
   }

   /** removes a previously added ProgressMonitor. */
   public void removeChild(ProgressMonitor child) {
      synchronized (this) {
         if (children != null) {
            for (int i=children.length; --i>=0;) {
               if (children[i] == child) {
                  children[i] = null;
                  child.removeProgressListener(this);
                  progressChanged(null);
                  return;
               }
            }
         }
      }
   }

   /**
    * returns the current state of progress.
    * @throws Throwable if this ProgressMonitor is in an error state.
    */
   public int getProgress() throws Throwable {
      if (error != null) {
         throw error;  
      }
      else {
         return progress;
      }
   }

   /**
    * returns the current state of progress as a percentage.
    * @throws Throwable if this ProgressMonitor is in an error state.
    */
   public int getProgressPercent() throws Throwable {
      return(getProgress() * 100) / maxProgress;
   }

   /** 
    * used internally to notify all listeners, as well as wake up
    * any thread waiting on waitForProgress().
    */
   private void notifyListeners() {
      synchronized (this) {            
         if (listeners != null) {
            for (int i=listeners.length; --i>=0;) {
               if (listeners[i] != null) {
                  listeners[i].progressChanged(this);
               }
            }
         }
         notifyAll();
      }
   }

   /** sets the progress of this ProgressMonitor. */
   public void setProgress(int progress) {
      synchronized (this) {
         if ((error == null) && (progress != this.progress)) {
            this.progress = progress;
            notifyListeners();
         }
      }
   }

   /**
    * puts this ProgressMonitor into an error state.
    * If error is null and this ProgressMonitor has no children,
    * this will clear the error condition and reset progress to zero.
    */
   public void setError(Throwable error) {
      synchronized (this) {
         if ((this.error == null) ^ (error == null)) {
            this.error = error;
            progress = 0;
            if (error == null) {
               progressChanged(null);
            }
            notifyListeners();
         }
      }
   }

   /**
    * waits for the progress of this ProgressMonitor to be at least the
    * specified value, or for an error state to be entered.
    * A negative value means to wait until the operation has completed.
    * @throws Throwable if an error state was entered.
    * @return the actual progress.
    */
   public int waitForProgress(int minProgress) throws Throwable {
      synchronized (this) {
         int p;            
         while (true) {
            p = getProgress();
            if ((p >= minProgress) || ((p > 0) && (p >= maxProgress))) {
               return p;
            }
            wait();
         }
      }
   }
}
